/*
 * Copyright (C) 2001 Eric Z. Beard, ericzbeard@hotmail.com Copyright (c) 2011
 * Gerhard Beck. All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
package org.gerhardb.lib.print;

import java.awt.*;
import java.awt.image.BufferedImage;
import java.io.File;

import javax.swing.JFrame;
import javax.swing.JProgressBar;

import org.gerhardb.lib.image.*;
import org.gerhardb.lib.util.startup.AppStarter;

public class PageableTest implements Pageable
{
	int pageIndex;

	/** Entered on the command line after '-img' */
	private BufferedImage sampleImage;

	/** All sizes will be derived from the desired pdf document size */
	private PageLayout pageLayout = PageLayout.makeDefault();

	PrintTestPattern ptp;

	public PageableTest(String sampleImageFileIn)
	{
		BufferedImage jpgImage = ImageFactory.getImageFactory().getImageEZ(
				new File(sampleImageFileIn));
		this.sampleImage = ImageChangeUtil.scale(jpgImage, 0.5f);
		this.ptp = new PrintTestPattern(getPageLayout().getInteriorLayout());
	}

	// ==========================================================================
	// Pageable Implementation
	// ==========================================================================

	public void setTopFrame(JFrame topFrame)
	{
		// we don't care
	}
	
	public SaveInfo getSaveInfo()
	{
		SaveInfo rtnMe = null;
		try
		{
			rtnMe = new SaveInfo(new File("D:/trash"));
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
		}
		return rtnMe;
	}

	public PageLayout getPageLayout()
	{
		return this.pageLayout;
	}

	public int getNumberOfPages()
	{
		return 5;
	}

	public String[] getFonts()
	{
		return new String[] { "SansSerif" }; // gnuJpdf will work & button will show up.
		//return new String[]{"SansSerif", "Old English Text MT"}; // gnuJpdf won't work & button will NOT show up.
	}

	public String convert(String text, int pageIndexIn) 
	{
		// PAGE_INDEX + " of " + PAGE_COUNT;
		if (text != null)
		{
			text = text.replaceAll(HeaderFooterInfo.PAGE_INDEX, Integer.toString(pageIndexIn));
			text = text.replaceAll(HeaderFooterInfo.PAGE_COUNT, Integer.toString(getNumberOfPages()));
		}
		return text;
	}
	
	public boolean backgroundTreadEnabled()
	{
		return false;
	}

	public void prepareToPrint(int pageIndexIn)
	{
		this.pageIndex = pageIndexIn;
	}
	
	public void print(Graphics g)
	{
		switch (this.pageIndex)
		{
		case 1:
			printLandscape(g);
			break;
		case 2:
			printTestPattern(g);
			break;
		case 3:
			printTwoImages(g);
			break;
		case 4:
			printStringsAndOvals(g);
			break;
		}
	}

	public void setProgressBar(JProgressBar bar)
	{
		// We don't care.
	}
	
	public Pageable getPageableForPrinting()
	{
		return this;
	}
	// ==========================================================================
	// Pages
	// ==========================================================================

	private void printLandscape(Graphics g)
	{
		g.setColor(Color.white);
		g.fillRect(0, 0, this.pageLayout.getPageWidth(), this.pageLayout
				.getPageHeight());

		this.ptp.drawBasicTestPattern(g);

		g.setFont(new Font("SansSerif", Font.PLAIN, 12));

		// No way to make it go vertical...
		g.drawString("This is a simple string", 200, 300);
	}

	private void printTestPattern(Graphics g)
	{
		g.setColor(Color.white);
		g.fillRect(0, 0, this.pageLayout.getPageWidth(), this.pageLayout
				.getPageHeight());

		//System.out.println(pl.toStringDPI());
		//System.out.println(pl.toStringInches());
		this.ptp.drawBasicTestPattern(g);

		strings(g);
		drawSampleImage(g, 310, 340);
	}

	private void printTwoImages(Graphics g)
	{
		g.setColor(Color.white);
		g.fillRect(0, 0, this.pageLayout.getPageWidth(), this.pageLayout
				.getPageHeight());

		smallObjects(g);
		strings(g);
		mediumArcs(g);
		bigArcs(g);
		drawSampleImage(g, 10, 340);
		drawSampleImage(g, 310, 340);
	}


	private void printStringsAndOvals(Graphics g)
	{
		g.setColor(Color.white);
		g.fillRect(0, 0, this.pageLayout.getPageWidth(), this.pageLayout
				.getPageHeight());
		g.setColor(Color.black);
		// test drawLine()
		g.drawLine(10, 10, 50, 50);

		// test drawRect()
		g.drawRect(30, 10, 10, 10);

		// test fillRect()
		g.fillRect(30, 90, 10, 10);

		// test drawPolygon()
		int xp[] = new int[] { 10, 10, 20, 15, 20 };
		int yp[] = new int[] { 50, 60, 60, 55, 50 };
		int np = xp.length;
		g.drawPolygon(xp, yp, np);

		// test drawPolygon()
		xp = new int[] { 60, 60, 70, 65, 70 };
		yp = new int[] { 80, 90, 90, 85, 80 };
		np = xp.length;
		g.drawPolyline(xp, yp, np);

		// test fillPolygon()
		xp = new int[] { 60, 60, 70, 65, 70 };
		yp = new int[] { 50, 60, 60, 55, 50 };
		np = xp.length;
		g.fillPolygon(xp, yp, np);

		// Now some text
		g.setFont(new Font("SansSerif", Font.PLAIN, 12));
		g.drawString("This is a simple string", 10, 120);

		g.drawString("This is a (complex) string", 10, 130);

		g.drawString("(complex) string (with ( (multiple brackets ))", 10, 140);

		// Now some arcs - first test is with a square
		g.drawRect(200, 60, 50, 50); // rectangle
		g.drawLine(200, 60, 250, 110); // join both corners
		g.drawLine(200, 110, 250, 60);
		g.drawLine(200, 85, 225, 60); // this should be a chord
		g.drawArc(200, 60, 50, 50, 45, 180); // the arc

		// June 20, 2001 ezb - Looks like ovals work as expected
		g.drawArc(100, 400, 100, 200, 0, 360);

		g.drawOval(200, 400, 100, 200);

		g.fillOval(300, 400, 100, 200);

		// These two tests act on a rectangular region (ie width != height)
		// Now the interesting thing is that they don't fit the arc within
		// the rectangle, but act on the width. This - by accident - matches the
		// way the Linux JDK operates...

		// Now the same test, but with a rectangle
		g.drawRect(300, 50, 80, 40);
		g.drawLine(300, 50, 380, 90);
		g.drawLine(300, 90, 380, 50);
		g.drawArc(300, 50, 80, 40, 135, 180);

		// Again the same test, but we will fill the arc
		g.drawRect(400, 50, 40, 80);
		g.drawLine(400, 50, 440, 130);
		g.drawLine(400, 130, 440, 50);
		g.setColor(Color.blue);
		g.fillArc(400, 50, 40, 80, 135, 180);
		g.setColor(Color.black);

		// Repeat again, but this time with different angles to the arc.
		// We do this to compare how Java and PDF render the arcs
		g.drawRect(400, 150, 40, 80);
		g.drawLine(400, 150, 440, 230);
		g.drawLine(400, 230, 440, 150);
		g.setColor(Color.blue);
		g.fillArc(400, 150, 40, 80, 135, 225);
		g.setColor(Color.black);

		// Finally draw a small table of all the fonts and styles
		String fonts[] = new String[] { "SansSerif", "Monospaced", "TimesRoman",
				"Helvetica", "Courier", "Dialog", "DialogInput" };
		String modes[] = new String[] { "Plain", "Bold", "Italic", "Bold+Italic" };
		int imodes[] = new int[] { Font.PLAIN, Font.BOLD, Font.ITALIC,
				Font.BOLD + Font.ITALIC };

		int ty = 170;

		for (int i = 0; i < modes.length; i++)
			g.drawString(modes[i], 100 + (50 * i), ty - 14);

		FontMetrics fm = g.getFontMetrics();
		for (int i = 0; i < fonts.length; i++)
			g.drawString(fonts[i], 98 - fm.stringWidth(fonts[i]), ty + (12 * i));

		for (int i = 0; i < fonts.length; i++)
		{
			for (int j = 0; j < modes.length; j++)
			{
				g.setFont(new Font(fonts[i], imodes[j], 10));
				g.drawString(modes[j], 100 + (50 * j), ty);
			}

			ty += 12;
		}
	}

	// ==========================================================================
	// Helper Function for drawing parts of pages
	// ==========================================================================

	private void drawSampleImage(Graphics g, int x, int y)
	{
		try
		{
			g.drawImage(this.sampleImage, x, y, this.sampleImage.getWidth(),
					this.sampleImage.getHeight(), Color.green, null);
		}
		catch (Exception e)
		{
			e.printStackTrace();
		}
	}

	private void smallObjects(Graphics g)
	{
		g.setColor(Color.black);

		// test drawLine()
		g.drawLine(10, 10, 50, 50);

		// test drawRect()
		g.drawRect(30, 10, 10, 10);

		// test fillRect()
		g.fillRect(30, 90, 10, 10);

		// test drawPolygon()
		int xp[] = new int[] { 10, 10, 20, 15, 20 };
		int yp[] = new int[] { 50, 60, 60, 55, 50 };
		int np = xp.length;
		g.drawPolygon(xp, yp, np);

		// test drawPolygon()
		xp = new int[] { 60, 60, 70, 65, 70 };
		yp = new int[] { 80, 90, 90, 85, 80 };
		np = xp.length;
		g.drawPolyline(xp, yp, np);

		// test fillPolygon()
		xp = new int[] { 60, 60, 70, 65, 70 };
		yp = new int[] { 50, 60, 60, 55, 50 };
		np = xp.length;
		g.fillPolygon(xp, yp, np);
	}

	private void strings(Graphics g)
	{
		InteriorLayout interior = this.pageLayout.getInteriorLayout();
		int spacing = 5;
		int onLeft = interior.getLeftMargin() + spacing;
		int onTop = interior.getTopMargin() + 50;

		g.setColor(Color.black);

		// Now some text
		g.setFont(new Font("SansSerif", Font.PLAIN, 12));
		g.drawString("This is a simple string", onLeft, onTop + 20);

		g.drawString("This is a (complex) string", onLeft, onTop + 30);

		g.drawString("(complex) string (with ( (multiple brackets ))", onLeft,
				onTop + 40);

		// Finally draw a small table of all the fonts and styles
		String fonts[] = new String[] { "SansSerif", "Monospaced", "TimesRoman",
				"Helvetica", "Courier", "Dialog", "DialogInput" };
		String modes[] = new String[] { "Plain", "Bold", "Italic", "Bold+Italic" };
		int imodes[] = new int[] { Font.PLAIN, Font.BOLD, Font.ITALIC,
				Font.BOLD + Font.ITALIC };

		int ty = onTop + 70;
		int fromLeft = interior.getLeftMargin() + 100;

		for (int i = 0; i < modes.length; i++)
			g.drawString(modes[i], fromLeft + (50 * i), ty - 14);

		FontMetrics fm = g.getFontMetrics();
		for (int i = 0; i < fonts.length; i++)
			g.drawString(fonts[i], fromLeft - 2 - fm.stringWidth(fonts[i]), ty
					+ (12 * i));

		for (int i = 0; i < fonts.length; i++)
		{
			for (int j = 0; j < modes.length; j++)
			{
				g.setFont(new Font(fonts[i], imodes[j], 10));
				g.drawString(modes[j], fromLeft + (50 * j), ty);
			}

			ty += 12;
		}
	}

	private void mediumArcs(Graphics g)
	{
		g.setColor(Color.black);
		// Now some arcs - first test is with a square
		g.drawRect(200, 60, 50, 50); // rectangle
		g.drawLine(200, 60, 250, 110); // join both corners
		g.drawLine(200, 110, 250, 60);
		g.drawLine(200, 85, 225, 60); // this should be a chord
		g.drawArc(200, 60, 50, 50, 45, 180); // the arc

		// These two tests act on a rectangular region (ie width != height)
		// Now the interesting thing is that they don't fit the arc within
		// the rectangle, but act on the width. This - by accident - matches the
		// way the Linux JDK operates...

		// Now the same test, but with a rectangle
		g.drawRect(300, 50, 80, 40);
		g.drawLine(300, 50, 380, 90);
		g.drawLine(300, 90, 380, 50);
		g.drawArc(300, 50, 80, 40, 135, 180);

		// Again the same test, but we will fill the arc
		g.drawRect(400, 50, 40, 80);
		g.drawLine(400, 50, 440, 130);
		g.drawLine(400, 130, 440, 50);
		g.setColor(Color.blue);
		g.fillArc(400, 50, 40, 80, 135, 180);
		g.setColor(Color.black);

		// Repeat again, but this time with different angles to the arc.
		// We do this to compare how Java and PDF render the arcs
		g.drawRect(400, 150, 40, 80);
		g.drawLine(400, 150, 440, 230);
		g.drawLine(400, 230, 440, 150);
		g.setColor(Color.blue);
		g.fillArc(400, 150, 40, 80, 135, 225);
	}

	private void bigArcs(Graphics g)
	{
		g.setColor(Color.black);
		// June 20, 2001 ezb - Looks like ovals work as expected
		g.drawArc(100, 400, 100, 200, 0, 360);
		g.drawOval(200, 400, 100, 200);
		g.fillOval(300, 400, 100, 200);
	}

	// ==========================================================================
	//                       Main
	// ==========================================================================
	public static void main(String[] args)
	{
		AppStarter.startUpApp(args, "org.gerhardb.jibs.Jibs", true);
		PageableTest pageableTest = new PageableTest("D:/dev/gnuJpdf/a.jpg");
		new PageablePreviewer(pageableTest, true, null);
	}
}
