/**
 * ****************************************************************************
 *
 * Classification: UNCLASSIFIED
 *
 * $Header: /cvsroot/img-browse-sort/jibs-swing/com/saic/isd/swing/filechooser/SimpleFileFilter.java,v 1.3 2006/05/14 15:14:40 gerhardb Exp $
 *
 * SAIC (C) Copyright 2004 Science Applications International Corporation.
 * All Rights Reserved.
 *
 * Developed by Science Applications International Corporation under
 * contract to the Department of Defense. All government restrictions
 * on software distribution apply.
 *
 * Subject to the GNU LESSER GENERAL PUBLIC LICENSE,
 * Version 2.1, February 1999 http://www.gnu.org/copyleft/lesser.html
 * A copy of that license is included with the source distribution.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * Peer Reviewed:
 * January 23, 2004
 * Craig Pell, Alvin Rearick, Ralph Finley, Gerhard Beck, Julie Popp
 *
 * Revision History:
 * $Lo$
 * ****************************************************************************
 */

package com.saic.isd.swing.filechooser;

import java.io.File;

import javax.swing.filechooser.FileFilter;

/**
 * A file filter which determines what files to accept for a given extension,
 * also accepting all directories.
 * An extension refers to the part of a file name after the final period.
 * Here are some examples:<br>
 * file.txt - txt is the extension<br>
 * network.xml - xml is the extension<br>
 * SAIC.RunThis.pl - pl is the extension<br>
 * a.ridiculus.example.doc - doc is the extension<br>
 * aSampleFile - there is no extension
 * <p>
 * Code originally borrowed from: <br>
 * Sun's ExampleFileFilter version 1.9 by Jeff Dinkins
 */
public class SimpleFileFilter
    extends FileFilter
{
   String myExtension;
   String myDescription;
   boolean iAmCaseSensitive = false;

   /**
    * Creates a new FileFilter which will accept files based on the
    * extension provided, ignoring case.
    * @param extension Required, may not contain a period.
    */
   public SimpleFileFilter( String extension ) // NO_UCD

   {
      this( extension, null, false );
   }

   /**
    * Creates a new FileFilter which will accept files based on the
    * extension provided, ignoring case.
    * @param extension Required, may not contain a period.
    * @param description
    */
   public SimpleFileFilter(
       String extension,
       String description )
   {
      this( extension, description, false );
   }

   /**
    * Creates a new FileFilter which will accept files based on the
    * extension provided.
    * @param extension Required, may not contain a period.
    * @param description
    * @param caseSensitive if true the extension's case must match exactly.
    */
   public SimpleFileFilter(
       String extension,
       String description,
       boolean caseSensitive )
   {
      if ( extension == null )
      {
         throw new IllegalArgumentException( "Extension required" ); //$NON-NLS-1$
      }
      this.myExtension = extension;
      this.myDescription = description;
      this.iAmCaseSensitive = caseSensitive;
   }

   // ==========================================================================
   //                   FileFilter Implementaion
   // ==========================================================================

   // Pull in interface comment.
   @Override
	public String getDescription()
   {
      return this.myDescription;
   }

   /**
    * Tests a file for acceptability.
    * @param Directories are accepted as are files with the correct
    * extension.  Accepting directories allows the user to navigate in
    * a file chooser dialog.
    * @return Whether the given file is accepted by this filter.
    */
   @Override
	public boolean accept( File f )
   {
      if ( f != null )
      {
         if ( f.isDirectory() )
         {
            return true;
         }
         String extension = getExtension( f );
         if ( extension != null )
         {
            if ( this.iAmCaseSensitive && extension.equals( this.myExtension ) )
            {
               return true;
            }
            return extension.equalsIgnoreCase( this.myExtension );
         }
      }
      return false;
   } // accept

   // ==========================================================================
   //                        Public Methods
   // ==========================================================================

   /**
    * Gets the extendsion.
    * @return extenion used for this filter
    */
   public String getExtension()
   {
      return this.myExtension;
   }

   /**
    * Adds the extension to a file if it does not have that extension already.
    * @param file
    * @return file with the extesion added
    */
   public File attachExtension( File file )
   {
      if ( accept( file ) )
      {
         return file;
      }
		return new File( file.getPath() + "." + this.myExtension ); //$NON-NLS-1$
   }

   /**
    * Returns the extension of a file.
    * @param f nulls not permitted.
    * @return extension returns null if there is no extension.
    */
   public static String getExtension( File f )
   {
      String filename = f.getName();
      int i = filename.lastIndexOf( '.' );
      if ( i > 0 && i < filename.length() - 1 )
      {
         return filename.substring( i + 1 );
      }
      return null;
   }

}
