/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package org.gerhardb.jibs.camera;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.io.IOException;

import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JFileChooser;
import javax.swing.JFormattedTextField;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.JTextField;
import javax.swing.border.LineBorder;
import javax.swing.border.TitledBorder;

import org.gerhardb.jibs.Jibs;
import org.gerhardb.jibs.util.FileLoopHelper;
import org.gerhardb.lib.swing.JPanelRows;

import com.saic.isd.printscreen.Util;
import com.saic.isd.util.report.Report;

/**
 * User interface for the Camera.
 */
public class CameraPanel
   extends JPanelRows
{
   Camera myCamera;
   WorkerCamera myWorker;
   
   JProgressBar myProgressBar = new JProgressBar();
   JCheckBox myRenameUniqueCheckBox = new JCheckBox(Jibs.getString("CameraPanel.0")); //$NON-NLS-1$
   JCheckBox myRenameEXIFCheckBox = new JCheckBox(Jibs.getString("CameraPanel.1")); //$NON-NLS-1$
   JLabel myCurrentAction = new JLabel(Jibs.getString("CameraPanel.2")); //$NON-NLS-1$
   File myLogFile;

   private static final int DIR_COL_WIDTH = 50;
   JFormattedTextField pictureNumberFld;
   JFormattedTextField logNumberFld;
   JTextField logFld;
   JButton myStartBtn = new JButton(Jibs.getString("CameraPanel.3")); //$NON-NLS-1$

   //===========================================================================
   //                          Constructor
   //===========================================================================

   public CameraPanel(Camera Camera)
   {
      super();
      this.myCamera = Camera;

      this.pictureNumberFld = FileLoopHelper.makeRelabelFld();
      this.logNumberFld = FileLoopHelper.makeRelabelFld();

      this.myRenameEXIFCheckBox.setSelected(CameraPreferences.getExifRenaming());
      if (CameraPreferences.getUniqueRenamingCamera())
      {
         this.myRenameUniqueCheckBox.setSelected(true);
         this.pictureNumberFld.setEditable(true);
      }
      else
      {
         this.myRenameUniqueCheckBox.setSelected(false);
         this.pictureNumberFld.setEditable(false);
      }

      this.myRenameUniqueCheckBox.addActionListener
         (
         new ActionListener()
      {
         public void actionPerformed(ActionEvent ae)
         {
            CameraPanel.this.pictureNumberFld.setEditable(CameraPanel.this.myRenameUniqueCheckBox.isSelected());
         }
      });

      this.pictureNumberFld.setValue(
         new Integer(CameraPreferences.getRelabelNumber()));
      this.pictureNumberFld.setColumns(10);

      this.logNumberFld.setValue(
         new Integer(CameraPreferences.getNextLogNumber()));
      this.logNumberFld.setColumns(10);

      this.logFld = new JTextField(CameraPreferences.getLogDirectory());
      this.logFld.setColumns(DIR_COL_WIDTH);
      this.logFld.setToolTipText(Jibs.getString("CameraPanel.4") //$NON-NLS-1$
         + Jibs.getString("CameraPanel.5")); //$NON-NLS-1$

      // Layout RENAME
      JPanelRows renamePanel = new JPanelRows();
      renamePanel.setBorder(
         new TitledBorder(
         new LineBorder(Color.BLACK),
         Jibs.getString("CameraPanel.6"), TitledBorder.LEFT, TitledBorder.TOP)); //$NON-NLS-1$

      JPanel aRow = renamePanel.topRow();
      aRow.add(this.myRenameEXIFCheckBox);
      aRow.add(new JLabel(
         Jibs.getString("CameraPanel.7"))); //$NON-NLS-1$

      aRow = renamePanel.nextRow();
      aRow.add(this.myRenameUniqueCheckBox);
      aRow.add(new JLabel(Jibs.getString("CameraPanel.9") + Jibs.getString("colon") + " ")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
      aRow.add(this.pictureNumberFld);

      // Layout LOG
      JButton btnLog = new JButton("..."); //$NON-NLS-1$
      btnLog.addActionListener
         (
         new ActionListener()
      {
         public void actionPerformed(ActionEvent ae)
         {
            String dir = CameraPanel.this.logFld.getText();
            if (dir == null)
            {
               dir = System.getProperty("user.home"); //$NON-NLS-1$
            }
            dir = lookupDir(Jibs.getString("CameraPanel.12"), dir); //$NON-NLS-1$
            CameraPanel.this.logFld.setText(dir);
            CameraPreferences.setLogDirectory(dir);
         }
      }
      );

      JPanelRows logPanel = new JPanelRows();
      logPanel.setBorder(
         new TitledBorder(
         new LineBorder(Color.BLACK),
         Jibs.getString("CameraPanel.13"), TitledBorder.LEFT, TitledBorder.TOP)); //$NON-NLS-1$

      aRow = logPanel.topRow();
      aRow.add(new JLabel(Jibs.getString("CameraPanel.14") + Jibs.getString("colon") + " ")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
      aRow.add(this.logNumberFld);

      aRow = logPanel.nextRow();
      aRow.add(new JLabel(Jibs.getString("CameraPanel.15") + Jibs.getString("colon") + " ")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
      aRow.add(this.logFld);
      aRow.add(btnLog);

      // Layout MAIN PANEL
      aRow = this.topRow();
      aRow.add(renamePanel);

      aRow = this.nextRow();
      aRow.add(logPanel);

      aRow = this.nextRow();
      aRow.add(layoutAction());

   }

   private JPanel layoutAction()
   {
   	final JButton btnStop = new JButton(Jibs.getString("CameraPanel.16")); //$NON-NLS-1$
      final JButton btnReport = new JButton(Jibs.getString("report")); //$NON-NLS-1$
      btnReport.setEnabled(false);

      JPanelRows rowMaker = new JPanelRows();

      JPanel currentRow = rowMaker.topRow();
      currentRow.add(this.myStartBtn);
      currentRow.add(btnStop);
      currentRow.add(btnReport);

      currentRow = rowMaker.nextRow();
      currentRow.add(new JLabel(" " + " " + Jibs.getString("CameraPanel.18") + Jibs.getString("colon") + " ")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$
      currentRow.add(this.myCurrentAction);

      currentRow = rowMaker.nextRow();
      currentRow.add(this.myProgressBar);
      Dimension dim = this.myProgressBar.getPreferredSize();
      dim.width = 500;
      this.myProgressBar.setPreferredSize(dim);

      this.myStartBtn.addActionListener(new java.awt.event.ActionListener()
      {
         public void actionPerformed(java.awt.event.ActionEvent evt)
         {
            doWork();
            save();
            btnReport.setEnabled(true);
            btnStop.setEnabled(true);
         }
      });

      btnStop.addActionListener(new java.awt.event.ActionListener()
      {
         public void actionPerformed(java.awt.event.ActionEvent evt)
         {
         	CameraPanel.this.myWorker.stop();
            btnStop.setEnabled(false);            
         }
      });
      
      btnReport.addActionListener(new java.awt.event.ActionListener()
      {
         public void actionPerformed(java.awt.event.ActionEvent evt)
         {
            try
            {
               String text = Util.getTextFromFile(CameraPanel.this.myLogFile);
               Report.showReport(
                  Jibs.getString("CameraPanel.30"), text, false); //$NON-NLS-1$
            }
            catch (Exception ex)
            {
               Report.showReport(
                  Jibs.getString("CameraPanel.20"), //$NON-NLS-1$
                  ex.getMessage(), false);
            }
         }
      });

      JPanel rtnMe = new JPanel(new BorderLayout());
      rtnMe.add(rowMaker, BorderLayout.NORTH);

      return rtnMe;
   }

   String lookupDir(String msg, String prior)
   {
      JFileChooser chooser = new JFileChooser();
      chooser.setDialogTitle(msg);
      chooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
      chooser.setCurrentDirectory(new File(prior));

      int returnVal = chooser.showOpenDialog(this);
      if (returnVal == JFileChooser.APPROVE_OPTION)
      {
         try
         {
            return chooser.getSelectedFile().getCanonicalPath();
         }
         catch (IOException ex)
         {
            ex.printStackTrace();
         }
      }
      return null;
   } // lookupDir()

   //===========================================================================
   //===========================================================================
   //                           WORK FUNCTIONS
   //===========================================================================
   //===========================================================================
   /**
    */
   void doWork()
   {
      int logNumber = FileLoopHelper.getIntFromField(this.logNumberFld, 0);
      int nextLogNumber = logNumber + 1;
      this.logNumberFld.setValue(new Integer(Integer.toString(nextLogNumber)));
      String logFileName =
         CameraPreferences.getLogDirectory()
         + System.getProperty("file.separator") + "jibsLog_" //$NON-NLS-1$ //$NON-NLS-2$
         + logNumber + ".txt"; //$NON-NLS-1$
      this.myLogFile = new File(logFileName);

      CameraPreferences.setUniqueRenamingCamera(
         this.myRenameUniqueCheckBox.isSelected());
      CameraPreferences.setExifRenaming(
         this.myRenameEXIFCheckBox.isSelected());

      try
      {
         FileLoopHelper helper =
            new FileLoopHelper(this.myCamera, "JIBS", this.pictureNumberFld, //$NON-NLS-1$
            this.myCurrentAction, this.myProgressBar, this.myLogFile);
         this.myWorker = new WorkerCamera(
                               this.myCamera.myWorkingList.getDirArray(),
                               this.myRenameUniqueCheckBox.isSelected(),
                               this.myRenameEXIFCheckBox.isSelected(),
                               helper
                               );

         // Evaluate this thread for EventQueue.invokeLater(new Runnable()
         Thread t = new Thread(this.myWorker);
         t.start();
      }
      catch (Exception ex)
      {
         JOptionPane.showMessageDialog(
            this,
            ex.getMessage(),
            Jibs.getString("CameraPanel.25"), //$NON-NLS-1$
            JOptionPane.ERROR_MESSAGE);
         ex.printStackTrace();
      }
   }

   void save()
   {
      String log = this.logFld.getText();
      if (log != null
         && log.length() > 0
         && !FileLoopHelper.checkDirectory(log, this.myStartBtn))
      {
         JOptionPane.showMessageDialog(
            this.myStartBtn,
            Jibs.getString("CameraPanel.26") //$NON-NLS-1$
            + Jibs.getString("CameraPanel.27"), //$NON-NLS-1$
            Jibs.getString("save.failed"), //$NON-NLS-1$
            JOptionPane.WARNING_MESSAGE);
         return;
      }
      CameraPreferences.setLogDirectory(this.logFld.getText());

      try
      {
         CameraPreferences.setRelabelNumber(
            FileLoopHelper.getIntFromField(this.pictureNumberFld, 0));
         CameraPreferences.setLogNumber(
            FileLoopHelper.getIntFromField(this.logNumberFld, 0));
      }
      catch (Exception ex)
      {
         ex.printStackTrace();
         JOptionPane.showMessageDialog(
            this.myStartBtn,
            ex.getMessage(),
            Jibs.getString("save.failed"), //$NON-NLS-1$
            JOptionPane.ERROR_MESSAGE);
      }
   }
}
