/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
package org.gerhardb.jibs.camera;

import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;

import org.gerhardb.lib.io.FileUtil;

/**
 * Wraps the preferences used by the viewer and provides an way of editing
 * the preferences.
 * @author  Gerhard Beck
 */
public class CameraPreferences
{
  	private static Preferences clsPrefs = Preferences.userRoot().node("/org/gerhardb/jibs/camera/CameraPreferences"); //$NON-NLS-1$

   //===========================================================================
   //                          Constructor
   //===========================================================================

   private CameraPreferences()
   {
		// Don't allow public creation
   }


   //===========================================================================
   //                          Log Directory
   //===========================================================================
   private static final String LOG_DIRECTORY = "logDirectory"; //$NON-NLS-1$
   private static final String LOG_NUMBER = "logNumber"; //$NON-NLS-1$

   /**
    * @return null if not correctly set.
    */
   public static String getLogDirectory()
   {
      String rtnMe = FileUtil.checkDirectory(clsPrefs.get(
      		LOG_DIRECTORY,
         System.getProperty("user.home"))); 
      if (rtnMe == null)
      {
      	rtnMe = System.getProperty("user.home");
      }
      return rtnMe;
   }

   public static void setLogDirectory(String dir)

   {
      String validated = FileUtil.safeValidateDirectory(dir);
      if (validated != null)
      {
         clsPrefs.put(LOG_DIRECTORY, validated);
      }
      else
      {
         clsPrefs.remove(LOG_DIRECTORY);
      }
   }

   public static int getNextLogNumber()
   {
      int count = clsPrefs.getInt(LOG_NUMBER, 0);
      count++;
      try
      {
         setLogNumber(count);
      }
      catch (Exception ex)
      {
         ex.printStackTrace();
      }
      return count;
   }

   public static void setLogNumber(int setting)
      throws Exception
   {
      clsPrefs.putInt(LOG_NUMBER, setting);
      clsPrefs.flush();
   }

   //===========================================================================
   //                          Rename Options
   //===========================================================================
   private static final String RELABEL_NUMBER = "relabelNumber"; //$NON-NLS-1$
   private static final String DIR_UTIL_UNIQUE_CAM = "uniqueRenamingCamera"; //$NON-NLS-1$
   private static final String DIR_UTIL_EXIF_RENAME = "exifRenaming"; //$NON-NLS-1$

   public static int getRelabelNumber()
   {
      return clsPrefs.getInt(RELABEL_NUMBER, 0);
   }

   public static void setRelabelNumber(int setting)
      throws Exception
   {
      clsPrefs.putInt(RELABEL_NUMBER, setting);
      clsPrefs.flush();
   }

   public static boolean getUniqueRenamingCamera()
   {
      return clsPrefs.getBoolean(DIR_UTIL_UNIQUE_CAM, false);
   }

   public static void setUniqueRenamingCamera(boolean setting)
   {
      clsPrefs.putBoolean(DIR_UTIL_UNIQUE_CAM, setting);
   }

   public static boolean getExifRenaming()
   {
      return clsPrefs.getBoolean(DIR_UTIL_EXIF_RENAME, false);
   }

   public static void setExifRenaming(boolean setting)
   {
      clsPrefs.putBoolean(DIR_UTIL_EXIF_RENAME, setting);
   }

   //===========================================================================
   //                              Other Public Functions
   //===========================================================================
   public static void flush()
   {
      try
      {
         clsPrefs.flush();
      }
      catch (BackingStoreException ex)
      {
      	ex.printStackTrace();
      }
   }

}
