/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package org.gerhardb.jibs.camera;

import java.io.File;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;

import org.gerhardb.jibs.Jibs;
import org.gerhardb.jibs.util.FileLoopHelper;
import org.gerhardb.lib.io.EzLogger;
import org.gerhardb.lib.image.ImageUtilDrew;


/**
 * Does all the actual filtering work.
 * <p>
 * Orginally intended to be able to be driven from a command line, but that
 * go lost in the suffle.
 * <p>
 * Notes: All files without an image file name are automatically DELETED!!!.
 */
public class WorkerCamera
   implements Runnable
{
   public static final String FILE_DATE_FORMAT = "yyyy.MM.dd-HH.mm.ss"; //$NON-NLS-1$
   public static final SimpleDateFormat FILE_DATE =
      new SimpleDateFormat(FILE_DATE_FORMAT);

   public static final SimpleDateFormat LONG_DATE =
      new SimpleDateFormat("yyyy.MMMMM.dd HH:mm:ss"); //$NON-NLS-1$

    // PASSED IN
   File[] myDirectories;
   FileLoopHelper myHelper = null;

   // OTHER VARIABLES
   File[] myFiles;
   boolean iStop;

   // RelabelUnique
   int myRelabelCount = CameraPreferences.getRelabelNumber();

   // PASSED IN
   boolean iRenameUnique;
   boolean iRenameEXIF;

   //===========================================================================
   //                              Constructor
   //===========================================================================
   public WorkerCamera(
      File[] directories,
      boolean rename,
      boolean exif,
      FileLoopHelper helper
      )
   throws Exception
   {
      this.myDirectories = directories;
      this.myHelper = helper;

      this.myHelper.setValue(0);
      this.myHelper.setMaximum(1);
      this.myHelper.setIndeterminate(false);

      this.myHelper.getLog().logLine(Jibs.getString("WorkerCamera.4") + Jibs.getString("colon") + " "); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
      for (int i = 0; i < this.myDirectories.length; i++)
      {
         this.myHelper.getLog().logLine("   " + this.myDirectories[i].getAbsolutePath() + ""); //$NON-NLS-1$ //$NON-NLS-2$
      }


      this.iRenameUnique = rename;
      this.iRenameEXIF = exif;
      
      this.myHelper.getLog().logLine(Jibs.getString("WorkerCamera.7") + Jibs.getString("colon") + " "); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
      this.myHelper.getLog().logLine("   " + Jibs.getString("WorkerCamera.8") + Jibs.getString("colon") + " " + this.iRenameUnique + ""); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$
      if (this.iRenameUnique)
      {
         this.myHelper.getLog().logLine("      " + Jibs.getString("WorkerCamera.10") + Jibs.getString("colon") + " " //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
            + FileLoopHelper.RELABEL_POSTFIX + ""); //$NON-NLS-1$
         this.myHelper.getLog().logLine(""); //$NON-NLS-1$
         this.myHelper.getLog().logLine("      " + Jibs.getString("WorkerCamera.13") + Jibs.getString("colon") + " " + this.myRelabelCount + ""); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$
      }
      this.myHelper.getLog().logLine("   " + Jibs.getString("WorkerCamera.3") + Jibs.getString("colon") + " " + this.iRenameEXIF + ""); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$
      this.myHelper.getLog().logLine(EzLogger.DIVIDER);
   }

   void stop()
   {
   	this.iStop = true;
   }
   // ==========================================================================
   // Runnable Interface
   // ==========================================================================
   public void run()
   {
      this.iStop = false;
      this.myHelper.getLog().logLine(EzLogger.DIVIDER);
      Date startDate = new Date();
      this.myHelper.getLog().logLine(
         Jibs.getString("WorkerCamera.2") + Jibs.getString("colon") + " " + LONG_DATE.format(startDate) + ""); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$

      if (continueRun())
      {
         this.myHelper.getLabel().setText(Jibs.getString("WorkerCamera.1")); //$NON-NLS-1$

         ArrayList<File> outList = this.myHelper.getFilesFromDirectories(this.myDirectories);
         this.myFiles = outList.toArray(new File[outList.size()]);

         this.myHelper.getLog().logLine(EzLogger.DIVIDER);
         this.myHelper.getLog().logLine(Jibs.getString("WorkerCamera.0") + Jibs.getString("colon") + " " + this.myFiles.length + ""); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
         this.myHelper.setValue(0);
         this.myHelper.setMaximum(this.myFiles.length - 1);

         for (int i = 0; i < this.myFiles.length; i++)
         {
         	if (this.iStop){break;}
            this.myHelper.setValue(i);
            int count = i + 1;
            String progress = Jibs.getString("WorkerCamera.37") + " " + count + " " + Jibs.getString("WorkerCamera.38") + " " + this.myFiles.length; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$
            this.myHelper.getLabel().setText(progress);

            // File could be null because of dedup process
            // That is, nulling the file out is how to indicate
            // the file has been fully processed.
            if (this.myFiles[i] != null)
            {
               processIndex(i);
               // We don't need this in memory any more!
               this.myFiles[i] = null;
            }
         }
         this.myHelper.getLog().logLine(EzLogger.DIVIDER);
         this.myHelper.getLog().logLine(Jibs.getString("WorkerCamera.39") + " " + this.myFiles.length + " " + Jibs.getString("WorkerCamera.40")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
      }

      Date stopDate = new Date();
      this.myHelper.getLog().logLine(
         Jibs.getString("WorkerCamera.41") + Jibs.getString("colon") + " " + LONG_DATE.format(stopDate) + ""); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$

      long seconds = ((stopDate.getTime() - startDate.getTime()) / 1000);
      this.myHelper.getLog().logLine(
         Jibs.getString("WorkerCamera.42") + Jibs.getString("colon") + " " + seconds + ""); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$

      this.myHelper.getLabel().setText(Jibs.getString("WorkerCamera.30")); //$NON-NLS-1$
      try
      {
         this.myHelper.getLog().close();
      }
      catch(Exception ex)
      {
         ex.printStackTrace();
      }
      this.myHelper.updateCountField();
      try{CameraPreferences.setRelabelNumber(this.myHelper.getRelableCount());}
      catch(Exception ex )
		{
			ex.printStackTrace();
		}				

      this.myHelper.done();
   }

   // ==========================================================================
   // Worker Interface
   // ==========================================================================
   /**
    * Does setup and first pass.  In some cases that may be all that is needed.
    * @return true to continue; false if there is no more work to be done.
    * Abort processing each file.
    */
   boolean continueRun()
   {
      if (this.iRenameUnique || this.iRenameEXIF)
      {
         return true;
      }
		return false;
   }

   //===========================================================================
   //                              Private Functions
   //===========================================================================
   void processIndex(int index)
   {
      // File could be null because of prior processing.
      if (this.myFiles[index] == null)
      {
         return;
      }

      final File oldName = this.myFiles[index];
      File newName = oldName;
      if (this.iRenameEXIF)
      {
         newName = renameExif(oldName);
      }
      if (this.iRenameUnique)
      {
         // Got to use new name in case EXIF changed it.
         newName = this.myHelper.renameUnique(newName);
      }
      if (!oldName.equals(newName))
      {
         this.myHelper.getLog().logLine("   " + Jibs.getString("WorkerCamera.31") + Jibs.getString("colon") + " " + oldName.getAbsolutePath() //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
            + "      " + Jibs.getString("WorkerCamera.32") + Jibs.getString("colon") + " " + newName.getAbsolutePath() + ""); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$
      }
   }

   private File renameExif(File currFile)
   {

      // Extract the EXIF date information.
      try
      {
      	ImageUtilDrew drew = new ImageUtilDrew(currFile);
      	Date picDate = drew.getPicDate();
         if (picDate != null)
         {
            String newName = currFile.getParent() + File.separator
                             + FILE_DATE.format(picDate) + ".jpg"; //$NON-NLS-1$
            File newFile = new File(newName);
            // Only do the work if the name isn't correct already.
            if (!newFile.equals(currFile))
            {
               //System.out.println("Renaming: " + name + " to: " + newName);
               currFile.renameTo(newFile);
               //System.out.println(newFile.getAbsolutePath());

               // Let next proces know the new name.
               return newFile;
            }
         }
      }
      catch (Exception ex)
      {
         ex.printStackTrace();
      }
      return currFile;
   }
}
