/*
 * Copyright (c) 2007 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.gerhardb.jibs.concatenater;

import java.awt.*;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;
import java.util.*;
import java.util.prefs.Preferences;

import javax.swing.DefaultBoundedRangeModel;
import javax.swing.JButton;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.JTextField;
import org.gerhardb.jibs.Jibs;
import org.gerhardb.lib.io.FileNameComparatorInsensative;
import org.gerhardb.lib.swing.JFileChooserExtra;
import org.gerhardb.lib.swing.JPanelRows;
import org.gerhardb.lib.swing.SwingUtils;
import org.gerhardb.lib.util.Icons;
import org.gerhardb.lib.util.StopCheckButton;
import org.gerhardb.lib.util.startup.AppStarter;

/**
 */

public class AutoConcatenate extends JFrame
{
	public static String APP_NAME = "AutoConcatenate";
	private static final String LAST_DIR = "LastDir"; //$NON-NLS-1$
	private static final String READY = "Ready"; //$NON-NLS-1$

	private static final Preferences clsPrefs = Preferences.userRoot().node("/org/gerhardb/jibs/concatenater/auto"); //$NON-NLS-1$

	JTextField myRootDir = new JTextField(60);
	DefaultBoundedRangeModel myRange = new DefaultBoundedRangeModel();
	JProgressBar myBar = new JProgressBar(this.myRange);
	StopCheckButton myStopCheckBtn;

	public AutoConcatenate(boolean iExitOnClose)
	{
		super("Auto Joiner"); //$NON-NLS-1$
		this.myStopCheckBtn = new StopCheckButton();

		layoutComponents();
		this.myRootDir.setText(clsPrefs.get(LAST_DIR, null));

		this.setIconImage(Icons.getIcon(Icons.JIBS_16).getImage());

		if (iExitOnClose)
		{
			this.addWindowListener(new WindowAdapter()
			{
				@Override
				public void windowClosing(WindowEvent evt)
				{
					System.exit(0);
				}
			});
		}

		EventQueue.invokeLater(new Runnable()
		{
			public void run()
			{
				AutoConcatenate.this.pack();
				SwingUtils.centerOnScreen(AutoConcatenate.this);
				AutoConcatenate.this.setVisible(true);
			}
		});
		

		//Dimension dim = myBar.getSize();
		//myBar.setSize(300, dim.height);
	}

	private void layoutComponents()
	{
		this.myBar.setStringPainted(true);
		this.myBar.setString(READY); 
		
		// Set up application
		this.setSize(new Dimension(600, 600));

		JButton goBtn = new JButton("Auto Join"); //$NON-NLS-1$
		goBtn.addActionListener(new java.awt.event.ActionListener()
		{
			public void actionPerformed(java.awt.event.ActionEvent evt)
			{
				go();
			}
		});

		JButton rootBtn = new JButton("..."); //$NON-NLS-1$
		rootBtn.addActionListener(new java.awt.event.ActionListener()
		{
			public void actionPerformed(java.awt.event.ActionEvent evt)
			{
				selectRoot();
			}
		});

		JPanelRows topPanel = new JPanelRows(FlowLayout.CENTER);
		JPanel aRow = topPanel.topRow();
		aRow
				.add(new JLabel(
						"Automatically joins files in a sequence like frodo.mpg.001, frodo.mpg.002 to frodo.mpg")); //$NON-NLS-1$

		aRow = topPanel.nextRow();
		aRow.add(new JLabel(" ")); //$NON-NLS-1$

		aRow = topPanel.nextRow();
		aRow.add(new JLabel(
				"Select the diretory containing the files to be joined.")); //$NON-NLS-1$

		aRow = topPanel.nextRow();
		aRow.add(new JLabel("Directory" + Jibs.getString("colon") + " ")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		aRow.add(this.myRootDir);
		aRow.add(rootBtn);

		aRow = topPanel.nextRow();
		aRow.add(goBtn);
		aRow.add(this.myStopCheckBtn.getStopButton());

		aRow = topPanel.nextRow(new BorderLayout());
		aRow.add(this.myBar, BorderLayout.CENTER);

		this.setContentPane(topPanel);
	}

	void selectRoot()
	{
		this.myBar.setString(READY); 
		JFileChooserExtra chooser = new JFileChooserExtra(clsPrefs.get(LAST_DIR,
				null));
		chooser.setSaveName(APP_NAME, Jibs.getString("RecreateDirectories.21")); //$NON-NLS-1$
		chooser.setApproveButtonText(Jibs.getString("RecreateDirectories.22")); //$NON-NLS-1$
		chooser.setDialogTitle("Directory to scan for auto-join"); //$NON-NLS-1$
		chooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
		chooser.setMultiSelectionEnabled(false);

		int returnVal = chooser.showOpenDialog(this);
		if (returnVal == JFileChooser.APPROVE_OPTION)
		{
			File picked = chooser.getSelectedFile();
			if (picked != null)
			{
				String rootString = picked.toString();
				this.myRootDir.setText(rootString);
				try
				{
					clsPrefs.put(LAST_DIR, rootString);
					clsPrefs.flush();
				}
				catch (Exception ex)
				{
					ex.printStackTrace();
				}
			}
		}
	}

	void go()
	{
		String rootString = this.myRootDir.getText();
		File rootFile = new File(rootString);
		if (!rootFile.exists())
		{
			JOptionPane
					.showMessageDialog(this, "Directory does not exist: "
							+ rootString, "Problem with Diretory",
							JOptionPane.ERROR_MESSAGE);
			return;
		}

		//System.out.println("RootString: " + rootString);
		// Remember for next time...
		try
		{
			clsPrefs.put(LAST_DIR, rootString);
			clsPrefs.flush();
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
		}
		
		if (this.myStopCheckBtn.isStopped()) { return; }
		setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
		try
		{
			ConcatAll ca = new ConcatAll(rootFile);
			//ca.run();
			new Thread(ca).start(); 
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
			JOptionPane.showMessageDialog(this, ex.getMessage(), Jibs
					.getString("Auto-Joiner"), //$NON-NLS-1$
					JOptionPane.ERROR_MESSAGE);
		}
		setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
	}
	
	private class ConcatAll implements Runnable
	{
		String fileName = null;
		String newFileName = null;
		ArrayList<File> list = null;
		File myCARootDir;
		
		ConcatAll(File rootDir)
		{
			this.myCARootDir = rootDir;
		}
		
		public void run()
		{
			// Boundary condition if list of files in directory is a single list to join.
			// That is, last file in list is one of the files to join so finish not triggered by next file.
			boolean needToFinish = false;
			System.out.println("Root Dir is: " + this.myCARootDir);
			File[] files = this.myCARootDir.listFiles();
			Arrays.sort(files, new FileNameComparatorInsensative() );
			for(int i = 0; i<files.length; i++)
			{
				if (AutoConcatenate.this.myStopCheckBtn.isStopped()) { return; }
				this.fileName = files[i].getName();
				System.out.println(files[i]);
				// If we find a starting file, start adding the files to the 
				// array until you find a different type of file then finish
				// and process the array you have.
				if (this.fileName.endsWith(".01") || this.fileName.endsWith(".001") || this.fileName.endsWith(".0001"))
				{
					start();
					needToFinish = true;
				}
				if (this.list != null)
				{
					if (fileToAdd())
					{
						this.list.add(files[i]);
					}
					else
					{
						finish();
						needToFinish = false;
					}
				}
			}		
			if (needToFinish)
			{
				finish();				
			}
			AutoConcatenate.this.myBar.setString("Done"); 	
			System.out.println("ALL DONE!!!");
		}	
		
		private void start()
		{
			System.out.println("STARTING");
			if (this.list != null)
			{
				finish();
			}
			this.newFileName = this.fileName.substring(0, this.fileName.lastIndexOf("."));
			File saveTo = null;
			saveTo = new File(this.myCARootDir + File.separator + this.newFileName);
			if (!saveTo.exists())
			{
				System.out.println("=====> PROCESSING: " + saveTo);
				this.list = new ArrayList<File>(100);				
			}	
			else
			{
				System.out.println("=====> SKIPPING: " + saveTo);
			}
		}
		
		private boolean fileToAdd()
		{
			if (this.fileName.startsWith(this.newFileName))
			{
				 StringTokenizer st = new StringTokenizer(this.fileName, ".");
				 String lastToken = st.nextToken();
			     while (st.hasMoreTokens()) 
			     {
			   	  lastToken = st.nextToken();
			     }
			     try
			     {
			   	  Integer.parseInt(lastToken);
					  System.out.println("Adding: " + this.fileName);
			   	  return true;
			     }
			     catch(Exception ex)
			      {
			      	ex.printStackTrace();
			      }
			}
			System.out.println("Not Adding: " + this.fileName);
			return false;
		}
		
		private void finish()
		{
			System.out.println("FINISHING");
			if (this.list == null)
			{
				return;
			}
			File[] files = new File[this.list.size()];
			files = this.list.toArray(files);
			this.list = null;
			AutoConcatenate.this.myBar.setStringPainted(false);
			File saveTo = new File(this.myCARootDir + File.separator + this.newFileName);
			System.out.println("SAVING TO: " + saveTo);
			ThreadedConcatenater tc = new ThreadedConcatenater(saveTo, files,
					AutoConcatenate.this, AutoConcatenate.this.myStopCheckBtn, AutoConcatenate.this.myRange, AutoConcatenate.this.myBar, false);
			tc.run();
			//new Thread(tc).start();
		}
	}

	// ===========================================================================
	// Main
	// ===========================================================================
	public static void main(String[] args)
	{
		AppStarter.startUpApp(args, "org.gerhardb.jibs.Jibs", true);
		new AutoConcatenate(true);
	}
}
