/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package org.gerhardb.jibs.dirtools;

import java.io.File;
import java.io.FileReader;
import java.text.SimpleDateFormat;
import java.util.Date;

import org.gerhardb.jibs.Jibs;
import org.gerhardb.lib.io.EzLogger;


/**
 * Dedups directories
 * <p>
 * Notes: All files without an image file name are automatically DELETED!!!.
 */
public class WorkerDedup
   implements Runnable
{
   static final SimpleDateFormat LONG_DATE =
      new SimpleDateFormat("yyyy.MMMMM.dd HH:mm:ss"); //$NON-NLS-1$
   static final String FAILED = "FAILED	";
   static final String SKIPPED = "SKIPPED	";
   static final String ABORTED = "==> ABORTED <==";
   private static final String FILE_DATE_FORMAT = "yyyy.MM.dd-HH.mm.ss"; //$NON-NLS-1$
   private static final SimpleDateFormat FILE_DATE =
      new SimpleDateFormat(FILE_DATE_FORMAT);
    
   // OTHER VARIABLES
   boolean iStop;
   DirDedupPanel myPanel;
   EzLogger myLog;
   File mySacredStableDir;
   File myDedupTargetDir;
   File myLogFile;
   boolean iCheckFileContents = true;
   boolean iAmRunningOnWindows = false;
   
   //===========================================================================
   //                              Constructor
   //===========================================================================
   public WorkerDedup(
   	DirDedupPanel panel,
      String sacredStableDir,
      String dedupTargetDir, 
      boolean checkFileContents
      )
   throws Exception
   {
   	this.myPanel = panel;
   	this.mySacredStableDir = new File(sacredStableDir);
   	this.myDedupTargetDir = new File(dedupTargetDir); 
   	this.iCheckFileContents = checkFileContents;
   	
   	if (this.mySacredStableDir.equals(this.myDedupTargetDir))
   	{
   		throw new WorkerDedupException("Target directory must be different than stable directory");
   	}
   	if (this.mySacredStableDir.getAbsolutePath().startsWith(this.myDedupTargetDir.getAbsolutePath()))
   	{
   		throw new WorkerDedupException("Stable directory may not be child of target directory");
   	}
      
      String date = FILE_DATE.format(new Date());      
      String logFileName =
      	sacredStableDir
         + System.getProperty("file.separator") + "jibsLog_" 
         + date + ".txt"; //$NON-NLS-1$
      this.myLogFile = new File(logFileName);
      this.myLog = new EzLogger(this.myLogFile);

      this.myLog.logLine("Stable Directory (leave alone): " + this.mySacredStableDir);    
      this.myLog.logLine("Dedup Directory (remove from here): " + this.myDedupTargetDir);    
      
		String os = System.getProperty("os.name").toLowerCase(); //$NON-NLS-1$
		if (os.startsWith("windows")) //$NON-NLS-1$
		{
			this.iAmRunningOnWindows = true;
		}
   }

   void stop()
   {
   	this.iStop = true;
   }
   
   // ==========================================================================
   // Runnable Interface
   // ==========================================================================
   public void run()
   {
      this.iStop = false;
      
      Date startDate = new Date();
      this.myLog.logLine("Started" + Jibs.getString("colon") + " " 
      		+ LONG_DATE.format(startDate) + ""); 
      this.myLog.logLine(EzLogger.DIVIDER);
      this.myPanel.myCurrentAction.setText("Started"); 

		processDirectory(this.mySacredStableDir, this.myDedupTargetDir);

      this.myLog.logLine(EzLogger.DIVIDER);
      Date stopDate = new Date();
      this.myLog.logLine(
         "Completed" + Jibs.getString("colon") + " " 
         + LONG_DATE.format(stopDate) + ""); 
      this.myPanel.myCurrentAction.setText("Stopped"); 

      try
      {
      	this.myLog.close();
      }
      catch(Exception ex)
      {
         ex.printStackTrace();
      }
      
      this.myPanel.myStopBtn.setEnabled(false);
   }

   //===========================================================================
   //                              Private Functions
   //===========================================================================
   class WorkerDedupException extends Exception
   {
   	WorkerDedupException(String msg)
   	{
   		super(msg);
   	}
   }
   
   private void processDirectory(File stableDir, File dedupTargeDir)
   {
   	if (this.iStop)
   	{
   		this.myLog.logLine(ABORTED);
   		return;
   	}
   	
		if (stableDir == null || !stableDir.isDirectory())
		{
		   this.myLog.logLine(FAILED + "Stable is null or not a directory: " + stableDir);   		
		   return;
		}
   	
		if (dedupTargeDir == null || !dedupTargeDir.isDirectory())
		{
		   this.myLog.logLine(FAILED + "Target is null or not a directory: " + dedupTargeDir);   		
		   return;
		}
		
      this.myLog.logLine(EzLogger.DIVIDER);
      this.myLog.logLine("Stable Directory: " + stableDir.getAbsolutePath());
      this.myLog.logLine("Dedup Directory: " + dedupTargeDir.getAbsolutePath());
		
      this.myPanel.myCurrentAction.setText(stableDir.getAbsolutePath()); 
      
      // Look up STABLE FILES ==================================================
      File[] stableFiles = null;
		try
		{	
			stableFiles = stableDir.listFiles();
		}
		catch(Exception ex)
		{
	      this.myLog.logLine(FAILED + "Could not LOOK UP files in STABLE directory: " + stableDir.getAbsolutePath());
	      return;
		}   	
		
		if (stableFiles == null)
		{
	      this.myLog.logLine(FAILED + "Could not LOOK UP files in STABLE directory: " + stableDir.getAbsolutePath());   	
	      return;			
		}
		
      // Look up DEDUP FILES ===================================================
      File[] targetDedupFiles = null;
		try
		{	
			targetDedupFiles = dedupTargeDir.listFiles();
		}
		catch(Exception ex)
		{
	      this.myLog.logLine(FAILED + "Could not LOOK UP files in TARGET directory: " + dedupTargeDir.getAbsolutePath());
	      return;
		}   	
		
		if (targetDedupFiles == null)
		{
	      this.myLog.logLine(FAILED + "Could not LOOK UP files in TARGET directory: " + dedupTargeDir.getAbsolutePath());   	
	      return;			
		}
		
		String[] targetDedupFileNames = new String[targetDedupFiles.length];
	  	for(int i=0; i<targetDedupFileNames.length; i++)
   	{
	  		targetDedupFileNames[i] = targetDedupFiles[i].getName();
   	}
 		
      // Do the comparison =====================================================		
   	for(int stableIndex=0; stableIndex<stableFiles.length; stableIndex++)
   	{
      	if (this.iStop)
      	{
		      this.myLog.logLine(ABORTED);  
      		return;
      	}
      	
      	for(int targetIndex=0; targetIndex<targetDedupFileNames.length; targetIndex++)
      	{
      		if (isNameSame(stableFiles[stableIndex].getName(), targetDedupFileNames[targetIndex] ))
      		{
      	  		if (stableFiles[stableIndex].isDirectory() && targetDedupFiles[targetIndex].isDirectory())
         		{
         			try
         			{	
         				processDirectory(stableFiles[stableIndex], targetDedupFiles[targetIndex]);
         			}
         			catch(Exception ex)
         			{
         		      this.myLog.logLine(FAILED + "Could not PROCESS directory: " + stableFiles[stableIndex] + "    BECAUSE: " + ex.getMessage());  
         		      ex.printStackTrace();
         			}  
         			
         			// Remove directory if it is now empty
         			if (targetDedupFiles[targetIndex].list().length == 0)
         			{
         				targetDedupFiles[targetIndex].delete();
         		      this.myLog.logLine("REMOVED EMPTY DIRECTORY: " + targetDedupFiles[targetIndex]);  
         			}
         		}
         		else if (stableFiles[stableIndex].isFile() && targetDedupFiles[targetIndex].isFile())
         		{
         			if (this.iCheckFileContents)
         			{
	         			try
	         			{
	         				if (stableFiles[stableIndex].length() == targetDedupFiles[targetIndex].length())
	         				{
	         					if (duplicateFiles(stableFiles[stableIndex], targetDedupFiles[targetIndex]))
	         					{
	         						targetDedupFiles[targetIndex].delete();
	         						this.myLog.logLine("REMOVED - identical file CONTENTS: " + targetDedupFiles[targetIndex]); 
	         					}
	         					else
	         					{
	         						this.myLog.logLine(SKIPPED + targetDedupFiles[targetIndex]);          						
	         					}
	         				}
	         			}
	         			catch(Exception ex)
	         			{
	         		      this.myLog.logLine(FAILED + targetDedupFiles[targetIndex] + "    BECAUSE: " + ex.getMessage());   				
	         			}
         			}
         			else
         			{
   						targetDedupFiles[targetIndex].delete();
   						this.myLog.logLine("REMOVED - identical file NAME: " + targetDedupFiles[targetIndex]);          				
         			}
         		}
      		}
      	}      	
    	}
   }
   
   boolean isNameSame(String a, String b)
   {
   	if (this.iAmRunningOnWindows)
   	{
   		return a.equalsIgnoreCase(b);
   	}
		return a.equals(b);
   }
   
   boolean duplicateFiles(File a, File b) throws Exception
   {
   	int aRead = 0;
   	char[] aChar = new char[1000];
   	FileReader aReader = new FileReader(a);
   	
   	int bRead = 0;
   	char[] bChar = new char[aChar.length];
   	FileReader bReader = new FileReader(b);
   	
   	try
   	{
	      aRead = aReader.read(aChar, 0, aChar.length);
	      while(aRead > -1)
	      {
		      bRead = bReader.read(bChar, 0, bChar.length);
		      if(aRead != bRead)
		      {
		      	return false;     	
		      }
	      	
		      for (int i = 0; i < aRead; i++)
		      {
		         if (aChar[i] != bChar[i])
		         {
		            return false;
		         }
		      }
		 	      	
	      	// Set up for next loop.
	         aRead = aReader.read(aChar, 0, aChar.length);      	
	      }
   	}
   	finally
   	{
   		aReader.close();
   		bReader.close();
   	}
   	return true;
   }
}