/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
package org.gerhardb.jibs.diskUsage;

import java.awt.*;
import java.awt.event.*;
import java.text.DecimalFormat;
import java.util.ArrayList;

import javax.swing.JComponent;

import org.gerhardb.lib.dirtree.DirectoryTreeNode;
import org.gerhardb.lib.dirtree.DirectoryTreeNode.TreeTotals;
import org.gerhardb.lib.util.Conversions;

/**
 * See Tile for ideas.
 * @author Gerhard Beck
 *
 */
class BoxPanel extends JComponent implements MouseListener
{
	static final BasicStroke BASIC_STROKE = new BasicStroke(2.0f);
	static final DecimalFormat FORMAT = new DecimalFormat("#,##0"); //$NON-NLS-1$
	static final int COLUMN_COUNT = 5;
	
	DirectoryTreeNode myRootNode; // Left most node going from top to bottom.
	DirectoryTreeNode myShowingNode;  // Current user selection painted green.
	ExplorerBox myExplorerBox;
	
	// Right hand boundary of each column.
	int[] myColumnBoundaries = new int[COLUMN_COUNT];
	//ArrayList<NodeYlocation>[] myColumns = new ArrayList[COLUMN_COUNT];
	ArrayList<ArrayList<NodeYlocation>> myColumns = new ArrayList<ArrayList<NodeYlocation>>(COLUMN_COUNT);	

	boolean iAmShowingRootNode = true;
			
	/**
	 * 
	 */
	public BoxPanel(ExplorerBox box)
	{
		this.myExplorerBox = box;
		addMouseListener(this);
		super.setOpaque(true);
	}

	// ========================================================================
	// Root Node
	// ========================================================================
	
	void setRootNode(DirectoryTreeNode rootNode)
	{
		//System.out.println("++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++");
		//System.out.println("rootNode set to: " + rootNode);
		//System.out.println("++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++");
		this.myRootNode = rootNode;
		this.myShowingNode = this.myRootNode;
		if (this.myRootNode.getParent() == null)
		{
			this.iAmShowingRootNode = true;
		}
		else
		{
			this.iAmShowingRootNode = false;
		}
		repaint();			
	}
	
	// ==========================================================================
	// Overridden Methods
	// ==========================================================================

	@Override
	protected void paintComponent(Graphics g)
	{
		Graphics2D g2 = (Graphics2D) g;
		
		g.setColor(Color.LIGHT_GRAY);
		g.fillRect(0, 0, getWidth(), getHeight());
		
		if (this.myRootNode == null)
		{
			return;
		}
		
		// Rest the column pointers
		this.myColumns.clear();
		int colWidth = getWidth() / COLUMN_COUNT;
		for(int i=0; i<this.myColumnBoundaries.length; i++)
		{
			this.myColumnBoundaries[i] = (i + 1) * colWidth;
			this.myColumns.add(new ArrayList<NodeYlocation>(20));
		}
				
		g2.setStroke(BASIC_STROKE);
		
		// Draw the first level
		if (this.iAmShowingRootNode && this.myExplorerBox.myTotalDiskSpace > 0 && this.myExplorerBox.iShowFreeSpace)
		{
			TreeTotals tt = this.myRootNode.getTreeTotals();
			long freeDiskSpace = this.myExplorerBox.myTotalDiskSpace - tt.aggregateTotalFileSize;

			int height = getHeight();
			int nodeHeight = (int)(height * tt.aggregateTotalFileSize/this.myExplorerBox.myTotalDiskSpace);
			int freeHeight = height - nodeHeight;
			
			// First draw used space.
			Rectangle usedRectangle = new Rectangle();
			usedRectangle.setBounds(0, 0, colWidth, nodeHeight);					
			NodeYlocation usedLocation = new NodeYlocation(this.myRootNode, tt, 0, nodeHeight);			
			this.myColumns.get(0).add(usedLocation);	// List of boxes in the column.			
			drawADirectory(g2, usedLocation, usedRectangle);
			
			// First draw used space.
			Rectangle freeRectangle = new Rectangle();
			freeRectangle.setBounds(0, nodeHeight, colWidth, freeHeight);			
			//NodeYlocation(DirectoryTreeNode node, TreeTotals tt, int topY, int bottomY)
			NodeYlocation freeLocation = new NodeYlocation(null, tt, nodeHeight, height);			
			this.myColumns.get(0).add(freeLocation);	// List of boxes in the column.		
			String freeSpace = Conversions.formattedBinaryBytes(freeDiskSpace);
			drawABox(g2, freeRectangle, Color.GRAY, "Free Space", freeSpace);
			
			// Now draw the children
			usedRectangle.x = usedRectangle.x + usedRectangle.width;
			drawChildren(g2, 1, usedLocation, usedRectangle);			
		}
		else
		{
			Rectangle rect = new Rectangle();
			rect.setBounds(0, 0, colWidth, getHeight());
			
			// Draw the first level
			TreeTotals tt = this.myRootNode.getTreeTotals();			
			NodeYlocation ny = new NodeYlocation(this.myRootNode, tt, 0, getHeight());
			this.myColumns.get(0).add(ny);				
			drawADirectory(g2, ny, rect);
			
			rect.x = rect.x + rect.width;

			drawChildren(g2, 1, ny, rect);
		}
		
		g.dispose();
	}
	
	private void drawChildren(Graphics2D g2, int column, NodeYlocation parentNY, Rectangle parentRect)
	{
		// Don't draw children of free space.
		if (parentNY.myNode == null)
		{
			return;
		}
		float boxHeight = parentRect.height;
		int childY = parentRect.y;
		int kids = parentNY.myNode.getChildCount();
		for(int i=0; i<kids; i++)
		{
			DirectoryTreeNode childNode = (DirectoryTreeNode)parentNY.myNode.getChildAt(i);
			//System.out.println("Drawing: " + childNode);
			TreeTotals childTotals = childNode.getTreeTotals();
			Rectangle childRect = new Rectangle(parentRect);
			childRect.y = childY;
			childRect.height = (int)(boxHeight * childTotals.aggregateTotalFileSize / parentNY.myTT.aggregateTotalFileSize);
			NodeYlocation childNY = new NodeYlocation(childNode, childTotals, childRect.y, childRect.y + childRect.height);
			this.myColumns.get(column).add(childNY);
			drawADirectory(g2, childNY, new Rectangle(childRect));		
			int nextColumn = column + 1;
			if (nextColumn < COLUMN_COUNT)
			{
				childRect.x = childRect.x + childRect.width;
				drawChildren(g2, nextColumn, childNY, childRect);
			}
			childY = childY + childRect.height;
		}
	}
	
	private void drawADirectory(Graphics2D g2, NodeYlocation ny, Rectangle rect)
	{
		String line_1 = null;			
		String line_2 = null;			
		Color color = Color.YELLOW;
		if (ny.myNode.equals(this.myShowingNode))
		{
			color = Color.GREEN;			
		}
		if ( ny.myNode != null)
		{
			line_1 = ny.myNode.getName();			
			line_2 = Conversions.formattedBinaryBytes(ny.myTT.aggregateTotalFileSize);			
		}
		
		drawABox(g2, rect, color, line_1, line_2);
	}
	
	private void drawABox(Graphics2D g2, Rectangle rect,
			Color color, String line_1, String line_2)
	{
		//System.out.println("Box: x=" + rect.x + " y=" + rect.y + " w=" + rect.width + " h=" + rect.height);
		g2.setColor(color);		
		g2.fillRect(rect.x, rect.y, rect.width, rect.height);		
		
		g2.setPaint(Color.BLACK);
		g2.draw(rect);
		
		if ( line_1 != null && line_2 != null)
		{
			Dimension textSize_1 = setFont(g2, line_1, rect.width);
			Dimension textSize_2 = setFont(g2, line_2, rect.width);
			
			// Try to print at least the directory name.
			final float HEIGHT_TEST = .9f;
			if ((textSize_1.height) > (rect.height * HEIGHT_TEST) || textSize_1.getWidth() <= 5)
			{
				return;
			}
			
			// Text is draw up from the Ascent and down from the Descent.
			//if (false)
			if (textSize_1.getWidth() > 5 && (textSize_1.height + textSize_2.height) < (rect.height * HEIGHT_TEST))
			{
				// Print both
				
				// Center the text.
				int textX = rect.x + ((rect.width - textSize_1.width ) / 2);
				int textY = rect.y + ((rect.height - textSize_1.height -  textSize_2.height) / 2);
				FontMetrics fm = g2.getFontMetrics();
				textY = textY + fm.getAscent();
				g2.drawString(line_1, textX, textY);				
				
				textX = rect.x + ((rect.width - textSize_2.width ) / 2);
				textY = textY + textSize_2.height;
				g2.drawString(line_2, textX, textY);
			}
			else
			{
				// Print directory only
				// Center the text.
				int textX = rect.x + ((rect.width - textSize_1.width ) / 2);
				int textY = rect.y + ((rect.height - textSize_1.height ) / 2);
				// Text is draw up from the Ascent and down from the Descent.
				FontMetrics fm = g2.getFontMetrics();
				textY = textY + fm.getAscent();
				g2.drawString(line_1, textX, textY);		
			}
		}
	}
	
	//g2.drawLine(0, super.getHeight(), super.getWidth(), super.getHeight());		
		// Draw the text just under the image now that we know the
		// final size of the image.
		// Text is draw up from the Ascent and down from the Descent.
		//FontMetrics fm = g.getFontMetrics();
		//int textY = (topOfText + fm.getMaxAscent());
		//g.drawString(msg, textX, textY);

	/**
	 * Get fond to fit in box size.
	 * http://www.particle.kth.se/~lindsey/JavaCourse/Book/Part1/Tech/Chapter06/plotDemo.html#PlotPanel
	 * Return height
	 */
	private static Dimension setFont(Graphics g, String msg, int box_width)
	{
		int type_size = 12;
		int type_size_min = 4;
		int msg_width;
		FontMetrics fm;

		do
		{
			Font f = new Font("Default", Font.PLAIN, type_size); //$NON-NLS-1$
			
			// Create the font and pass it to the Graphics context
			// g.setFont(new Font("Monospaced", Font.PLAIN, type_size));
			g.setFont(f);

			// Get measures needed to center the message
			fm = g.getFontMetrics();

			// How many pixels wide is the string
			msg_width = fm.stringWidth(msg);

			// See if the text will fit
			if (msg_width < box_width)
			{
				// { x = x_box + (box_width / 2) - (msg_width / 2);}
				break;
			}

			// Try smaller type
			type_size -= 2;
		}
		while (type_size >= type_size_min);

		// Don't display the numbers if they did not fit
		if (type_size < type_size_min) { return new Dimension(0, 0); }

		return new Dimension(msg_width, fm.getHeight());
	}
	
	private void showRootInfo(NodeYlocation node)
	{
		// Set all the defaults
		this.myExplorerBox.myDirName.setText(" "); //$NON-NLS-1$
		
		this.myExplorerBox.myTotalDiretoryCount.setText(""); //$NON-NLS-1$
		this.myExplorerBox.myTotalAllFileCount.setText(""); //$NON-NLS-1$
		this.myExplorerBox.myTotalAllFileSize.setText("");		 //$NON-NLS-1$
		this.myExplorerBox.myTotalImageFileCount.setText(""); //$NON-NLS-1$
		this.myExplorerBox.myTotalImageFileSize.setText(""); //$NON-NLS-1$
		
		this.myExplorerBox.myDiretoryCount.setText("");			 //$NON-NLS-1$
		this.myExplorerBox.myNodeFileCount.setText(""); //$NON-NLS-1$
		this.myExplorerBox.myNodeFileSize.setText("");	 //$NON-NLS-1$
		this.myExplorerBox.myNodeImageFileCount.setText(""); //$NON-NLS-1$
		this.myExplorerBox.myNodeImageFileSize.setText("");	 //$NON-NLS-1$
		
		this.myExplorerBox.mySubDiretoryCount.setText(""); //$NON-NLS-1$
		this.myExplorerBox.mySubDirFileCount.setText(""); //$NON-NLS-1$
		this.myExplorerBox.mySubDirFileSize.setText("");	 //$NON-NLS-1$
		this.myExplorerBox.mySubDirImageFileCount.setText(""); //$NON-NLS-1$
		this.myExplorerBox.mySubDirImageFileSize.setText("");	 //$NON-NLS-1$
		
		if (node == null || node.myNode == null)
		{
			this.myShowingNode = null;
		}
		else
		{
			this.myShowingNode = node.myNode;
			//DirectoryTreeNode.TreeTotals allTotals = myRootNode.getTreeTotals();
			DirectoryTreeNode.TreeTotals subTreeTotals = this.myShowingNode.getTreeTotals();
	
			this.myExplorerBox.myDirName.setText(this.myShowingNode.getAbsolutePath());

			// Totals for ALL directories in the tree
			this.myExplorerBox.myTotalDiretoryCount.setText(FORMAT.format(subTreeTotals.aggregateDirectoryCount));
			this.myExplorerBox.myTotalAllFileCount.setText(FORMAT.format(subTreeTotals.aggregateTotalFileCount));	
			this.myExplorerBox.myTotalAllFileSize.setText(Conversions.formattedBinaryBytes(subTreeTotals.aggregateTotalFileSize));		
			this.myExplorerBox.myTotalImageFileCount.setText(FORMAT.format(subTreeTotals.aggregateTargetFileCount));	
			this.myExplorerBox.myTotalImageFileSize.setText(Conversions.formattedBinaryBytes(subTreeTotals.aggregateTargetFileSize));

			// Totals for the SELECTED directory
			this.myExplorerBox.myDiretoryCount.setText(FORMAT.format(this.myShowingNode.getDirCountForTree()));
			this.myExplorerBox.myNodeFileCount.setText(FORMAT.format(this.myShowingNode.getTotalFileCount()));	
			this.myExplorerBox.myNodeFileSize.setText(Conversions.formattedBinaryBytes(this.myShowingNode.getTotalFileSize()));		
			this.myExplorerBox.myNodeImageFileCount.setText(FORMAT.format(this.myShowingNode.getTargetFileCount()));	
			this.myExplorerBox.myNodeImageFileSize.setText(Conversions.formattedBinaryBytes(this.myShowingNode.getTargetFileSize()));		
			
			// Totals for the SUBDIRECTORIES of the selected directory
			this.myExplorerBox.mySubDiretoryCount.setText(FORMAT.format(subTreeTotals.aggregateDirectoryCount - this.myShowingNode.getDirCountForTree()));
			this.myExplorerBox.mySubDirFileCount.setText(FORMAT.format(subTreeTotals.aggregateTotalFileCount - this.myShowingNode.getTotalFileCount()));	
			this.myExplorerBox.mySubDirFileSize.setText(Conversions.formattedBinaryBytes(subTreeTotals.aggregateTotalFileSize - this.myShowingNode.getTotalFileSize()));		
			this.myExplorerBox.mySubDirImageFileCount.setText(FORMAT.format(subTreeTotals.aggregateTargetFileCount - this.myShowingNode.getTargetFileCount()));	
			this.myExplorerBox.mySubDirImageFileSize.setText(Conversions.formattedBinaryBytes(subTreeTotals.aggregateTargetFileSize - this.myShowingNode.getTargetFileSize()));		
		}
	}

	// ==========================================================================
	// MouseListener
	// ==========================================================================
	public void mouseClicked(MouseEvent event)
	{
		int x = event.getX();
		
		// First, figure out what column was clicked.
		int column = -1;
		for(int i=0; i<this.myColumnBoundaries.length; i++)
		{
			//System.out.println("x: " + x + "      " + myColumnBoundaries[i]);
			if ( x < this.myColumnBoundaries[i])
			{
				column = i;
				break;
			}
		}
		
		int y = event.getY();
		NodeYlocation node = null;
		ArrayList<NodeYlocation> columnList = this.myColumns.get(column);
		int size = columnList.size();
		for(int i=0; i<size; i++)
		{
			NodeYlocation ny = columnList.get(i);
			//System.out.println("column# " + column + ": " + ny.myTopY + "      " + y + " - " + ny.myBottomY);
			if ( ny.myTopY < y && y < ny.myBottomY)
			{
				node = ny;
				//System.out.println("Box Panel FOUND NODE: " + node.myNode);
				break;
			}
		}
		
      if (node == null || node.myNode == null)
      {
      	return;
      }
      
		showRootInfo(node);
		
		boolean showNewRoot = false;
      if (event.getClickCount() == 2)
      {
      	showNewRoot = true;
      }
      else if ((event.getModifiers() & InputEvent.BUTTON2_MASK) != 0)
      {
      	showNewRoot = true;
      }
      else if ((event.getModifiers() & InputEvent.BUTTON3_MASK) != 0)
      {
      	showNewRoot = true;
      }

      if (showNewRoot)
      {
   		if (node.myNode != this.myRootNode)
   		{
   	  		setRootNode(node.myNode);
   		}
   		else if (node.myNode.getParent() != null)
   		{
   	  		setRootNode((DirectoryTreeNode)node.myNode.getParent());   			
   		}
     		else
   		{
      		this.getToolkit().beep();    			
   		}
       }
		
		BoxPanel.this.repaint();
	}

	public void mouseEntered(MouseEvent e)
	{
		// We don't care
	}

	public void mouseExited(MouseEvent e)
	{
		// We don't care
	}

	public void mousePressed(MouseEvent e)
	{
		// We don't care
	}

	public void mouseReleased(MouseEvent e)
	{
		// We don't care
	}
	
	// ==========================================================================
	// Inner classes
	// ==========================================================================
	class NodeYlocation
	{
		DirectoryTreeNode myNode;
		TreeTotals myTT;
		int myTopY;
		int myBottomY;
		
		NodeYlocation(DirectoryTreeNode node, TreeTotals tt, int topY, int bottomY)
		{
			this.myNode = node;
			this.myTT = tt;
			this.myTopY = topY;
			this.myBottomY = bottomY;
		}		
	}
}