/*
 * Copyright (c) 2007 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.gerhardb.jibs.diskUsage;

import javax.swing.*;

import org.gerhardb.lib.awt.GraphicsUtil;
import org.gerhardb.lib.dirtree.DTNReaderWriter;
import org.gerhardb.lib.dirtree.rdp.PathManager;
import org.gerhardb.lib.scroller.IScroll;
import org.gerhardb.lib.util.Conversions;

import java.awt.*;
import java.awt.event.*;
import java.io.*;

class DiskBar extends JComponent
{
	static final int PREFERRED_WIDTH = 800;
	
	File myRoot;
	long myFree;
	long myUsable;
	long myTotal;
	IScroll myScroller;
	int myWidth;
	
	DiskBar(File root, IScroll scroller)
	{
		this.myRoot = root;
		this.myScroller = scroller;
		super.setMinimumSize(new Dimension(PREFERRED_WIDTH,20));
		super.setPreferredSize(new Dimension(PREFERRED_WIDTH,20));
		updateUsage();
		super.addMouseListener(new Mouse());
	}

	void updateUsage()
	{
		this.myFree = this.myRoot.getFreeSpace();
		this.myUsable = this.myRoot.getUsableSpace();
		this.myTotal = this.myRoot.getTotalSpace();
		/*
		if (false)
		{
			myTotal = 100;
			myFree = 40;
			myUsable = 1;
		}
		*/
	}

	void adjustRelativeSize(long maxDiskSize, int maxScreenWidth)
	{
		this.myWidth = (int)((float)maxScreenWidth * (float)this.myTotal/maxDiskSize);
	}
	
	@Override
	public Dimension getPreferredSize()
	{
		return new Dimension(this.myWidth,20);
	}
	
	@Override
	public Dimension getMinimumSize()
	{
		return getPreferredSize();
	}
	
	@Override
	public Dimension getMaximumSize()
	{
		return getPreferredSize();
	}
	
	// ==========================================================================
	// Overridden Methods
	// ==========================================================================

	@Override
	protected void paintComponent(Graphics g)
	{
		//System.out.println("Root: " + getDiskName());
		//System.out.println("Total: " + myTotal);
		//System.out.println("Usable: " + myUsable);
		//System.out.println("Free: " + myFree);
		
		// Fast exit if the disk is not mounted.
		if (this.myTotal == 0)
		{
			g.setColor(Color.BLUE);
			g.fillRect(0, 0, getWidth(), getHeight());
			return;
		}
		
		Graphics2D g2 = (Graphics2D) g;
		
		// Compute percentages.
		int freePercent = (int)((float)this.myFree / (float)this.myTotal * 100f);
		int usedPercent = 100 - freePercent;
		int usablePercent = (int)((float)this.myUsable / (float)this.myTotal * 100f);
		// We only care about difference between free and usable.
		// Free will always be equal or bigger.	
		// If usable and free are the same, report all as usable.
		freePercent = freePercent - usablePercent;		
		
		String msgUsed = Integer.toString(usedPercent) + "%"; //$NON-NLS-1$
		String msgFree = Integer.toString(freePercent) + "%"; //$NON-NLS-1$
		String msgUsable = Integer.toString(usablePercent) + "%"; //$NON-NLS-1$
		
		int unusedPercent = 100 - freePercent - usablePercent;
		int usedWidth = (int)((float)getWidth() * (float)unusedPercent / 100f);
		int freeWidth = (int)((float)getWidth() * (float)freePercent / 100f);
		int usableWidth = (int)((float)getWidth() * (float)usablePercent / 100f);
		
		//System.out.println("used ratio: " + msgUsed + " / " + usedWidth);
		//System.out.println("free ratio: " + msgFree + " / " + freeWidth);
		//System.out.println("usable ratio: " + msgUsable + " / " + usableWidth);

		// ================= Print USED  =================
		// Draw in our entire space, even if isOpaque is false.
		// Default colors mainly for if there is no image.
		// Also fill if there is a problem with the images.
		g.setColor(Color.BLACK);
		g.fillRect(0, 0, getWidth(), getHeight());	
		printMsg(g2, msgUsed, Color.WHITE, 0, usedWidth, this.myTotal - this.myFree);
		
		// ================= Print FREE  =================
		g.setColor(Color.GRAY);
		g.fillRect(usedWidth, 0, freeWidth, getHeight());					
		printMsg(g2, msgFree, Color.BLACK, usedWidth, freeWidth, this.myFree);

		
		// ================= Print USABALE  =================
		//System.out.println("usablePercent: " + usablePercent);
		if (usablePercent > 10)
		{
			g.setColor(Color.GREEN);
			
		}
		else if (usablePercent > 5)
		{
			g.setColor(Color.YELLOW);			
		}
		else
		{
			g.setColor(Color.RED);			
		}
		g.fillRect(usedWidth + freeWidth, 0, usableWidth, getHeight());					
		printMsg(g2, msgUsable, Color.BLACK, usedWidth + freeWidth, usableWidth, this.myUsable);
		
		// ================= ALL DONE  =================
		g.dispose();
	}
	
	// ==========================================================================
	// Private Methods
	// ==========================================================================
	private void printMsg(Graphics2D g2, String msg, Color color, int x, int width, long bytes)
	{
		g2.setPaint(color);
		
		String fullMsg = msg + " " + Conversions.formattedBinaryBytes(bytes); //$NON-NLS-1$
		Dimension textSize = GraphicsUtil.setFont(g2, fullMsg, width);
		if (textSize.getWidth() > 0)
		{
			int textX = x + (int) ((width - textSize.getWidth()) / 2);
			int textY = (int) ((super.getHeight() - textSize.getHeight()) / 2);
			
			// Text is draw up from the Ascent and down from the Descent.
			FontMetrics fm = g2.getFontMetrics();
			textY = textY + fm.getAscent();
			
			g2.drawString(fullMsg, textX, textY);
			return;
		}		
		
		// Just try message.		
		textSize = GraphicsUtil.setFont(g2, msg, width);
		if (textSize.getWidth() > 0)
		{
			int textX = x + (int) ((width - textSize.getWidth()) / 2);
			int textY = (int) ((super.getHeight() - textSize.getHeight()) / 2);
			
			// Text is draw up from the Ascent and down from the Descent.
			FontMetrics fm = g2.getFontMetrics();
			textY = textY + fm.getAscent();
			
			g2.drawString(msg, textX, textY);
		}		
	}
	
	// ==========================================================================
	// Inner Classes
	// ==========================================================================
	class Mouse extends MouseAdapter
	{
		@Override
		public void mouseClicked(MouseEvent e)
		{
			//targetType, countTargetFiles, StartingJibsFile
			DTNReaderWriter dtn = new DTNReaderWriter("Disk", true, PathManager.getStartingJibsFile());
			// standAlone, label, IScroll, dtn
			boolean countHiddenDirectories = true;
			
			ExplorerBox box = new ExplorerBox(false, true, DiskBar.this.myScroller, dtn, countHiddenDirectories, true);	
			try
			{
				box.setRootDirectory(DiskBar.this.myRoot, true);
			}
			catch(Exception ex)
			{
				ex.printStackTrace();
			}
		}
	}
}
