/*
 * Copyright (c) 2007 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.gerhardb.jibs.diskUsage;

import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;

import javax.swing.*;
import org.gerhardb.jibs.Jibs;
import org.gerhardb.jibs.LicenseBox;
import org.gerhardb.lib.awt.SpringUtilities;
import org.gerhardb.lib.scroller.IScroll;
import org.gerhardb.lib.swing.SwingUtils;
import org.gerhardb.lib.util.Icons;
import org.gerhardb.lib.util.app.AboutBox;
import org.gerhardb.lib.util.startup.AppStarter;

import java.io.*;
import java.util.*;
import java.util.prefs.Preferences;

/**
 * A directory node designed to count its files and populate all of its
 * sub-directories.
 * <p>
 * The sub-directory population occurs in a depth-first fashion. Specifically,
 * for a given root, its children are identified and files counted. Then the
 * same recursively occurs for each sub-directory of the root.
 * <p>
 * This allows the highest level directory to be displayed immediately while
 * other directories are populating. If a user chooses an unpopulated
 * sub-directory, then that sub-directory is immediately populated, but its
 * children are not recursed until the normal time.
 * <p>
 * This is ALWAYS a directory and is tested on construction.
 */
public class JibsDiskUsage extends JFrame
{
   static final Preferences clsPrefs =
   	Preferences.userRoot().node("/org/gerhardb/jibs6/diskUsage/JibsDiskUsage"); //$NON-NLS-1$
   private static final String RELATIVE = "relative"; //$NON-NLS-1$
   
   boolean iExitOnClose;
	File[] myRoots;
	DiskBar[] myBars;
	IScroll myScroller;
	long myMaxDiskSize = 0;

	// ==========================================================================
	// Constructor
	// ==========================================================================
	public JibsDiskUsage(boolean exitOnClose, IScroll scroller)
	{
		super(Jibs.getString("JibsDiskUsage.2")); //$NON-NLS-1$
		this.myScroller = scroller;
		
		this.iExitOnClose = exitOnClose;
		if (this.iExitOnClose)
		{
			this.addWindowListener(new WindowAdapter()
			{
				@Override
				public void windowClosing(WindowEvent evt)
				{
					System.exit(0);
				}
			});
		}

		setupMenus();
		layoutPanel();
      this.setIconImage(Icons.getIcon(Icons.JIBS_16).getImage());

      SwingUtils.centerOnScreen(JibsDiskUsage.this);
   	JibsDiskUsage.this.setVisible(true);      
	}
	
	public static File[] listDrives()
	{
		String systemType = System.getProperty("os.name"); //$NON-NLS-1$
		try
		{
			if (systemType.equals("Linux")) //$NON-NLS-1$
			{
				return listLinuxRoots();
			}
		}
		catch(Exception ex)
		{
			ex.printStackTrace();
		}
		return File.listRoots();
	}

	private static File[] listLinuxRoots()
	throws Exception
	{
		ArrayList<File> hardDrivesList = new ArrayList<File>(20);
		//System.out.println("Listing Linux Roots");
		//System.out.println("===========================================");
		File fstab = new File("/etc/fstab"); //$NON-NLS-1$
		BufferedReader reader = new BufferedReader(new FileReader(fstab));
		
		String aLine = reader.readLine();
		while (aLine != null)
		{
			if (!aLine.startsWith("#")) //$NON-NLS-1$
			{
				//System.out.println(aLine);
				// Pattern is:
				// depth ^t file ^t fileCount ^t fileCountTree ^t dirCountTree
				StringTokenizer tokens = new StringTokenizer(aLine); //$NON-NLS-1$
				// Count is optional, so we only process lines with depth & file.
				// Skip empty lines
				if (tokens.countTokens() > 1)
				{
					tokens.nextToken(); // Skip first token
					String aDrive = tokens.nextToken();
					//System.out.println(aDrive);
					if (aDrive.startsWith("/") && !aDrive.startsWith("/proc")) //$NON-NLS-1$ //$NON-NLS-2$
					{
						hardDrivesList.add(new File(aDrive));
					}					
				}
			}
			aLine = reader.readLine();
		}
		reader.close();
		//System.out.println("===========================================");
		File[] rtnMe = new File[hardDrivesList.size()];
		rtnMe = hardDrivesList.toArray(rtnMe);
		Arrays.sort(rtnMe);
		return rtnMe;
	}

	void updateUsage()
	{
		for(int i=0; i<this.myBars.length; i++)
		{
			this.myBars[i].updateUsage();
		}
		repaint();
	}
	
  private void setupMenus()
	{
		class CloseAction extends AbstractAction
		{
			CloseAction()
			{
				if (JibsDiskUsage.this.iExitOnClose)
				{
					super.putValue(Action.NAME, Jibs.getString("JibsDiskUsage.9")); //$NON-NLS-1$
				}
				else
				{
					super.putValue(Action.NAME, Jibs.getString("JibsDiskUsage.10"));					 //$NON-NLS-1$
				}
			}
			
			public void actionPerformed(ActionEvent e)
			{
		      if (JibsDiskUsage.this.iExitOnClose)
		      {
		      	System.exit(0);
		      }
				else
				{
					JibsDiskUsage.this.dispose();					
				}
			}
		}
		
		class UpdateAction extends AbstractAction
		{			
			UpdateAction()
			{
				super.putValue(Action.NAME, Jibs.getString("JibsDiskUsage.11")); //$NON-NLS-1$
			}
			public void actionPerformed(ActionEvent e)
			{
				updateUsage();
			}
		}

		class RelativeAction extends AbstractAction
		{			
			RelativeAction()
			{
				super.putValue(Action.NAME, Jibs.getString("JibsDiskUsage.12")); //$NON-NLS-1$
			}
			public void actionPerformed(ActionEvent e)
			{
				clsPrefs.putBoolean(RELATIVE, true);
				try{clsPrefs.flush();}catch(Exception ex)
				{ex.printStackTrace();}
				
				layoutPanel();
			}
		}

		class ConstantAction extends AbstractAction
		{			
			ConstantAction()
			{
				super.putValue(Action.NAME, Jibs.getString("JibsDiskUsage.13")); //$NON-NLS-1$
			}
			public void actionPerformed(ActionEvent e)
			{
				clsPrefs.putBoolean(RELATIVE, false);
				try{clsPrefs.flush();}catch(Exception ex)
				{ex.printStackTrace();}
				
				layoutPanel();
			}
		}

		JMenu fileMenu = new JMenu(Jibs.getString("JibsDiskUsage.14")); //$NON-NLS-1$
		fileMenu.add(new JMenuItem(new UpdateAction()));
		fileMenu.add(new JSeparator());
		fileMenu.add(new JMenuItem(new CloseAction()));

		JRadioButton relBtn = new JRadioButton(new RelativeAction());
		JRadioButton constBtn = new JRadioButton(new ConstantAction());
		ButtonGroup group = new ButtonGroup();
		group.add(relBtn);
		group.add(constBtn);
		if (clsPrefs.getBoolean(RELATIVE, true))
		{
			relBtn.setSelected(true);
		}
		else
		{
			constBtn.setSelected(true);
		}
		
		JMenu viewMenu = new JMenu(Jibs.getString("JibsDiskUsage.15")); //$NON-NLS-1$
		viewMenu.add(relBtn);
		viewMenu.add(constBtn);
		
		JMenuBar menuBar = new JMenuBar();
		menuBar.add(fileMenu);
		menuBar.add(viewMenu);
		menuBar.add(getHelpMenu());
		this.setJMenuBar(menuBar);		
	}  

  void layoutPanel()
  {
	  // Look up info and set the bars before we layout anything.
		this.myRoots = listDrives();
		this.myBars = new DiskBar[this.myRoots.length];
		for (int i = 0; i < this.myRoots.length; i++)
		{
			this.myBars[i] = new DiskBar(this.myRoots[i], this.myScroller);
			if ( this.myBars[i].myTotal > this.myMaxDiskSize)
			{
				this.myMaxDiskSize = this.myBars[i].myTotal;
			}
		}
		
		for (int i = 0; i < this.myRoots.length; i++)
		{
			this.myBars[i].adjustRelativeSize(this.myMaxDiskSize, DiskBar.PREFERRED_WIDTH);
		}		
		
		// OK, now we can layout stuff...
		if (clsPrefs.getBoolean(RELATIVE, true))
		{
			layoutRelativePanel();
		}
		else
		{
			layoutConstantPanel();
		}
	}
  
   private void layoutConstantPanel()
	{
   	JPanel springPanel = new JPanel(new SpringLayout());
	   for (int i = 0; i < this.myRoots.length; i++) 
	   {
			 JLabel label = new JLabel(this.myRoots[i].getAbsolutePath());
			 springPanel.add(label);
			 label.setLabelFor(this.myBars[i]);
	       springPanel.add(this.myBars[i]);
	   }

	   //Lay out the panel.
	   SpringUtilities.makeCompactGrid(springPanel,
	   		this.myRoots.length, 2, //rows, cols
	                                   6, 6,        //initX, initY
	                                   6, 6);       //xPad, yPad  		
		this.getContentPane().removeAll();
		this.getContentPane().add(springPanel);		
	 	JibsDiskUsage.this.pack();
	}  
 	
	JMenu getHelpMenu()
	{
		JMenu	rtnMe = new JMenu(Jibs.getString("JibsDiskUsage.16")); //$NON-NLS-1$
		rtnMe.add(new JMenuItem(new HelpAction()));
		rtnMe.add(new JMenuItem(new AboutAction()));
		rtnMe.add(new JMenuItem(new LicenseAction()));
		return rtnMe;
	}

	class HelpAction extends AbstractAction
	{
		HelpAction()
		{
			super.putValue(Action.NAME, Jibs.getString("JibsDiskUsage.17")); //$NON-NLS-1$
		}
		

		public void actionPerformed(ActionEvent e)
		{
			JOptionPane.showMessageDialog(JibsDiskUsage.this, Jibs.getString("SortScreen.10"), //$NON-NLS-1$
					Jibs.getString("SortScreen.51"), JOptionPane.INFORMATION_MESSAGE); //$NON-NLS-1$
		}
	}

	class AboutAction extends AbstractAction
	{
		AboutAction()
		{
			super.putValue(Action.NAME, Jibs.getString("JibsDiskUsage.18")); //$NON-NLS-1$
		}
		
		public void actionPerformed(ActionEvent e)
		{
			new AboutBox(JibsDiskUsage.this);
		}
	}

	class LicenseAction extends AbstractAction
	{
		LicenseAction()
		{
			super.putValue(Action.NAME, Jibs.getString("JibsDiskUsage.19")); //$NON-NLS-1$
		}
		

		public void actionPerformed(ActionEvent e)
		{
			new LicenseBox(JibsDiskUsage.this);
		}
	}
	
	//	==========================================================================
	// Inner Classes // LayoutManager
	// ==========================================================================
	 private void layoutRelativePanel()
	{
	   JPanel mainPanel = new JPanel(new LabelLayout());
		for (int i = 0; i < this.myBars.length; i++)
		{
			JLabel aLabel = new JLabel(this.myRoots[i].getAbsolutePath());
			aLabel.setLabelFor(this.myBars[i]);
			mainPanel.add(aLabel);
			mainPanel.add(this.myBars[i]);
		}	
		
		this.getContentPane().removeAll();
		this.getContentPane().setLayout(new BorderLayout());
		this.getContentPane().add(mainPanel, BorderLayout.CENTER);
   	JibsDiskUsage.this.pack();
	}  
	   
	class LabelLayout implements LayoutManager
	{
		private int GAP = 10;
		int myMaxLabelWidth = 0;
		int myMaxItemWidth = 0;
		int myMaxLabelHeight = 0;
		int myMaxItemHeight = 0;
		int myRowHeight = 0;
		
		public void addLayoutComponent(String name, Component comp)
		{
			// We don't have any components with names so this is never called.
		}
	
		public void layoutContainer(Container parent)
		{
			Component[] components = parent.getComponents();
			calcMaximums(components, parent.getWidth());
			
			// Where to start drawing from.  
			int x = 0;
			int yRow = 0;

			for (int i = 0; i < components.length; i++)
			{
				if (components[i] instanceof JLabel)
				{
					JLabel aLabel = (JLabel)components[i];
					Dimension pref = aLabel.getPreferredSize();
					x = this.GAP + this.myMaxLabelWidth - pref.width;
					aLabel.setBounds(
							x, yRow, pref.width, this.myRowHeight);		
					//System.out.println("Label:  " + aLabel.getText()  + "  " + aLabel.getBounds());
					
					Component item = aLabel.getLabelFor();
					if (item != null)
					{
						pref = item.getPreferredSize();
						item.setBounds(this.myMaxLabelWidth + this.GAP + this.GAP, yRow, pref.width, this.myRowHeight);
						//System.out.println("Item:  " + item.getBounds());
					}
					yRow = yRow + this.myRowHeight + this.GAP;
				}
			}	
		}
	
		public Dimension minimumLayoutSize(Container parent)
		{
			//System.out.println("layout minimumLayoutSize"); //$NON-NLS-1$
			return new Dimension(100, 900);
		}
	
		public Dimension preferredLayoutSize(Container parent)
		{
			return calcMaximums(parent.getComponents(), 900);
		}
	
		public void removeLayoutComponent(Component comp)
		{
			// We don't care
		}
		
		private Dimension calcMaximums(Component[] components, int containerWidth)
		{
				int rowCount = 0;
			// Figure out maximums.
			for (int i = 0; i < components.length; i++)
			{
				if (components[i] instanceof JLabel)
				{
					JLabel aLabel = (JLabel)components[i];
					rowCount++;
					Dimension pref = aLabel.getPreferredSize();
					if (pref.width > this.myMaxLabelWidth)
					{
						this.myMaxLabelWidth = pref.width;
					}
					if (pref.height > this.myMaxLabelHeight)
					{
						this.myMaxLabelHeight = pref.height;
					}
					
					for (int k = 0; k < JibsDiskUsage.this.myRoots.length; k++)
					{
						int width = containerWidth - (this.GAP * 3) - this.myMaxLabelWidth;
						JibsDiskUsage.this.myBars[k].adjustRelativeSize(JibsDiskUsage.this.myMaxDiskSize, width);
					}		
					
					Component item = aLabel.getLabelFor();
					if (item != null)
					{
						pref = item.getPreferredSize();
						if (pref.width > this.myMaxItemWidth)
						{
							this.myMaxItemWidth = pref.width;
						}
						if (pref.height > this.myMaxItemHeight)
						{
							this.myMaxItemHeight = pref.height;
						}
					}
				}
			}	

			//System.out.println("maxLabelWidth: " + myMaxLabelWidth);
			//System.out.println("maxItemWidth: " + myMaxItemWidth);
			//System.out.println("maxLabelHeight: " + myMaxLabelHeight);
			//System.out.println("maxItemHeight: " + myMaxItemHeight);
			
			this.myRowHeight = this.myMaxItemHeight;
			if (this.myMaxLabelHeight > this.myMaxItemHeight)
			{
				this.myRowHeight = this.myMaxLabelHeight;
			}
			return new Dimension(
					this.GAP + this.myMaxLabelWidth + this.GAP + this.myMaxItemWidth + this.GAP, 
					(this.myRowHeight + this.GAP) * rowCount);
		}	
	}

	// ==========================================================================
	// TEST
	// ==========================================================================
	public static void main(String[] args)
	{
		AppStarter.startUpApp(args, "org.gerhardb.jibs.Jibs", true);
		new JibsDiskUsage(true, new org.gerhardb.lib.scroller.Scroller());
	}
	
	/*
	private static void aTest()
	{
		javax.swing.filechooser.FileSystemView fsv =  javax.swing.filechooser.FileSystemView.getFileSystemView();
		System.out.println("7777777777777777777777777777777777777777777777777777");
		File[] ezRoots = File.listRoots();
		for (int i=0; i<ezRoots.length; i++)
		{
			System.out.println("================================================");
			File aRoot = ezRoots[i];
			printInfo(fsv, aRoot);
			if (fsv.isTraversable(aRoot))
			{
				File[] kids = fsv.getFiles(aRoot, false);
				for (int j=0; j<kids.length; j++)
				{
					System.out.println(kids[j]);
					//printInfo(fsv, kids[j]);
				}
			}
		}
		
		System.out.println("&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&");
		System.out.println("getHomeDirectory: " + fsv.getHomeDirectory());
		System.out.println("getDefaultDirectory: " + fsv.getDefaultDirectory());
		
		File[] roots = fsv.getRoots();
		for (int i=0; i<roots.length; i++)
		{
			System.out.println("================================================");
			File aRoot = roots[i];
			printInfo(fsv, aRoot);
			if (fsv.isTraversable(aRoot))
			{
				File[] kids = fsv.getFiles(aRoot, false);
				System.out.println("------------------------------------------------");
				for (int j=0; j<kids.length; j++)
				{
					System.out.println(kids[j]);
					//printInfo(fsv, kids[j]);
				}
			}
		}
	}
	
	private static void printInfo(FileSystemView fsv, File aFile)
	{
		System.out.println(aFile);
		System.out.println("getSystemDisplayName: " + fsv.getSystemDisplayName(aFile));
		System.out.println("getSystemTypeDescription: " + fsv.getSystemTypeDescription(aFile));
		System.out.println("getSystemIcon: " + fsv.getSystemIcon(aFile));
		System.out.println("isComputerNode: " + fsv.isComputerNode(aFile));
		System.out.println("isDrive: " + fsv.isDrive(aFile));
		System.out.println("isFileSystem: " + fsv.isFileSystem(aFile));
		System.out.println("isFileSystemRoot: " + fsv.isFileSystemRoot(aFile));
		System.out.println("isFloppyDrive: " + fsv.isFloppyDrive(aFile));
		System.out.println("isHiddenFile: " + fsv.isHiddenFile(aFile));
		System.out.println("isRoot: " + fsv.isRoot(aFile));
		System.out.println("isTraversable: " + fsv.isTraversable(aFile));		
	}
	*/
}

