/*
 * Copyright (c) 2007 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.gerhardb.jibs.editor;

import java.awt.BorderLayout;
import java.awt.event.ActionEvent;
import java.io.File;
import java.util.prefs.Preferences;

import javax.imageio.ImageIO;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JPanel;

import org.gerhardb.jibs.Jibs;
import org.gerhardb.lib.io.FileSaveAs;
import org.gerhardb.lib.swing.JPanelRows;
import org.gerhardb.lib.swing.SwingUtils;
import org.gerhardb.lib.util.Icons;
import org.gerhardb.lib.util.startup.AppStarter;

/**
 * See Tile for ideas.
 * @author Gbeck
 *
 */
public class Editor extends JFrame
{
	private static final String PREF_PATH = "/org/gerhardb/jibs/lib/filetree/Editor"; //$NON-NLS-1$
	static final Preferences myPrefs = Preferences.userRoot().node(
			PREF_PATH);
	private static final String PRIOR_DIR = "priorDir"; //$NON-NLS-1$

	boolean iExitOnClose;
	PicturePanel myPicturePanel = new PicturePanel();
	JLabel myInfo = new JLabel(Jibs.getString("Editor.1")); //$NON-NLS-1$
	
	public Editor(boolean standAlone)
	{
		this.iExitOnClose = standAlone;
		if (this.iExitOnClose)
		{
			super.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		}

		if (this.iExitOnClose)
		{
			JMenu fileMenu = new JMenu(Jibs.getString("Editor.2")); //$NON-NLS-1$
			fileMenu.add(new JMenuItem(new OpenAction()));
			fileMenu.add(new JMenuItem(new SaveAction()));
			fileMenu.addSeparator();
			fileMenu.add(new JMenuItem(new CloseAction()));

			JMenuBar bar = new JMenuBar();
			bar.add(fileMenu);

			super.setJMenuBar(bar);
		}
		
		ColorPanel colorPanel = new ColorPanel();
		
		JPanel contentPanel = new JPanel(new BorderLayout());
		contentPanel.add(colorPanel, BorderLayout.WEST);
		contentPanel.add(this.myPicturePanel, BorderLayout.CENTER);
		contentPanel.add(infoPanel(), BorderLayout.SOUTH);

		super.getContentPane().add(contentPanel);

		this.setIconImage(Icons.getIcon(Icons.JIBS_16).getImage());
		int size = 700;
		super.setSize(size, size);
		SwingUtils.centerOnScreen(this);
		super.setVisible(true);
		
		this.myPicturePanel.setPictureFile(new File("d:/testpics/grindstone-PAINT.GIF")); //$NON-NLS-1$
		repaint();
	}
	
	JPanel infoPanel()
	{
		JPanelRows leftPanel = new JPanelRows();
		JPanel aRow = leftPanel.topRow();
		aRow.add(this.myInfo);
		
		JPanel rtnMe = new JPanel(new BorderLayout());
		rtnMe.add(leftPanel, BorderLayout.EAST);
		//rtnMe.add(rightPanel, BorderLayout.CENTER);
		return rtnMe;
	}

	// ===========================================================================
	// Inner Classes
	// ===========================================================================
	class OpenAction extends AbstractAction
	{
		OpenAction()
		{
			super.putValue(Action.NAME, Jibs.getString("Editor.4")); //$NON-NLS-1$
		}

		public void actionPerformed(ActionEvent e)
		{
			File picFile = null;
			try
			{
				String priorDir = myPrefs.get(PRIOR_DIR, null);

				if (priorDir == null)
				{
					priorDir = System.getProperty("user.home"); //$NON-NLS-1$
				}

				JFileChooser chooser = new JFileChooser();
				chooser.setDialogTitle(Jibs.getString("Editor.5")); //$NON-NLS-1$
				chooser.setFileSelectionMode(JFileChooser.FILES_AND_DIRECTORIES);
				chooser.setCurrentDirectory(new File(priorDir));

				int returnVal = chooser.showOpenDialog(Editor.this);
				if (returnVal == JFileChooser.APPROVE_OPTION)
				{
					File picked = chooser.getSelectedFile();
					if (picked.isDirectory())
					{
						myPrefs.put(PRIOR_DIR, picked.getAbsolutePath());
					}
					else
					{
						picFile = picked;
						myPrefs.put(PRIOR_DIR, picked.getParentFile()
								.getAbsolutePath());
					}
					myPrefs.flush();
				}
			}
			catch (Exception ex)
			{
				ex.printStackTrace();
			}
			Editor.this.myPicturePanel.setPictureFile(picFile);
			repaint();
		}
	}

	class SaveAction extends AbstractAction
	{
		SaveAction()
		{
			super.putValue(Action.NAME, Jibs.getString("Editor.6")); //$NON-NLS-1$
		}

		public void actionPerformed(ActionEvent e)
		{
			String priorDir = myPrefs.get(PRIOR_DIR, null);
			if (priorDir == null)
			{
				priorDir = System.getProperty("user.home"); //$NON-NLS-1$
			}
			
			JFileChooser chooser = new JFileChooser();
			chooser.setCurrentDirectory(new File(priorDir));
			chooser.setDialogTitle(Jibs.getString("Editor.7"));  //$NON-NLS-1$
			chooser.setFileSelectionMode(JFileChooser.FILES_AND_DIRECTORIES);
			int returnVal = chooser.showSaveDialog(Editor.this);
			if (returnVal == JFileChooser.APPROVE_OPTION)
			{
				try
				{
					File saveAsFile = chooser.getSelectedFile();
					if (FileSaveAs.PossibleDuplicateNameOK(saveAsFile, Editor.this))
					{
						ImageIO.write(Editor.this.myPicturePanel.myImage, "gif", saveAsFile); //$NON-NLS-1$
						System.out.println("Wrote GIF"); //$NON-NLS-1$
					}
				}
				catch (Exception ex)
				{
					// So remote as to ignore
					ex.printStackTrace();
				}
			}
		}
	}

	class CloseAction extends AbstractAction
	{
		CloseAction()
		{
			if (Editor.this.iExitOnClose)
			{
				super.putValue(Action.NAME, Jibs.getString("Editor.10")); //$NON-NLS-1$
			}
			else
			{
				super.putValue(Action.NAME, Jibs.getString("Editor.11")); //$NON-NLS-1$
			}
		}

		public void actionPerformed(ActionEvent e)
		{
			if (Editor.this.iExitOnClose)
			{
				System.exit(0);
			}
			else
			{
				Editor.this.dispose();
			}
		}
	}

	// ===========================================================================
	// Main
	// ===========================================================================
	public static void main(String[] args)
	{
		AppStarter.startUpApp(args, "org.gerhardb.jibs.Jibs", true);
		new Editor(true);
		//quickTest(); 
	}
	
	/*
	static void quickTest()
	{
		{
			BufferedImage img = new BufferedImage(50, 50,
					BufferedImage.TYPE_INT_ARGB);
			Graphics2D g2 = img.createGraphics();
			g2.setColor(Color.YELLOW);
			g2.fillRect(0, 0, 10, 10);
			g2.dispose();
			try
			{
				ImageIO.write(img, "png", new File( //$NON-NLS-1$
						"c:/gbroot/My Pictures/EZtest.png")); //$NON-NLS-1$
				ImageIO.write(img, "gif", new File( //$NON-NLS-1$
						"c:/gbroot/My Pictures/EZtest.gif")); //$NON-NLS-1$
			}
			catch (Exception ex)
			{
				ex.printStackTrace();
			}
		}

		{
			BufferedImage bi = new BufferedImage(50, 50,
					BufferedImage.TYPE_INT_ARGB);
			Graphics2D g2 = bi.createGraphics();
			Color transparentWhite = new Color(Color.WHITE.getRed(), Color.WHITE
					.getGreen(), Color.WHITE.getBlue(), 0);
			int tw = transparentWhite.getRGB();
			// Following code does not work correctly: I suspect fillRect does not set the RGB value if the
			// transparency is 0: while this might seem like a reasonable optimisation, it means that
			// it is not doing what it is supposed to do and one is left with incorrect color values
			// in the image so cannot do things like stripping the alpha channel to get the correct
			// non-alpha image.
			// g.setColor(transparentWhite);
			// g.fillRect(0, 0, maxX - minX, maxY - minY);
			for (int x = 0; x < bi.getWidth(); x++)
			{
				for (int y = 0; y < bi.getHeight(); y++)
				{
					bi.setRGB(x, y, tw);
				}
			}
			g2.setColor(Color.YELLOW);
			g2.fillRect(0, 0, 10, 10);
			g2.dispose();
			try
			{
				ImageIO.write(bi, "png", new File( //$NON-NLS-1$
						"c:/gbroot/My Pictures/EZtest2.png")); //$NON-NLS-1$
				ImageIO.write(bi, "gif", new File( //$NON-NLS-1$
						"c:/gbroot/My Pictures/EZtest2.gif")); //$NON-NLS-1$
			}
			catch (Exception ex)
			{
				ex.printStackTrace();
			}
		}

		try
		{
			BufferedImage img = ImageFactory.getImageFactory().getImageEZ(new File(
					"c:/gbroot/My Pictures/grindstone.GIF")); //$NON-NLS-1$
			Graphics2D g2 = img.createGraphics();
			Color transparentWhite = new Color(Color.WHITE.getRed(), Color.WHITE
					.getGreen(), Color.WHITE.getBlue(), 0);
			int tw = transparentWhite.getRGB();
			Color opaqueWhite = new Color(Color.WHITE.getRed(), Color.WHITE
					.getGreen(), Color.WHITE.getBlue(), 1);
			int ow = opaqueWhite.getRGB();
			// Following code does not work correctly: I suspect fillRect does not set the RGB value if the
			// transparency is 0: while this might seem like a reasonable optimisation, it means that
			// it is not doing what it is supposed to do and one is left with incorrect color values
			// in the image so cannot do things like stripping the alpha channel to get the correct
			// non-alpha image.
			// g.setColor(transparentWhite);
			// g.fillRect(0, 0, maxX - minX, maxY - minY);
			for (int x = 0; x < img.getWidth(); x++)
			{
				for (int y = 0; y < img.getHeight(); y++)
				{
					if (img.getRGB(x, y) == ow)
					{
						img.setRGB(x, y, tw);
					}
				}
			}

			// FORCED TEST
			for (int x = 0; x < img.getWidth() / 2; x++)
			{
				for (int y = 0; y < img.getHeight(); y++)
				{
					img.setRGB(x, y, tw);
				}
			}
			g2.dispose();
			try
			{
				ImageIO.write(img, "png", new File( //$NON-NLS-1$
						"c:/gbroot/My Pictures/grindstoneTEST.png")); //$NON-NLS-1$
				ImageIO.write(img, "gif", new File( //$NON-NLS-1$
						"c:/gbroot/My Pictures/grindstoneTEST.gif")); //$NON-NLS-1$
			}
			catch (Exception ex)
			{
				ex.printStackTrace();
			}
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
		}

		System.out.println("Files written");		 //$NON-NLS-1$
	}
	*/
}
