/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
package org.gerhardb.jibs.games.missingTile;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.EventQueue;
import java.awt.FlowLayout;
import java.awt.Image;
import java.awt.Toolkit;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;
import java.util.Random;
import java.util.prefs.Preferences;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.KeyStroke;

import org.gerhardb.jibs.Jibs;
import org.gerhardb.lib.image.IOImage;
import org.gerhardb.lib.image.ImageFactory;
import org.gerhardb.lib.swing.JPanelRows;
import org.gerhardb.lib.swing.SwingUtils;
import org.gerhardb.lib.util.ActionHelpers;
import org.gerhardb.lib.util.Icons;
import org.gerhardb.lib.util.startup.AppStarter;

/**
 * A puzzle game with one missing tile.
 * IF YOU EVER FIGURE OUT HOW TO BEAT THE GAME,
 * COMMENT IN THE FINISH BUTTON AND ADD THE LOGIC THERE.
 * 
 * ALSO, THE TEST BUTTON IS COMMENTE OUT.  IT ROTATES THE 
 * EMPTY SQUARE TO TEST VICTORY.
 */
public class MissingTile extends JFrame
{
   static final Preferences clsPrefs =
   	Preferences.userRoot().node("/org/gerhardb/jibs/games/MissingTile"); //$NON-NLS-1$
   static final String LAST_IMAGE = "lastImage"; //$NON-NLS-1$
   static final String LAST_DIRECTORY = "lastDirectory"; //$NON-NLS-1$
   private static final String TILE_COUNT = "tileCount"; //$NON-NLS-1$
   private final String MOVES = " " + Jibs.getString("MissingTile.0"); //$NON-NLS-1$ //$NON-NLS-2$
   
   int myTileCount = 3;
   TilePanel myTilePanel;
   int myTestCount = 0;
   JButton myPlayBtn = new JButton(Jibs.getString("MissingTile.1")); //$NON-NLS-1$
   JButton myFinishBtn = new JButton(Jibs.getString("MissingTile.2")); //$NON-NLS-1$
   JLabel myMovesMade = new JLabel("0" + this.MOVES); //$NON-NLS-1$
   boolean iExitOnClose;
   
   public MissingTile(boolean exitOnClose)
   {
      super(Jibs.getString("MissingTile.4")); //$NON-NLS-1$
      
      this.iExitOnClose = exitOnClose;
      if (this.iExitOnClose)
      {
         this.addWindowListener(
            new WindowAdapter()
         {
            @Override
				public void windowClosing(WindowEvent evt)
            {
               System.exit(0);
            }
         });
      }

      // Must be before layoutComponents().
      this.myTileCount = clsPrefs.getInt(TILE_COUNT, this.myTileCount);
		this.setIconImage(Icons.getIcon(Icons.JIBS_16).getImage());

      layoutComponents();

      
      EventQueue.invokeLater(new Runnable()
      {
         public void run()
         {
         	MissingTile.this.pack();
         	MissingTile.this.setVisible(true);
            SwingUtils.centerOnScreen(MissingTile.this);
            showLastPicture();
         }
      });
   }
   
   void showLastPicture()
   {
  		String imageName = clsPrefs.get(LAST_IMAGE, null);
		if (imageName != null)
		{
			try
			{
				IOImage ioImage = ImageFactory.getImageFactory().makeImage(new File(imageName));
				this.myTilePanel.setImage(ioImage.getImage());
			}
			catch(Exception ex)
			{
				//ex.printStackTrace();
			}
		}
   }

   private void layoutComponents()
	{
		// Set up application
		this.setSize(new Dimension(600, 600));

		class ImageAction extends AbstractAction
		{
			ImageAction()
			{
				// Reads "Select"
				super.putValue(Action.NAME, Jibs.getString("MissingTile.16")); //$NON-NLS-1$
				super.putValue(Action.MNEMONIC_KEY, ActionHelpers.getKeyCodeAsInteger("S")); //$NON-NLS-1$			
			}
			
			public void actionPerformed(ActionEvent e)
			{
				quit();

				JFileChooser chooser = new JFileChooser();
				chooser.setDialogTitle("Select Image to Show"); //$NON-NLS-1$
				chooser.setFileSelectionMode(JFileChooser.FILES_AND_DIRECTORIES);

				String dirName = clsPrefs.get(LAST_DIRECTORY, null);

				if (dirName != null)
				{
					chooser.setCurrentDirectory(new File(dirName));
				}

				int returnVal = chooser.showOpenDialog(MissingTile.this);
				if (returnVal == JFileChooser.APPROVE_OPTION)
				{
					try
					{
						String imageName = chooser.getSelectedFile().getCanonicalPath();

						if (imageName == null) { return; }

						IOImage ioImage = ImageFactory.getImageFactory().makeImage(new File(imageName));
						MissingTile.this.myTilePanel.setImage(ioImage.getImage());
						clsPrefs.put(LAST_IMAGE, imageName);
						clsPrefs.put(LAST_DIRECTORY, imageName);
					}
					catch (Exception ex)
					{
						JOptionPane.showMessageDialog(MissingTile.this, ex
								.getMessage(), Jibs.getString("MissingTile.17"), //$NON-NLS-1$
								JOptionPane.ERROR_MESSAGE);
					}
				}
				quit();
			}
		}
		
		class TestPatternAction extends AbstractAction
		{
			TestPatternAction()
			{
				// Test Pattern
				super.putValue(Action.NAME, Jibs.getString("MissingTile.18")); //$NON-NLS-1$
				super.putValue(Action.MNEMONIC_KEY, ActionHelpers.getKeyCodeAsInteger("T")); //$NON-NLS-1$
			}
			
			public void actionPerformed(ActionEvent e)
			{
				clsPrefs.remove(LAST_IMAGE);
				MissingTile.this.myTilePanel.setImage(null);
			}
		}
		
		class PasteAction extends AbstractAction
		{
			PasteAction()
			{
				super.putValue(Action.NAME, Jibs.getString("paste")); //$NON-NLS-1$
				KeyStroke ks = KeyStroke.getKeyStroke("control V");
				super.putValue(Action.ACCELERATOR_KEY, ks); //$NON-NLS-1$
				super.putValue(Action.MNEMONIC_KEY, ActionHelpers.getKeyCodeAsInteger("P")); //$NON-NLS-1$
			}
			
			public void actionPerformed(ActionEvent e)
			{
				Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
				Transferable transferable = clipboard.getContents(this);
				DataFlavor[] flavors = transferable.getTransferDataFlavors();
				for(int i=0; i<flavors.length; i++)
				{
					if (DataFlavor.imageFlavor.equals(flavors[i]))
					{
						try
						{
							Object obj = transferable.getTransferData(DataFlavor.imageFlavor);
							if (obj != null && obj instanceof Image)
							{
								MissingTile.this.myTilePanel.setImage((Image)obj);
							}
						}
						catch(Exception ex)
						{
							ex.printStackTrace();
						}
					}
				}
			}
		}
		
		class CloseAction extends AbstractAction
		{
			CloseAction()
			{
				if (MissingTile.this.iExitOnClose)
				{
					super.putValue(Action.NAME, Jibs.getString("MissingTile.19")); //$NON-NLS-1$
					super.putValue(Action.MNEMONIC_KEY, ActionHelpers.getKeyCodeAsInteger("X")); //$NON-NLS-1$
				}
				else
				{
					super.putValue(Action.NAME, Jibs.getString("MissingTile.20"));					 //$NON-NLS-1$
					super.putValue(Action.MNEMONIC_KEY, ActionHelpers.getKeyCodeAsInteger("C")); //$NON-NLS-1$
				}
			}
			
			public void actionPerformed(ActionEvent e)
			{
		      if (MissingTile.this.iExitOnClose)
		      {
		      	System.exit(0);
		      }
				else
				{
					MissingTile.this.dispose();					
				}
			}
		}
		
		class CountAction extends AbstractAction
		{
			int myCount;
			JRadioButton myButton;
			
			CountAction(int count)
			{
				this.myCount = count;
				super.putValue(Action.NAME, Integer.toString(this.myCount));
			}
			
			public void actionPerformed(ActionEvent e)
			{
				MissingTile.this.myTileCount = this.myCount; 
				clsPrefs.putInt(TILE_COUNT, this.myCount);
				this.myButton.setSelected(clsPrefs.getInt(TILE_COUNT, 0) == MissingTile.this.myTileCount);
				MissingTile.this.myTilePanel = new TilePanel(MissingTile.this, MissingTile.this.myTileCount);
				MissingTile.this.getContentPane().add(MissingTile.this.myTilePanel, BorderLayout.CENTER);
				MissingTile.this.showLastPicture();
				quit();	
			}
			
			void setButton(JRadioButton button)
			{
				this.myButton = button;
			}
		}
		
		JButton testBtn = new JButton("Test"); //$NON-NLS-1$
		testBtn.addActionListener(new java.awt.event.ActionListener()
		{
			public void actionPerformed(java.awt.event.ActionEvent evt)
			{
				quit();

				// Go ahead and increment now since tiles count from 1 not zero.
				MissingTile.this.myTestCount++;
				MissingTile.this.myTilePanel.emptySeed(MissingTile.this.myTestCount);
				if (MissingTile.this.myTestCount >= MissingTile.this.myTilePanel.myTiles.length)
				{
					MissingTile.this.myTestCount = 0;
				}
			}
		});

		JButton scrambleBtn = new JButton(Jibs.getString("MissingTile.6")); //$NON-NLS-1$
		scrambleBtn.addActionListener(new java.awt.event.ActionListener()
		{
			public void actionPerformed(java.awt.event.ActionEvent evt)
			{
				MissingTile.this.myTilePanel.scramble();
			}
		});

		this.myFinishBtn.addActionListener(new java.awt.event.ActionListener()
		{
			public void actionPerformed(java.awt.event.ActionEvent evt)
			{
				finish();
			}
		});

		this.myPlayBtn.addActionListener(new java.awt.event.ActionListener()
		{
			public void actionPerformed(java.awt.event.ActionEvent evt)
			{
				if (MissingTile.this.myTilePanel.isGameInProgress)
				{
					quit();
				}
				else
				{
					pickEmpty();
				}
			}
		});

		// "File"
		JMenu fileMenu = new JMenu(Jibs.getString("MissingTile.21")); //$NON-NLS-1$
		fileMenu.setMnemonic('F');
		fileMenu.add(new JMenuItem(new CloseAction()));

		// "Image"
		JMenu imageMenu = new JMenu(Jibs.getString("MissingTile.22")); //$NON-NLS-1$
		imageMenu.setMnemonic('I');
		imageMenu.add(new JMenuItem(new ImageAction()));
		imageMenu.add(new JMenuItem(new PasteAction()));
		imageMenu.add(new JMenuItem(new TestPatternAction()));
		
		// "Tile"
		JMenu tileMenu = new JMenu(Jibs.getString("MissingTile.23")); //$NON-NLS-1$
		tileMenu.setMnemonic('T');
		ButtonGroup tileCountGrp = new ButtonGroup();
		for(int i=3; i<9; i++)
		{
			CountAction a = new CountAction(i);
			JRadioButton button = new JRadioButton(a);
			switch(i)
			{
				case 3:
					button.setMnemonic(KeyEvent.VK_3);
					break;
				case 4:
					button.setMnemonic(KeyEvent.VK_4);
					break;
				case 5:
					button.setMnemonic(KeyEvent.VK_5);
					break;
				case 6:
					button.setMnemonic(KeyEvent.VK_6);
					break;
				case 7:
					button.setMnemonic(KeyEvent.VK_7);
					break;
				case 8:
					button.setMnemonic(KeyEvent.VK_8);
					break;
				case 9:
					button.setMnemonic(KeyEvent.VK_9);
					break;
			}
			a.setButton(button);
			tileMenu.add(button);
			tileCountGrp.add(button);
			button.setSelected(i == this.myTileCount);
		}
		
		JMenuBar menuBar = new JMenuBar();
		menuBar.add(fileMenu);
		menuBar.add(imageMenu);
		menuBar.add(tileMenu);
		this.setJMenuBar(menuBar);		
		
		JPanelRows bottomPanel = new JPanelRows(FlowLayout.CENTER);
		JPanel row = bottomPanel.topRow();
		row.add(this.myMovesMade);

		row = bottomPanel.nextRow();
		row.add(this.myPlayBtn);
		row.add(scrambleBtn);
		//row.add(myFinishBtn);
		//row.add(testBtn); //tests victory

		this.myTilePanel = new TilePanel(this, this.myTileCount);

		this.getContentPane().setLayout(new BorderLayout());
		//this.getContentPane().add(topPanel, BorderLayout.NORTH);
		this.getContentPane().add(this.myTilePanel, BorderLayout.CENTER);
		this.getContentPane().add(bottomPanel, BorderLayout.SOUTH);

		this.myFinishBtn.setEnabled(this.myTilePanel.isGameInProgress);
	}
  
   /**
    * This was supposed to have the logic to show the user how to finish...
    * 
    */
   void finish()
	{
 		quit();
	}
   
   void pickEmpty()
	{
   	int seed = 1;
   	Random random = new Random();
		int candidate = random.nextInt(this.myTilePanel.myTiles.length);
		// candidate of zero is not valid since tiles are numbered from 1.
		if ( candidate > 0 )
		{
			seed = candidate;
		}
  		this.myTilePanel.emptySeed(seed);
 		this.myPlayBtn.setText(Jibs.getString("MissingTile.7")); //$NON-NLS-1$
      this.myFinishBtn.setEnabled(this.myTilePanel.isGameInProgress);
 	}
   
   void quit()
	{
  		this.myPlayBtn.setText(Jibs.getString("MissingTile.8")); //$NON-NLS-1$
   	this.myTilePanel.resetGame();
   	this.myTilePanel.repaint();
      this.myFinishBtn.setEnabled(this.myTilePanel.isGameInProgress);
	}
   
   void victory()
   {
  		this.myPlayBtn.setText(Jibs.getString("MissingTile.9")); //$NON-NLS-1$
      this.myFinishBtn.setEnabled(this.myTilePanel.isGameInProgress);
   	this.myTilePanel.resetGame();
      this.myFinishBtn.setEnabled(this.myTilePanel.isGameInProgress);
      JOptionPane.showMessageDialog(
            this,
            Jibs.getString("MissingTile.10"), //$NON-NLS-1$
            Jibs.getString("MissingTile.11"), //$NON-NLS-1$
            JOptionPane.INFORMATION_MESSAGE );	
   }
   
   void updateMovesMade(int moves)
   {
   	this.myMovesMade.setText(Integer.toString(moves) + this.MOVES);	
   }
   	
//===========================================================================
//                              Main
//===========================================================================
   public static void main(String[] args)
   {
		AppStarter.startUpApp(args, "org.gerhardb.jibs.Jibs", true);
      new MissingTile(true);
   }
}
