/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
package org.gerhardb.jibs.optimizer;

import java.awt.DisplayMode;
import java.awt.GraphicsDevice;
import java.awt.GraphicsEnvironment;
import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;

import org.gerhardb.lib.io.FileUtil;

/**
 * Wraps the preferences used by the viewer and provides an way of editing
 * the preferences.
 * @author  Gerhard Beck
 */
public class OptimizerPreferences
{
   private static final String PREF_PATH =
      "/org/gerhardb/jibs/optimizer/OptimizerPreferences"; //$NON-NLS-1$

   private static Preferences clsPrefs;

   private static int clsFullScreenHeight = 600;
   private static int clsFullScreenWidth = 800;

   static
	{
		GraphicsEnvironment ge = GraphicsEnvironment
				.getLocalGraphicsEnvironment();
		if (ge.isHeadlessInstance()) { throw new RuntimeException(
				"Java is reporting that this instance " //$NON-NLS-1$
						+ "of your GraphicsEnvironment is headless.  A monitor " //$NON-NLS-1$
						+ " and keyboard are required to run JIBS."); } //$NON-NLS-1$

		try
		{
			GraphicsDevice gd = ge.getDefaultScreenDevice();
			DisplayMode dm = gd.getDisplayMode();
			clsFullScreenWidth = dm.getWidth();
			clsFullScreenHeight = dm.getHeight();
		}
		catch (RuntimeException ex)
		{
			ex.printStackTrace();
			System.out.println("------------------------------------------------"); //$NON-NLS-1$
			System.out.println(ge.getDefaultScreenDevice());
			System.out.println("------------------------------------------------");				 //$NON-NLS-1$
			/*
				int answer = JOptionPane
						.showConfirmDialog(
								null,
								"JIBS encountered an unexpected Java error.\n"
										+ "You can procede but in some cases "
										+ "full screen may not work propertly.\n "
										+ "Run JIBS from a command line for " 
										+ "technical details.\n"
										+ "Do you want to conintue?\n",
								"Load JIBS Problem", JOptionPane.YES_NO_OPTION);
				if (answer == JOptionPane.NO_OPTION) { throw ex; }
				*/
		}
	}
 
   //===========================================================================
   //                          Constructor
   //===========================================================================

   private OptimizerPreferences()
   {
   	// Don't allow public creation.
   }

   /**
    * @throws SecurityException
    */
   public static void init()
   {
      // Only do once.
      if (clsPrefs != null)
      {
         return;
      }
      clsPrefs = Preferences.userRoot().node(PREF_PATH);
   }

   //===========================================================================
   //                          General Information
   //                       Computed ONCE at startup.
   //                             NEVER saved!!
   //===========================================================================
   public static int getFullScreenHeight()
   {
      return clsFullScreenHeight;
   }

   public static int getFullScreenWidth()
   {
      return clsFullScreenWidth;
   }

   //===========================================================================
   //                          Log Directory
   //===========================================================================
   private static final String LOG_DIRECTORY = "logDirectory"; //$NON-NLS-1$
   private static final String LOG_NUMBER = "logNumber"; //$NON-NLS-1$

   /**
    * @return null if not correctly set.
    */
   public static String getLogDirectory()
   {
      String rtnMe = FileUtil.checkDirectory(clsPrefs.get(
      		LOG_DIRECTORY,
         System.getProperty("user.home"))); 
      if (rtnMe == null)
      {
      	rtnMe = System.getProperty("user.home");
      }
      return rtnMe;
   }

   public static void setLogDirectory(String dir)

   {
      String validated = FileUtil.safeValidateDirectory(dir);
      if (validated != null)
      {
         clsPrefs.put(LOG_DIRECTORY, validated);
      }
      else
      {
         clsPrefs.remove(LOG_DIRECTORY);
      }
   }

   public static int getLogNumber()
   {
      return clsPrefs.getInt(LOG_NUMBER, 0);
   }

   /*
   public static int getNextLogNumber()
   {
      int count = clsPrefs.getInt(LOG_NUMBER, 0);
      count++;
      try
      {
         setLogNumber(count);
      }
      catch (Exception ex)
      {
         ex.printStackTrace();
      }
      return count;
   }
*/
   
   public static void setLogNumber(int setting)
      throws Exception
   {
      clsPrefs.putInt(LOG_NUMBER, setting);
      clsPrefs.flush();
   }

   //===========================================================================
   //                          Dedup Options
   //===========================================================================
   private static final String DEDUP = "dedup"; //$NON-NLS-1$
   private static final String DIR_UTIL_ACROSS = "dirAcross"; //$NON-NLS-1$

   public static boolean getDedup()
   {
      return clsPrefs.getBoolean(DEDUP, false);
   }

   public static void setDedup(boolean setting)
   {
      clsPrefs.putBoolean(DEDUP, setting);
   }

   public static boolean getDirectoryAcross()
   {
      return clsPrefs.getBoolean(DIR_UTIL_ACROSS, false);
   }

   public static void setDirectoryAcross(boolean setting)
   {
      clsPrefs.putBoolean(DIR_UTIL_ACROSS, setting);
   }

   //===========================================================================
   //                          Rename Options
   //===========================================================================
   private static final String RELABEL_NUMBER = "relabelNumber"; //$NON-NLS-1$
   private static final String DIR_UTIL_UNIQUE_DOWN = "uniqueRenamingDowload"; //$NON-NLS-1$
   private static final String DIR_UTIL_WINDOWS = "dirUtilWindows"; //$NON-NLS-1$
   private static final String DIR_UTIL_FIRST_ENDING = "dirUtilFirstEnding"; //$NON-NLS-1$

   public static int getRelabelNumber()
   {
      return clsPrefs.getInt(RELABEL_NUMBER, 0);
   }

   public static void setRelabelNumber(int setting)
      throws Exception
   {
      clsPrefs.putInt(RELABEL_NUMBER, setting);
      clsPrefs.flush();
   }

   public static boolean getUniqueRenamingDownload()
   {
      return clsPrefs.getBoolean(DIR_UTIL_UNIQUE_DOWN, false);
   }

   public static void setUniqueRenamingDownload(boolean setting)
   {
      clsPrefs.putBoolean(DIR_UTIL_UNIQUE_DOWN, setting);
   }

   public static boolean getWindowsFileConvention()
   {
      return clsPrefs.getBoolean(DIR_UTIL_WINDOWS, false);
   }

   public static void setWindowsFileConvention(boolean setting)
   {
      clsPrefs.putBoolean(DIR_UTIL_WINDOWS, setting);
   }

   public static boolean getFirstEndingConvention()
   {
      return clsPrefs.getBoolean(DIR_UTIL_FIRST_ENDING, false);
   }

   public static void setFirstEndingConvention(boolean setting)
   {
      clsPrefs.putBoolean(DIR_UTIL_FIRST_ENDING, setting);
   }

   //===========================================================================
   //                          Strain Options
   //===========================================================================

   private static final String STRAIN = "strainOption"; //$NON-NLS-1$
   static final int STRAIN_NOTHING = 0;
   static final int STRAIN_MOVE = 1;
   static final int STRAIN_DELETE = 2;

   public static int getStrain()
   {
      return clsPrefs.getInt(STRAIN, STRAIN_NOTHING);
   }

   public static void setStrain(int setting)
   {
      clsPrefs.putInt(STRAIN, setting);
   }

   //===========================================================================
   //                          Resize Options
   //===========================================================================
   private static final String RESIZE = "dirResize"; //$NON-NLS-1$
   private static final String DIR_UTIL_HEIGHT = "dirUtilHeight"; //$NON-NLS-1$
   private static final String DIR_UTIL_WIDTH = "dirUtilWidth"; //$NON-NLS-1$

   public static boolean getResize()
   {
      return clsPrefs.getBoolean(RESIZE, false);
   }

   public static void setResize(boolean setting)
   {
      clsPrefs.putBoolean(RESIZE, setting);
   }

   public static int getResizeWidth()
   {
      return clsPrefs.getInt(DIR_UTIL_WIDTH, clsFullScreenWidth);
   }

   public static void setResizeWidth(int setting)
   {
      clsPrefs.putInt(DIR_UTIL_WIDTH, setting);
   }

   public static int getResizeHeight()
   {
      return clsPrefs.getInt(DIR_UTIL_HEIGHT, clsFullScreenHeight);
   }

   public static void setResizeHeight(int setting)
   {
      clsPrefs.putInt(DIR_UTIL_HEIGHT, setting);
   }

   //===========================================================================
   //                          Repack Options
   //===========================================================================
   private static final String REPACK = "repack"; //$NON-NLS-1$

   public static boolean getRepack()
   {
      return clsPrefs.getBoolean(REPACK, false);
   }

   public static void setRepack(boolean setting)
   {
      clsPrefs.putBoolean(REPACK, setting);
   }

   //===========================================================================
   //                          Skip Recheck Options
   //===========================================================================
   private static final String SKIP_RECHECK = "skip_recheck"; //$NON-NLS-1$

   public static boolean getSkipRecheck()
   {
      return clsPrefs.getBoolean(SKIP_RECHECK, false);
   }

   public static void setSkipRecheck(boolean setting)
   {
      clsPrefs.putBoolean(SKIP_RECHECK, setting);
   }

   //===========================================================================
   //                              Other Public Functions
   //===========================================================================
   public static void flush()
   {
      try
      {
         clsPrefs.flush();
      }
      catch (BackingStoreException ex)
      {
      	ex.printStackTrace();
      }
   }

}
