/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package org.gerhardb.jibs.optimizer;

import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;

import org.gerhardb.jibs.Jibs;
import org.gerhardb.jibs.util.FileLoopHelper;
import org.gerhardb.lib.dirtree.rdp.PathManager;
import org.gerhardb.lib.dirtree.rdp.RDPplugins;
import org.gerhardb.lib.image.IOImage;
import org.gerhardb.lib.image.ImageChangeUtil;
import org.gerhardb.lib.image.ImageFactory;
import org.gerhardb.lib.io.EzLogger;
import org.gerhardb.lib.io.FileUtil;
import org.gerhardb.lib.util.Killer;

/**
 * Does all the actual filtering work.
 * <p>
 * Orginally intended to be able to be driven from a command line, but that
 * go lost in the suffle.
 * <p>
 * Notes: All files without an image file name are automatically DELETED!!!.
 */
public class WorkerDownload
   implements Runnable
{   
   static final String[] ENDINGS = new String[]{"jpg", "jpeg", "j2k", //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
		"jif", "jiff", "jp2", "jpe", "jpf", "jpx", //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$ //$NON-NLS-6$ 
		"tif", "tiff", "dcm", "fits", "pgm", "bmp", "gif", //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$ //$NON-NLS-6$ //$NON-NLS-7$
		"apng", "png", "psd", //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$ //$NON-NLS-6$ //$NON-NLS-7$
		"pnm", "pgm", "pbm", "ppm", }; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-4$
   
   Killer myKiller;

   // PASSED IN
   boolean iDedup;
   boolean iAcrossDirectories;
   boolean iRenameUnique;
   boolean iWindows;
   boolean iCleanUpEnding;
   int myStrain;
   boolean iResize;
   boolean iSkipRechecking;
   boolean iOptomize;
   int myWidth;
   int myHeight;

   // OTHER VARIABLES
   // int is all file read supports anyway!
   int myRemovedCount = 0;
   int myMaxSameSize = 0;
   File myParkDir;

   public static final SimpleDateFormat LONG_DATE =
      new SimpleDateFormat("yyyy.MMMMM.dd HH:mm:ss"); //$NON-NLS-1$

   // PASSED IN
   File[] myWorkingDirs = new File[0];
   File[] myStableDirs = new File[0];
   FileLoopHelper myHelper = null;

   // OTHER VARIABLES
   File[] myWorkingFiles;

   // RelabelUnique
   int myRelabelCount;

   //===========================================================================
   //                              Constructor
   //===========================================================================
   private WorkerDownload()
   {
   	// Don't allow public creation.   	
   }
   
   public WorkerDownload(
      File[] workingDirs,
      File[] stableDirs,
      boolean dedup,
      boolean across,
      boolean rename,
      boolean windows,
      boolean cleanUpEnding,
      int strain,
      boolean resize,
      boolean optomize,
      //float quality,
      int width,
      int height,
      FileLoopHelper helper,
      Killer killer,
      RDPplugins rdpPlugins,
      boolean skipRechecking
      )
      throws Exception
   {
      if ( workingDirs != null ){ this.myWorkingDirs = workingDirs; }
      if ( stableDirs != null ){ this.myStableDirs = stableDirs; }
      this.myHelper = helper;
      this.myRelabelCount = OptimizerPreferences.getRelabelNumber();

      this.myHelper.setValue(0);
      this.myHelper.setMaximum(1);
      this.myHelper.setIndeterminate(false);

      this.myHelper.getLog().logLine(Jibs.getString("WorkerDownload.14") + ": ");  //$NON-NLS-1$//$NON-NLS-2$
      for (int i = 0; i < this.myWorkingDirs.length; i++)
      {
         this.myHelper.getLog().logLine("   " + this.myWorkingDirs[i].getAbsolutePath() //$NON-NLS-1$
            + ""); //$NON-NLS-1$
      }

      this.iDedup = dedup;
      this.iAcrossDirectories = across;
      this.iRenameUnique = rename;
      this.iWindows = windows;
      this.iCleanUpEnding = cleanUpEnding;
      this.myStrain = strain;
      this.iResize = resize;
      this.iSkipRechecking = skipRechecking;
      this.iOptomize = optomize;
      this.myWidth = width;
      this.myHeight = height;
      this.myKiller = killer;

      String parkDir = rdpPlugins.getPathManager().getDirectoryAbsolute(PathManager.DIR_PARK);
      if (parkDir != null)
      {
         this.myParkDir = new File(parkDir);
      }

      this.myHelper.getLog().logLine(Jibs.getString("WorkerDownload.13") + ": "); //$NON-NLS-1$ //$NON-NLS-2$
      this.myHelper.getLog().logLine("   " + Jibs.getString("WorkerDownload.15") + ": " + this.iDedup + ""); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
      this.myHelper.getLog().logLine("   " + Jibs.getString("WorkerDownload.19") + ": " //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
         + this.iAcrossDirectories
         + ""); //$NON-NLS-1$
      this.myHelper.getLog().logLine(Jibs.getString("WorkerDownload.11") + ": "); //$NON-NLS-1$ //$NON-NLS-2$
      switch (this.myStrain)
      {
         case OptimizerPreferences.STRAIN_NOTHING:
            this.myHelper.getLog().logLine(Jibs.getString("WorkerDownload.23")); //$NON-NLS-1$
            break;
         case OptimizerPreferences.STRAIN_MOVE:
            this.myHelper.getLog().logLine("   " + Jibs.getString("WorkerDownload.25")); //$NON-NLS-1$ //$NON-NLS-2$
            this.myHelper.getLog().logLine("      " + Jibs.getString("WorkerDownload.27") + ": " + ImageFactory.getImageFactory().getFilter() + ""); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
            this.myHelper.getLog().logLine("      " + Jibs.getString("WorkerDownload.31") + ": "); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
            if (parkDir != null)
            {
               this.myHelper.getLog().logLine(parkDir + ""); //$NON-NLS-1$
            }
            else
            {
               this.myHelper.getLog().logLine(Jibs.getString("WorkerDownload.34")); //$NON-NLS-1$
            }
            break;
         case OptimizerPreferences.STRAIN_DELETE:
            this.myHelper.getLog().logLine("   " + Jibs.getString("WorkerDownload.36")); //$NON-NLS-1$ //$NON-NLS-2$
            this.myHelper.getLog().logLine("      " + Jibs.getString("WorkerDownload.38") + Jibs.getString("colon") + ImageFactory.getImageFactory().getFilter() + ""); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
            break;
      }
      this.myHelper.getLog().logLine(Jibs.getString("WorkerDownload.41") + ": "); //$NON-NLS-1$ //$NON-NLS-2$
      this.myHelper.getLog().logLine("   " + Jibs.getString("WorkerDownload.44") + ": " + this.iRenameUnique + ""); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
      this.myHelper.getLog().logLine("      " + Jibs.getString("WorkerDownload.48") + ": " //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
         + FileLoopHelper.RELABEL_POSTFIX + ""); //$NON-NLS-1$
      this.myHelper.getLog().logLine(""); //$NON-NLS-1$
      this.myHelper.getLog().logLine("      " + Jibs.getString("WorkerDownload.53") + ": " + this.myRelabelCount + ""); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
      this.myHelper.getLog().logLine("   " + Jibs.getString("WorkerDownload.57") + ": " + this.iWindows //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
         + ""); //$NON-NLS-1$
      this.myHelper.getLog().logLine("   " + Jibs.getString("WorkerDownload.61") + ": " + this.iCleanUpEnding + ""); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
      this.myHelper.getLog().logLine(Jibs.getString("WorkerDownload.64") + ": "); //$NON-NLS-1$ //$NON-NLS-2$
      this.myHelper.getLog().logLine("   " + Jibs.getString("WorkerDownload.67") + ": " + this.iResize + ""); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
      this.myHelper.getLog().logLine("   " + Jibs.getString("WorkerDownload.71") + ": " + this.myWidth + ""); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
      this.myHelper.getLog().logLine("   " + Jibs.getString("WorkerDownload.75") + ": " + this.myHeight + ""); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
      this.myHelper.getLog().logLine(Jibs.getString("WorkerDownload.78") + ": "); //$NON-NLS-1$ //$NON-NLS-2$
      this.myHelper.getLog().logLine("   " + Jibs.getString("WorkerDownload.81") + ": " + this.iOptomize + ""); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
   }
   
   boolean iDoSomeThingBesidesDedup()
   {
   	return this.iRenameUnique || this.iWindows || this.iCleanUpEnding
      || (this.myStrain != OptimizerPreferences.STRAIN_NOTHING)
      || this.iResize || this.iOptomize;
   }

   // ==========================================================================
   // Runnable Interface
   // ==========================================================================
   public void run()
   {
      this.myHelper.getLog().logLine(EzLogger.DIVIDER);
      Date startDate = new Date();
      this.myHelper.getLog().logLine(
         Jibs.getString("WorkerDownload.88") + ": " + LONG_DATE.format(startDate) + ""); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$

      this.myHelper.getLabel().setText(Jibs.getString("WorkerDownload.91")); //$NON-NLS-1$

    	// Skip if just dedup within directories.
      if (this.iAcrossDirectories || iDoSomeThingBesidesDedup() || this.myStableDirs.length > 0)
      {
        	ArrayList<File> outList = this.myHelper.getFilesFromDirectories(this.myWorkingDirs);
      	this.myWorkingFiles = outList.toArray(new File[outList.size()]);
      }

      if (this.iDedup && this.myKiller.live())
      {
         dedup();
      }

      // See if there is anything else to do now that we finished
      // doing the dedup.
      if (this.myKiller.live() && iDoSomeThingBesidesDedup())
      {
         this.myHelper.getLog().logLine(EzLogger.DIVIDER);
         this.myHelper.getLog().logLine(Jibs.getString("WorkerDownload.92") + ": " + this.myWorkingFiles.length + ""); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
         this.myHelper.setValue(0);
         this.myHelper.setMaximum(this.myWorkingFiles.length - 1);

         for (int i = 0; i < this.myWorkingFiles.length; i++)
         {
            if (this.myKiller.die())
            {
               break;
            }
            this.myHelper.setValue(i);
            int count = i + 1;
            String progress = Jibs.getString("WorkerDownload.95") + " " + count + " " + Jibs.getString("WorkerDownload.97") + " " //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$
                              + this.myWorkingFiles.length;
            this.myHelper.getLabel().setText(progress);

            // File could be null because of dedup process
            // That is, nulling the file out is how to indicate
            // the file has been fully processed.
            if (this.myWorkingFiles[i] != null)
            {
               processIndex(i);
               // We don't need this in memory any more!
               this.myWorkingFiles[i] = null;
            }
         }
         this.myHelper.getLog().logLine(EzLogger.DIVIDER);
         this.myHelper.getLog().logLine(Jibs.getString("WorkerDownload.98") + " " + this.myWorkingFiles.length + " " //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
            + Jibs.getString("WorkerDownload.99")); //$NON-NLS-1$
      }

      // Close operations.
      summary();

      Date stopDate = new Date();
      this.myHelper.getLog().logLine(
         Jibs.getString("WorkerDownload.100") + ": " + LONG_DATE.format(stopDate) + ""); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$

      long seconds = ((stopDate.getTime() - startDate.getTime()) / 1000);
      this.myHelper.getLog().logLine(
         Jibs.getString("WorkerDownload.103") + ": " + seconds + ""); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$

      this.myHelper.getLabel().setText(Jibs.getString("WorkerDownload.106")); //$NON-NLS-1$
      try
      {
         this.myHelper.getLog().close();
      }
      catch (Exception ex)
      {
         ex.printStackTrace();
      }

      this.myHelper.updateCountField();
      try
      {
         OptimizerPreferences.setRelabelNumber(this.myHelper.getRelableCount());
      }
      catch (Exception ex)
      {
      	ex.printStackTrace();
      }

      // Now that this thread has concluded its work,
      // let the caller know.
      this.myKiller.dead();
      
      this.myHelper.done();
   }

   // ==========================================================================
   // Workder Interface
   // ==========================================================================
   void dedup()
   {
      // Log Setup
      Date startTime = new Date();

      Deduper deduper = new Deduper(this.myHelper, this.myStableDirs, this.myKiller);

      // Stable Files
      this.myHelper.getLabel().setText(
         Jibs.getString("WorkerDownload.107")); //$NON-NLS-1$
      
      if (!deduper.setWorking(this.myWorkingFiles))
      {
         // Fast return if any problems.
         // If there are no working files across all directories,
         // there sure won't be any within a directory!!
         this.myHelper.getLog().logLine(Jibs.getString("WorkerDownload.108")); //$NON-NLS-1$
         this.myHelper.getLog().logLine(EzLogger.DIVIDER);
         return;      	
      }
      
      // Only do is we have some stable files to work on.
      // Otherwise myWorkingFiles may have a null in it.
      if (this.myStableDirs.length > 0)
      {
         deduper.dedupStable();
      }

      if (this.iAcrossDirectories)
      {
         this.myHelper.getLog().logLine(Jibs.getString("WorkerDownload.109")); //$NON-NLS-1$
         this.myHelper.getLabel().setText(Jibs.getString("WorkerDownload.110")); //$NON-NLS-1$

         // Use entire working list already set.
         deduper.dedupWorking();
      }
      else
      {
         this.myHelper.getLog().logLine(Jibs.getString("WorkerDownload.111")); //$NON-NLS-1$
         this.myHelper.getLog().logLine(Jibs.getString("WorkerDownload.112") + Jibs.getString("colon") + " " //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
            + this.myWorkingDirs.length + ""); //$NON-NLS-1$
         NoDirectories noDirectories = new NoDirectories();
         for (int i = 0; i < this.myWorkingDirs.length; i++)
         {
            if (this.myKiller.die())
            {
                break;
            }
            String progress =
               Jibs.getString("WorkerDownload.114") + i + ": " + this.myWorkingDirs[i]; //$NON-NLS-1$ //$NON-NLS-2$
            this.myHelper.getLabel().setText(progress);
            this.myHelper.getLog().logLine(progress);

            // Replace working list with the individual directory lists
            // before processing.
            File[] dedupThese = this.myWorkingDirs[i].listFiles(noDirectories);
            if (deduper.setWorking(dedupThese))
            {
               deduper.dedupWorking();
            }
         }
      }

      // Log Close Out
      Date stopTime = new Date();
      this.myRemovedCount = deduper.getRemovedCount();
      this.myMaxSameSize = deduper.getMaxSameSize();
      this.myHelper.getLog().logLine(Jibs.getString("WorkerDownload.116")); //$NON-NLS-1$
      this.myHelper.getLog().logLine(this.myRemovedCount + " " + Jibs.getString("WorkerDownload.117")); //$NON-NLS-1$ //$NON-NLS-2$
      this.myHelper.getLog().logTime(Jibs.getString("WorkerDownload.118"), startTime, stopTime); //$NON-NLS-1$
      this.myHelper.getLog().logLine(Jibs.getString("WorkerDownload.119") + ": " //$NON-NLS-1$ //$NON-NLS-2$
         + this.myMaxSameSize
         + System.getProperty("line.separator")); //$NON-NLS-1$
      this.myHelper.getLog().logLine(EzLogger.DIVIDER);
   }

   void summary()
   {
      this.myHelper.getLog().logLine(this.myRemovedCount + " " + Jibs.getString("WorkerDownload.8")); //$NON-NLS-1$ //$NON-NLS-2$
      this.myHelper.getLog().logLine(
         Jibs.getString("WorkerDownload.123") + ": " + this.myMaxSameSize //$NON-NLS-1$ //$NON-NLS-2$
         + System.getProperty("line.separator")); //$NON-NLS-1$
      try
      {
         OptimizerPreferences.setRelabelNumber(this.myRelabelCount);
      }
      catch (Exception ex)
      {
         ex.printStackTrace();
      }
   }

//===========================================================================
//                              Private Functions
//===========================================================================

   /**
    * Do things BEFORE reading the file.
    * Once file is read, it will keep a lock preventing
    * moves, deletes, or renames.
    * @param index
    */
   void processIndex(int index)
   {
      // File could be null because of dedup process
      // That is, nulling the file out is how to indicate the file has been
      // fully processed.
      if (this.myWorkingFiles[index] == null)
      {
         return;
      }
      
      // If we have already seen this file and we are not rechecking,
      // we can just stop now...
      if (this.iSkipRechecking)
      {
      	if (FileLoopHelper.alreadyProcessed(this.myWorkingFiles[index].getName()))
      	{
      		return;
      	}
      }      

      // Take care of any straining based on the name ending alone.
      if (this.myStrain != OptimizerPreferences.STRAIN_NOTHING)
      {
         // If we are straining, we can get rid of any files without the right
         // file name.
         if (!ImageFactory.getImageFactory().getFilter().accept(this.myWorkingFiles[index]))
         {
            moveOrDelete(this.myWorkingFiles[index]);
            return;
         }
      }

      if (this.iRenameUnique || this.iWindows || this.iCleanUpEnding)
      {
         // Need to update the file name from this routine if renamed.
         File oldName = this.myWorkingFiles[index];
         this.myWorkingFiles[index] = rename(this.myWorkingFiles[index]);
         if (!oldName.equals(this.myWorkingFiles[index]))
         {
            this.myHelper.getLog().logLine("   " + Jibs.getString("WorkerDownload.6") + ": " //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
               + oldName.getAbsolutePath()
               + "\n      " + Jibs.getString("WorkerDownload.3") + ": " + this.myWorkingFiles[index].getAbsolutePath()  //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
               + "\n"); //$NON-NLS-1$
         }
      }

      if ((this.myStrain != OptimizerPreferences.STRAIN_NOTHING) || this.iResize
         || this.iOptomize)
      {
         doFileOperations(this.myWorkingFiles[index]);
      }

      // We don't need this in memory any more!
      this.myWorkingFiles[index] = null;
   }

   private void doFileOperations(File theFile)
   {
      BufferedImage originalImage = null;
      IOImage goodImg = ImageFactory.getImageFactory().makeImageEZ(theFile);
      if (goodImg != null)
      {
         try
         {
            originalImage = goodImg.getImage();
         }
         catch (Exception ex)
         {
            // we don't care
         }
      }

      if (goodImg == null || originalImage == null)
      {
         if (this.myStrain != OptimizerPreferences.STRAIN_NOTHING)
         {
            moveOrDelete(theFile);
         }
         return;
      }

      if (this.iResize || this.iOptomize)
      {
      	long startLength = theFile.length();
         BufferedImage saveImg = null;
         try
         {
            if (this.iResize)
            {
               saveImg = ImageChangeUtil.resizeImageDown(
                         originalImage, this.myWidth, this.myHeight);
               if (saveImg != null)
               {
                  this.myHelper.getLog().logLine(
                     "   " + Jibs.getString("WorkerDownload.132") + ": " + theFile.getAbsolutePath()); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
               }
            }

            // In case resize off or it did not need resizing.
            if (saveImg == null && this.iOptomize)
            {
               saveImg = ImageChangeUtil.scale(originalImage, 1.0F);
               if (saveImg != null)
               {
                  this.myHelper.getLog().logLine(
                     "   " + Jibs.getString("WorkerDownload.136") + ": " + theFile.getAbsolutePath()); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
               }
            }

            if (saveImg != null)
            {
               goodImg.save(saveImg);
            }
         	long endLength = theFile.length();
            this.myHelper.getLog().logLine("  Size from " + startLength + " to " + endLength + " reduced by " + (startLength - endLength)); 
         }
         catch (Exception ex)
         {
            ex.printStackTrace();
            this.myHelper.getLog().logLine(
               "   " + Jibs.getString("WorkerDownload.140") + ": " //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
               + theFile.getAbsolutePath() + "\n"); //$NON-NLS-1$
         }
      }
   }

   private File rename(File currFile)
   {
   	File rtnMe = currFile;
      // Prevents duplicates from operations below.
      if (this.iRenameUnique)
      {
      	rtnMe = this.myHelper.renameUnique(rtnMe);
      }

      if (this.iCleanUpEnding)
      {
      	rtnMe = cleanUpEnding(rtnMe);
      }

      if (this.iWindows)
      {
      	rtnMe = windowsCleanup(rtnMe);
      }

      return rtnMe;
   }

   private void moveOrDelete(File currFile)
   {
      switch (this.myStrain)
      {
         case OptimizerPreferences.STRAIN_NOTHING:
            break;
         case OptimizerPreferences.STRAIN_MOVE:
            try
            {
               FileUtil.moveFile(this.myParkDir, currFile, 
               		null, this.myHelper.getLog(), true);
               this.myHelper.getLog().logLine("   " + Jibs.getString("WorkerDownload.144") + ": " //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
                  + currFile.getAbsolutePath() + "\n"); //$NON-NLS-1$
            }
            catch (Exception ex)
            {
               this.myHelper.getLog().logLine("   " + Jibs.getString("WorkerDownload.0") + ": " //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
                  + ex.getMessage() + "\n"); //$NON-NLS-1$

            }
            break
               ;
         case OptimizerPreferences.STRAIN_DELETE:
            if (currFile.delete())
            {
               this.myHelper.getLog().logLine("   " + Jibs.getString("WorkerDownload.152") + ": " //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
                  + currFile.getAbsolutePath() + "\n"); //$NON-NLS-1$
            }
            else
            {
               this.myHelper.getLog().logLine("   " + Jibs.getString("WorkerDownload.156") + ": " //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
                  + currFile.getAbsolutePath() + "\n"); //$NON-NLS-1$
            }
            break;
      }
   }
   

   /**
    * Valid characters for a windows file name.
    * everything but < > : " \ / | ? *
    * Anything less than 32
    */
   /*
   static final char[] GOOD = // NO_UCD
      {
      'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n',
      'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z',
      'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N',
      'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z',
      '1', '2', '3', '4', '5', '6', '7', '8', '9', '0',
      '.', '-', '_', '~', '!', '@', '#', '$', '%', '^', '&', '(', ')', '+',
      '=',
      '`', '\'', ',', ';', '{', '}', '[', ']', ' '
   };
   */
   
   static final char[] BAD = {
      '<', '>', ':', '"', '\\', '/', '|', '?', '*'
   };
   
   boolean isGood(char aCharacter)
   {
   	if (aCharacter < 32)
   	{
         return false;      		
   	}
      for (int i = 0; i < BAD.length; i++)
      {
         if (aCharacter == BAD[i])
         {
            return false;
         }      
      }
      return true;
   }
   
   boolean isNameGood(File file)
   {
      char[] oldName = file.getName().toCharArray();
      for (int i = 0; i < oldName.length; i++)
      {
      	if (!isGood(oldName[i]))
      	{
      		return false;           		
      	}
      }
   	return true;
   }
   
	/**
	 * Strips from the file name any characters which are not allowed in Windows.
	 * @param file
	 * @return
	 */
   File windowsCleanup(File file)
   {
   	// Fast exit for general case where name is perfectly good.
   	if (isNameGood(file))
   	{
   		return file;
   	}
   	
   	// OK, bad name, let's do work of renaming it.
      StringBuffer buff = new StringBuffer();
      try
      {
         buff.append(new File(file.getParent()).getCanonicalPath());
      }
      catch (IOException x)
      {
         return file;
      }
      buff.append(File.separatorChar);
      char[] oldName = file.getName().toCharArray();
      for (int i = 0; i < oldName.length; i++)
      {
      	if (isGood(oldName[i]))
      	{
            buff.append(oldName[i]);      		
      	}
      	else
      	{
            buff.append('_');           		
      	}
      }

      File newFile = new File(buff.toString());
      if (!file.equals(newFile))
      {
         if (newFile.exists())
         {
            this.myHelper.getLog().logLine(
               Jibs.getString("WorkerDownload.164") + ": " + newFile + "\n"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
         }
         else
         {
            file.renameTo(newFile);
            return newFile;
         }
      }
      return file;
   }
   

   /**
    * take care of common jpg_(1_1) problems and 1_jpg.
    * @param file
    */
   File cleanUpEnding(File file)
   {
   	//System.out.println("----------------------------------------- > " + file);   
   	String name = file.getName();
      int lastPeriod = name.lastIndexOf('.');
      // We don't care if the file has no ending.
      // If the file has more than one ending, take the first one.
      // In any event, if it looks like an image file, assume it is.
      // And we need a front and a back.
      if (lastPeriod < 1)
      {
      	//System.out.println("No Change: No period in file name");   
         return file;
      }
      	
      String front = name.substring(0, lastPeriod);
      String back = name.substring(lastPeriod + 1);
   	//System.out.println(name + ": " + front + " " + back);

      String backProblem = back.toLowerCase();

      // Ending is not already an image.
      // Correct if it should be.
      boolean changeEnding = false;
      for (int i = 0; i < ENDINGS.length; i++) 
      {
      	//System.out.println("Checking: " + ENDINGS[i]);
         if (backProblem.indexOf(ENDINGS[i]) > -1) // contains() is 1.5 or greater
         {
            // Quick check to see if the ending is already OK.
            if (backProblem.equals(ENDINGS[i]))
            {
            	//System.out.println("Quick Return");
               return file;
            }
            back = ENDINGS[i];
           	//System.out.println("Bingo! - " + back);
            changeEnding = true;
            break;
         }
      }

      if (changeEnding)
      {
         String newName = file.getParent()
                          + File.separator + front + "." + back; //$NON-NLS-1$

         File newFile = new File(newName);
         if (newFile.exists())
         {
            this.myHelper.getLog().logLine(Jibs.getString("WorkerDownload.161") + ": " + newFile + "\n"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
         }
         else
         {
            file.renameTo(newFile);
         	//System.out.println("Rename: " + newFile);   
            return newFile;
         }
      }
      else
      {
      	System.out.println("No Change: Could not find ending.  This message should never appear.");         	
      }
      return file;
   }

   
   public static void main(String[] args)
   {
   	WorkerDownload wd = new WorkerDownload();
   	System.out.println(wd.isNameGood(new File("c:/pics/ann.txt"))); //$NON-NLS-1$
   	System.out.println(wd.isNameGood(new File("c:/pics/a>n.txt"))); //$NON-NLS-1$
   	
   	/*
   	System.out.println(wd.cleanUpEnding(new File("c:/pics/ann"))); //$NON-NLS-1$
   	System.out.println(wd.cleanUpEnding(new File("c:/pics/ann.doc"))); //$NON-NLS-1$
   	System.out.println(wd.cleanUpEnding(new File("c:/pics/ann.jpg"))); //$NON-NLS-1$
   	System.out.println(wd.cleanUpEnding(new File("c:/pics/ann.JPG"))); //$NON-NLS-1$
   	System.out.println(wd.cleanUpEnding(new File("c:/pics/ann.1_jpg"))); //$NON-NLS-1$
   	System.out.println(wd.cleanUpEnding(new File("c:/pics/ann.jpg_1")));   	 //$NON-NLS-1$
   	System.out.println(wd.cleanUpEnding(new File("c:/pics/ann.jpf"))); //$NON-NLS-1$
   	System.out.println(wd.cleanUpEnding(new File("c:/pics/ann.JPF"))); //$NON-NLS-1$
   	System.out.println(wd.cleanUpEnding(new File("c:/pics/ann.1_jpf"))); //$NON-NLS-1$
   	System.out.println(wd.cleanUpEnding(new File("c:/pics/ann.JPF_1")));   	 //$NON-NLS-1$
   	*/
   }
}
