/*
 * Copyright (c) 2008 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
package org.gerhardb.jibs.producer;

import java.io.*;
import java.util.*;

public class PlayFromPipedFormat 
{
	File myFile = null;
	String myDirectory = null;
	
	/**
	Assumes files are all in the same directory as the play list, 
	or sub-directories.
	*/
	public PlayFromPipedFormat( File file )
	throws Exception
	{
		if ( file == null ) throw new Exception( "BasicPageFactory requires non-null file." );
		if ( file.isDirectory() ) throw new Exception( "BasicPageFactory requires a file, not a directory." );
		
		this.myFile = file;
		this.myDirectory = this.myFile.getParentFile().getAbsolutePath() + "/";
		//System.out.println( myDirectory );
	}
	
	/**
	Assumes files are relative to the picture root dir.
	*/
	public PlayFromPipedFormat( File file, File pictureRootDir )
	throws Exception
	{
		if ( file == null ) throw new Exception( "BasicPageFactory requires non-null file." );
		if ( !pictureRootDir.isDirectory() ) throw new Exception( "BasicPageFactory pictureRootDir requires a directory, not a file." );
		
		this.myFile = file;
		this.myDirectory = pictureRootDir.getAbsolutePath() + "/";
		//System.out.println( myDirectory );
	}
	
	/*
	=============================================================================
	 													 Public Functions
	=============================================================================
	*/
	public Play[] getPlayList( )
	{
		ArrayList<Play> pages = new ArrayList<Play>();
 		try
		{
			RandomAccessFile readFile = new RandomAccessFile( this.myFile, "r" );
			String sLine = readFile.readLine();
			while( sLine != null )
			{
				Play aPage = extractPlayFromLine( sLine );
				if ( aPage != null ) pages.add( aPage );
				sLine = readFile.readLine();
			}
			readFile.close();
		}
		catch( IOException ex)
      {
      	ex.printStackTrace();
      }
		
		//System.out.println("Finished reading file");
		
		if (pages.isEmpty() )	return null;
		
		Play[] rtnMe = new Play[pages.size()];
		for( int i=0; i<rtnMe.length; i++ )
		{
			rtnMe[i] = pages.get(i);
		}
		
		return rtnMe;
	} // getPlayList
	
	/*
	public boolean saveToFile( String fileName )
	{
		try
		{
		FileOutputStream fileStream = new FileOutputStream( fileName );
		PrintWriter aPrintWriter = new PrintWriter( fileStream, true);
		Enumeration enum = myPlayList.elements();
		while( enum.hasMoreElements() )
		{
			aPrintWriter.println( enum.nextElement() );
		}
		aPrintWriter.close();
		fileStream.close();
		}
		catch( IOException e){return false;}
		return true;
	} //saveToFile( String fileName )
	*/
	
	/*
	=============================================================================
	 													 Private Protected Functions
	=============================================================================
	*/
	
	/**
	Expeceted Line Structure:
	title|time|image|sound1|sound2
	
	Where there can be any number of sounds.
	Empty elements must have a tilde ~.
	Tilde is required because normal java tokenizer does not return a null if 
	there are a set of empty tokens - it returns the next non-empty token.
	*/
	private Play extractPlayFromLine( String line )
	{
		//System.out.println( "Line: " + line );
		line = line.trim();
		if (line.length() == 0) return null;
		
		int time = 0;
		String caption = null;
		String pic = null;
		
		StringTokenizer tokens = new StringTokenizer(line, "|");
		
		if( tokens.hasMoreTokens() )
		{
			String token = tokens.nextToken();
			//System.out.println( "Title: " + token );
			caption = noTilde( token );
		}
		
		if( tokens.hasMoreTokens() )
		{
			String token = tokens.nextToken();
			//System.out.println( "Time: " + token );
			try{ time = Integer.parseInt( token ); }
			catch( NumberFormatException ex )
	      {
	      	ex.printStackTrace();
	      }			
		}
		
		if( tokens.hasMoreTokens() )
		{
			String token = tokens.nextToken();
			//System.out.println( "Picture: " + token );
			pic = noTilde( token );
			if ( pic != null ) pic = this.myDirectory + pic;
		}
		
		Play play = new Play();
		play.setImageFile(pic);
		play.setMinSeconds(time);
		play.setCaption(caption);
		
		while( tokens.hasMoreTokens() )
		{
			String sound = noTilde( tokens.nextToken() );
			//System.out.println( "Sound: " + sound );
			if ( sound != null )
			{
				play.addAudio(new Audio(this.myDirectory + sound));
			}
		}
		
		return play;
	} // extractPlayFromLine()
	
	
	private String noTilde( String input )
	{
		if ( input.equals("~") )
		{
			return null;
		}
		return input;
	}
	
	/*
	PlayLine makePlay( String sLine )
	{
		StringTokenizer tokens = new StringTokenizer(sLine, "|");
		PlayLine aLine = new PlayLine();
		if( tokens.hasMoreTokens() )
			aLine.title = noTilde( tokens.nextToken() );
		if( tokens.hasMoreTokens() )
			aLine.setTime( noTilde( tokens.nextToken() ) );
		if( tokens.hasMoreTokens() )
			aLine.image = noTilde( tokens.nextToken() );
		if( tokens.hasMoreTokens() )
			aLine.sound1 = noTilde( tokens.nextToken() );
		if( tokens.hasMoreTokens() )
				aLine.sound2 = noTilde( tokens.nextToken() );
		// Take sound duplicates
		if ( aLine.sound1 ==  null )
			aLine.sound1 = aLine.sound2;
		return aLine;
	}
	*/

	/*
	=============================================================================
	 													 Test Harness
	=============================================================================
	*/
	public static void main( String[] args )
	{
		System.out.println( "BasicPageFactory TEST PROGRAM starting!\n" );
		
		PlayFromPipedFormat factory = null;
		File playFile = new File("/dev/workspaces/JIBS/src/test/java/org/gerhardb/jibs6/producer/PipedPlayList.txt");
		try{ factory = new PlayFromPipedFormat( playFile ); }
		catch( Exception ex )
		{ 
			System.out.println( ex.getMessage() );
			return;
		}
		
		Play aPage = factory.extractPlayFromLine( "aTitle|100|BasicPageFactory.java|Page.java" );
		System.out.println( aPage );
		
		aPage = factory.extractPlayFromLine( " " );
		System.out.println( aPage );
		
		aPage = factory.extractPlayFromLine( "~|~|BasicPageFactory.java" );
		System.out.println( aPage );
		
		aPage = factory.extractPlayFromLine( "~|~|BasicPageFactory.java|Page.java" );
		System.out.println( aPage );
		
		aPage = factory.extractPlayFromLine( "aTitle|~|BasicPageFactory.java" );
		System.out.println( aPage );
		
		aPage = factory.extractPlayFromLine( "aTitle|0|BasicPageFactory.java" );
		System.out.println( aPage );
		
		aPage = factory.extractPlayFromLine( "aTitle|0|BasicPageFactory.java|Page.java" );
		System.out.println( aPage );
		
		aPage = factory.extractPlayFromLine( "aTitle|0|BasicPageFactory.java|Page.java|BasicPage.java" );
		System.out.println( aPage );
		
		aPage = factory.extractPlayFromLine( "aTitle|0|BasicPageFactory.java|Page.java|BasicPage.java|FilePage.java" );
		System.out.println( aPage );
		
		aPage = factory.extractPlayFromLine( "aTitle|0|BasicPageFactory.java|Page.java|BasicPage.java|FilePage.java|Scroll.java" );
		System.out.println( aPage );
		
		aPage = factory.extractPlayFromLine( "aTitle|0|BasicPageFactory.java|Page.java|BasicPage.java|FilePage.java|Scroll.java" );
		System.out.println( aPage );
		
		System.out.println( "###################################################################" );
		PlayFromPipedFormat factory2 = null;
		try
		{ 
			factory2 = new PlayFromPipedFormat( 
				new File( "/Kids/a.play" ) ); 
		}
		catch( Exception ex )
		{ 
			System.out.println( ex.getMessage() );
			return;
		}
		
		Play[] list = factory2.getPlayList( );
		for( int i=0; i<list.length; i++ )
		{
			System.out.println( list[i] );
		}
		
		System.out.println( "!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!" );
		PlayFromPipedFormat factory3 = null;
		try
		{ 
			factory3 = new PlayFromPipedFormat( 
				new File( "/dev/workspaces/JIBS/src/test/java/org/gerhardb/jibs6/producer/PipedPlayList.txt" ),
				new File( "/Kids/" ) ); 
		}
		catch( Exception ex )
		{ 
			System.out.println( ex.getMessage() );
			return;
		}
		
		list = factory3.getPlayList( );
		for( int i=0; i<list.length; i++ )
		{
			System.out.println( list[i] );
		}
		
		System.out.println( "BasicPageFactory TEST PROGRAM EXITING NORMALLY!" );
	} // main
	
} // class BasicPageFactory

