/*
 * Copyright (c) 2010 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.gerhardb.jibs.textPad;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.ComponentOrientation;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;
import java.util.prefs.Preferences;
import javax.swing.*;
import javax.swing.tree.TreePath;

import org.gerhardb.jibs.*;
import org.gerhardb.lib.dirtree.ExtendedDirectoryTreePopUp;
import org.gerhardb.lib.dirtree.rdp.BasicOptionsManager;
import org.gerhardb.lib.dirtree.rdp.IListShowTree;
import org.gerhardb.lib.dirtree.rdp.ILoad;
import org.gerhardb.lib.dirtree.rdp.ListShowTreeCoordinator;
import org.gerhardb.lib.dirtree.rdp.PathManager;
import org.gerhardb.lib.io.FilenameFileFilter;
import org.gerhardb.lib.scroller.ScrollerChangeEvent;
import org.gerhardb.lib.scroller.ScrollerListener;
import org.gerhardb.lib.scroller.ScrollerSlider;
import org.gerhardb.lib.swing.JFileChooserExtra;
import org.gerhardb.lib.swing.SwingUtils;
import org.gerhardb.lib.swing.UIMenu;
import org.gerhardb.lib.util.ActionHelpers;
import org.gerhardb.lib.util.Icons;
import org.gerhardb.lib.util.app.InfoPlugins;
import org.gerhardb.lib.util.startup.AppPluginStartup;
import org.gerhardb.lib.util.startup.AppStarter;
import org.gerhardb.lib.util.startup.Loading;

/**
 * Info Display.
 */
public class TextPad extends JFrame implements ScrollerListener, IListShowTree, InfoPlugins, ILoad
{
	private static final String LAST_FILE = "LastFile";  //$NON-NLS-1$
	private static final String[] FILE_NAME_ENDINGS = new String[]{".txt",".htm",".html"};
	
	static final String COUNT_TEXT_FILES = "COUNT_TEXT_FILES";
	
	private static final String FILE_PANEL_SIZE = "ListPicturePanelDividerLocation";  //$NON-NLS-1$
	private static final String TREE_PANEL_SIZE = "TreePanelSize";  //$NON-NLS-1$
	private static final String FONT_SIZE = "font_size";  //$NON-NLS-1$
	public static final int TREE_MIN = 100;
	public static final int FILE_MIN = 1000;
	
	static final Preferences clsPrefs = Preferences.userRoot().node(
			"/org/gerhardb/jibs/textPad/TextPad");  //$NON-NLS-1$

	JEditorPane myEditorPane = new JEditorPane();
	private JSplitPane myListPanel = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);	
	private JSplitPane myTreePanel = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);
	TextPadActions myActions;
	JToolBar myToolBar = new JToolBar();
	private ScrollerSlider mySlider;
	UIMenu myUImenu = new UIMenu(this, getAppIDandTargetType());
	private JComboBox myFontSize = new JComboBox(new String[]{"10", "12", "14", "16", "18", "20", "22", "24", "26", "28", "32", "48", "64", "72", "80", "96", "128"});
	
	boolean iExitOnClose;
	
	// Used because you could get two exit messages from the exit hooks.
	boolean iNeedToFinalize = true; 
	
	ListShowTreeCoordinator myLST;	
	
	Loading myLoading = new Loading(this, "Text Pad");
	
	/*
	 * =========================================================================
	 *                         Constructor
	 * =========================================================================
	 */

	/**
	 * Prints basic Information Report.
	 */
	public TextPad(boolean exitOnClose)
	{
		this.iExitOnClose = exitOnClose;
		
		this.myLST = new ListShowTreeCoordinator(this);
		this.myLST.init(this);		 
		this.myActions = new TextPadActions(this);	
		this.myLST.addActions(this.myActions);
		
		buildAndDisplayInterface();			
		
		File loadFile = new File(this.myLST.getPathManager().getDirectoryAbsolute(PathManager.DIR_TREE));
		this.myLST.load(loadFile, this.myLoading, this, Thread.currentThread());
	}
	
	public ListShowTreeCoordinator getLST()
	{
		return this.myLST;
	}
	
	// ==========================================================================
	// InfoPlugins
	// ==========================================================================
	public void info(StringBuffer sb)
	{		
		sb.append("From TextPad"); 
	}
	
	// ==========================================================================
	// IListShowTree
	// ==========================================================================
		public String getAppIDandTargetType()
		{
			return "Text";
		}
		
		public JFrame getTopFrame()
		{
			return this;
		}
		
		public ScrollerListener getScrollerListener()
		{
			return this;
		}
		
		public FilenameFileFilter getFilenameFileFilter()
		{
			return new FilenameFileFilter(FILE_NAME_ENDINGS);
		}
		
		public boolean getCountTargetFiles()
		{
			return clsPrefs.getBoolean(COUNT_TEXT_FILES, true);
		}
		
		// equal to showImage() in Viewer.
		public void showFile(File file)
		{
			if (file == null || !file.exists())
			{
				return;
			}
			String url = file.toURI().toString();
			//System.out.println("TextPad File: " + file);
			//System.out.println("TextPad URL: " + url);
			try
			{
				this.myEditorPane.setPage(url);
				this.myEditorPane.setCaretPosition(0);
			}
			catch (Exception ex)
			{
				System.out.println("Can't find file: " + url);
				ex.printStackTrace();
			}			
		}

		public JPopupMenu getPopupMenu(TreePath path)
		{
			return new ExtendedDirectoryTreePopUp(this.myLST.getTree(), path, getCountTargetFiles());
		}

	// ==========================================================================
	// ScrollerListener
	// ==========================================================================
	public void scrollerChanged(ScrollerChangeEvent ce)
	{
		showFile(this.myLST.getScroller().getCurrentFile());
		/*
		// These are just here in case I decide I need to do something different
		// based on event type.  As of Sept 2010, I don't think its needed.
		switch (ce.getEventType())
		{
		case ScrollerChangeEvent.CURRENT_IMAGE_CHANGED:
			if (!ViewerGlobals.gblScroller.getValueIsAdjusting())
			{
				showImage();
			}
			break;
		case ScrollerChangeEvent.CURRENT_PAGE_CHANGED:
			if (!ViewerGlobals.gblScroller.getValueIsAdjusting())
			{
				showPageFromScroller();
			}
			break;
		case ScrollerChangeEvent.ONE_FILE_REMOVED:
      case ScrollerChangeEvent.LIST_RELOADED:
			if (ViewerGlobals.gblScroller.getMaximumZeroBased() < 0)
			{
				repaint();
			}
			showPageFromScroller();
			break;
		}
		*/
	}
	
	/**
	 * Put setVisable on the AWT Event Queue to avoid following error message:
	 * Exception in thread "AWT-EventQueue-0" java.lang.NullPointerException
	 * at javax.swing.plaf.basic.BasicSliderUI.calculateTrackBuffer(Unknown Source)
	 *
	 */
	public void awtComplete()
	{
		showDefaultDirectory();
		
		this.myLoading.dispose();
		this.myLoading = null;

		System.out.println("\n" //$NON-NLS-1$
				+ "**************************************************** \n" //$NON-NLS-1$
				+ "*         JIBS Background Population Ended         * \n" //$NON-NLS-1$
				+ "**************************************************** \n"); //$NON-NLS-1$
		
	}

	// ==========================================================================
	// BUILD THE DISPLAY
	// ==========================================================================
	private void buildAndDisplayInterface()
	{		
		addWindowListener(new WindowAdapter()
		{
			@Override
			public void windowClosing(WindowEvent e)
			{
				exit();
			}
		});
		this.setIconImage(Icons.getIcon(Icons.JIBS_16).getImage());		
		
		setupToolBar();
		
		// Set the three key elements.
		this.mySlider = new ScrollerSlider(SwingConstants.VERTICAL, this.myLST.getScroller());
		this.mySlider.setInverted(true);
		this.mySlider.applyComponentOrientation(ComponentOrientation.RIGHT_TO_LEFT);
		
		Dimension size = new Dimension(700, 400);
		setSize(size);

		setTitle("JIBS TextPad"); //$NON-NLS-1$

		this.myEditorPane.setEditable(false);
		this.myEditorPane.setDisabledTextColor(Color.black);
		this.myEditorPane.setEditorKit(JEditorPane
				.createEditorKitForContentType("text/html")); //$NON-NLS-1$

		JScrollPane center = new JScrollPane(this.myEditorPane,
				ScrollPaneConstants.VERTICAL_SCROLLBAR_AS_NEEDED,
				ScrollPaneConstants.HORIZONTAL_SCROLLBAR_NEVER);

		this.myTreePanel.setLeftComponent(center);
		this.myTreePanel.setRightComponent(new JScrollPane(this.myLST.getExtendedDirectoryTree()));
		
		JPanel sliders = new JPanel(new BorderLayout());
		sliders.add(new JScrollPane(this.myLST.getFileList()), BorderLayout.CENTER);
		sliders.add(new JScrollPane(this.mySlider), BorderLayout.WEST);
		
		this.myListPanel.setLeftComponent(sliders);
		this.myListPanel.setRightComponent(this.myTreePanel);
		
		getContentPane().setLayout(new BorderLayout());
		getContentPane().add(this.myToolBar, BorderLayout.NORTH);
		getContentPane().add(this.myListPanel, BorderLayout.CENTER);
		getContentPane().add(this.myLST.getStatusBarManager().getStatusPanel(), BorderLayout.SOUTH);		
		
		this.setJMenuBar(this.makeMenus());
		this.myUImenu.initLookAndFeel();
		
		this.updateFont();
		
		// Size to 75% of screen.
		SwingUtils.sizeScreen(this, .75f);
		// Do after sizing (pack or SwingUtils.sizeScreen)...
		validate();

		// Based on sizing, so do afterwards
		this.myListPanel.setDividerLocation(clsPrefs.getInt(FILE_PANEL_SIZE, FILE_MIN));
		this.myTreePanel.setDividerLocation(clsPrefs.getInt(TREE_PANEL_SIZE, TREE_MIN));

		SwingUtils.centerOnScreen(this); // seems to need to be here also
		
		super.setVisible(true);		
	}
	
	void setupToolBar()
	{
		//myToolBar.add(myActions.makeSaveButton());
		//myToolBar.add(myActions.getToolBarButton("file", "saveas")); //$NON-NLS-1$ //$NON-NLS-2$
		//myToolBar.add(myActions.getToolBarButton("file", "rename")); //$NON-NLS-1$ //$NON-NLS-2$
		//myToolBar.addSeparator();

		this.myToolBar.add(this.myLST.getRDPmanager().getMoveManager().getUndoButton()); 
		this.myToolBar.add(this.myLST.getRDPmanager().getMoveManager().getRedoButton());
		this.myToolBar.addSeparator();

		this.myLST.getFileListManager().addButtonsToToolBar(this.myToolBar, this.myActions);
		this.myToolBar.add(this.myActions.getToolBarButton("view", "reload")); //$NON-NLS-1$ //$NON-NLS-2$
		this.myToolBar.addSeparator();

		this.myToolBar.add(new JLabel("Font Size  ")); 
		//myFontSize.setEditable(false);
		this.myFontSize.setSelectedItem(clsPrefs.get(FONT_SIZE, "12"));
		this.myFontSize.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent ae)
			{
				TextPad.this.updateFont();
			}
		});
		this.myFontSize.setMaximumSize(new Dimension(20,20));
		this.myToolBar.add(this.myFontSize); 
		this.myToolBar.addSeparator();		
		
		// BIG GAP - to stuff on the right
		// Separator is needed when the toolbar is undocked.
		this.myToolBar.addSeparator();
		this.myToolBar.add(javax.swing.Box.createHorizontalGlue());

		this.myLST.getRDPmanager().addButtonsToToolbar(this.myToolBar);
		this.myToolBar.addSeparator();

		this.myToolBar.add(this.myActions.getToolBarButton("help", "about")); //$NON-NLS-1$ //$NON-NLS-2$
		this.myToolBar.add(this.myActions.getToolBarButton("help", "help")); //$NON-NLS-1$ //$NON-NLS-2$
	}
	
	void exit()
	{
		if (this.iNeedToFinalize)
		{
			this.iNeedToFinalize = false;
			
			clsPrefs.putInt(FILE_PANEL_SIZE, this.myListPanel.getDividerLocation());
			clsPrefs.putInt(TREE_PANEL_SIZE, this.myTreePanel.getDividerLocation());
			clsPrefs.put(FONT_SIZE, (String)this.myFontSize.getSelectedItem());
			try
			{
				clsPrefs.flush();
			}
			catch(Exception ex)
			{
				ex.printStackTrace();
			}
			
			this.myLST.storeDirectories();
			
			this.myLST.getPathManager().flush();
			
			System.out.println("" //$NON-NLS-1$
					+ "**************************************************** \n" //$NON-NLS-1$
					+ "*           JIBS Text Pad Exited Normally          * \n" //$NON-NLS-1$
					+ "**************************************************** \n"); //$NON-NLS-1$
			System.out.println(AppPluginStartup.JIBS_OUT); //$NON-NLS-1$
			
			AppStarter.closeDownApp();
			if (this.iExitOnClose)
			{
				System.exit(0);
			}
			else
			{
				// TODO: May want to kill any running worker thread.
				// Would need to track in here and manage from the actual workers
				// adding an interupt reutine to the workers.
				this.setVisible(false);
				dispose();
			}
		}
	}

	void open()
	{
		String fileName = this.myLST.getTreeManager().getDTNReaderWriter().getFileName(); 
		JFileChooserExtra chooser = new JFileChooserExtra(clsPrefs.get(LAST_FILE, null));
		chooser.setSaveName(getAppIDandTargetType(), fileName); 
		chooser.setDialogTitle(Jibs.getString("RecreateDirectories.20") + fileName);  //$NON-NLS-1$
		chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
		chooser.setMultiSelectionEnabled(false);
		//chooser.addChoosableFileFilter(new JibsTreeFileFilter());  /////////////////////??????????????????????????????????????????////////////////////////////////////////////

		int returnVal = chooser.showOpenDialog(this);
		if (returnVal == JFileChooser.APPROVE_OPTION)
		{
			File picked = chooser.getSelectedFile();
			if (picked == null || !picked.exists())
			{
				return;
			}
			
			try
			{
				clsPrefs.put(LAST_FILE, picked.toString());
				clsPrefs.flush();
			}
			catch (Exception ex)
			{
				ex.printStackTrace();
			}			
			
			showFile(picked);
		}
	}
		
	JMenuBar makeMenus()
	{
		JMenu toolsMenu = new JMenu("Tools");
		class OptionsAction extends AbstractAction
		{
			public void actionPerformed(ActionEvent e)
			{
				new TextPadOptionsDialog(TextPad.this);
			}
		}
		JMenuItem options = new JMenuItem("Options");
		toolsMenu.add(options);
		options.addActionListener(new OptionsAction());
		
		JMenuBar menuBar = new JMenuBar();
		menuBar.add(this.myActions.getFileMenu());
		menuBar.add(this.myActions.getEditMenu());
		menuBar.add(this.myLST.getSortManager().getSortMenu());
		menuBar.add(this.myActions.getViewMenu());
		menuBar.add(toolsMenu);
		menuBar.add(this.myUImenu.populateUIMenu(ActionHelpers.makeMenu("ui"))); //$NON-NLS-1$
		menuBar.add( Box.createHorizontalGlue() );
		menuBar.add( this.myActions.getHelpMenu() ); 
		return menuBar;
	}	

	void updateFont()
	{
		int fontSize = 12;
		try
		{
			fontSize = Integer.parseInt((String)this.myFontSize.getSelectedItem());
			Font font = new Font("Serif", Font.PLAIN, fontSize);
			this.myEditorPane.setFont(font);
		}
		catch(Exception ex)
		{
			this.myFontSize.setSelectedItem("12");
		}
	}
	
	/**
	 * Attempt to use the Java Print Service.
	 * Unfortunately, the server printable produces same output.
	 * http://java.sun.com/j2se/1.4.1/docs/guide/jps/spec/JPSTOC.fm.html
	 */
	/*
	public void printWithService()  /////////////////////////////////////////////////// make generic??
	{
		Printable printable = new PrintableText(myEditorPane);
		DocFlavor flavor = DocFlavor.SERVICE_FORMATTED.PRINTABLE;

		PrintRequestAttributeSet pras = new HashPrintRequestAttributeSet();
		//pras.add(new Copies(1));
		//pras.add(MediaSize.getMediaSizeForName(MediaSizeName.ISO_C6));
		//pras.add(Sides.DUPLEX);
		PrintService[] allServices = PrintServiceLookup.lookupPrintServices(
				flavor, pras);
		if (allServices.length == 0)
		{
			// TO DO - INFORM USER!
			System.out.println("No print services found"); //$NON-NLS-1$
			return;
		}
		PrintService defaultService = PrintServiceLookup
				.lookupDefaultPrintService();
		PrintService service = ServiceUI.printDialog(null, 200, 200, allServices,
				defaultService, flavor, pras);
		if (service == null)
		{
			System.out.println("User cancelled print"); //$NON-NLS-1$
			return;
		}
		DocPrintJob job = service.createPrintJob();
		DocAttributeSet das = new HashDocAttributeSet();
		Doc doc = new SimpleDoc(printable, flavor, das);
		try
		{
			job.print(doc, pras);
		}
		catch (PrintException pe)
		{
			pe.printStackTrace();
		}
	}
*/
	private void showDefaultDirectory()
	{
		// Decide what directory to start with
		// AFTER we have decided the size so the picture does not influences
		// size.

		String dir = null;
		BasicOptionsManager bom = this.myLST.getBasicOptionsManager();
		switch (bom.getStartType())
		{
		case BasicOptionsManager.START_PARTICULAR_DIRECTORY:
			dir = this.myLST.getPathManager().getDirectoryAbsolute(PathManager.DIR_START);
			try
			{
				this.myLST.getFileListManager().setTargetDir(this.myLST.getPathManager().getDirectoryAbsolute(PathManager.DIR_START));
			}
			catch (Exception ex)
			{
				ex.printStackTrace();
			}
			break;
		case BasicOptionsManager.START_LAST_VIEWED_DIRECTORY:
			dir = this.myLST.getPathManager().getDirectoryAbsolute(PathManager.DIR_LAST_VIEWED);
			break;
		case BasicOptionsManager.START_NO_DIRECTORY:
		default:
			break;
		}
		
		try
		{
			if (dir != null)
			{
				this.myLST.getFileListManager().setTargetDir(dir);
			}
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
		}
	}
	
	// ==========================================================================
	// Main
	// ==========================================================================
	public static void main(String[] args)
	{		
		// "org.gerhardb.jibs.Jibs" is correct for bundleName,
		// it gets expanded out to: org.gerhardb.jibs.Jibs.properties
		// or whatever the locale version is (DE for Germany etc.)
		AppStarter.startUpApp(args, "org.gerhardb.jibs.Jibs", true);

		new TextPad(true);
	}
	
}
