/*
 * Copyright (c) 2007 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.gerhardb.jibs.textPad.rankPad;

import java.awt.Component;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.InputEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.File;
import java.io.FileFilter;
import java.util.Enumeration;

import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.tree.*;

import org.gerhardb.lib.dirtree.DirectoryTree;
import org.gerhardb.lib.dirtree.DirectoryTreeNode;
import org.gerhardb.lib.io.FileUtil;
import org.gerhardb.lib.util.startup.AppStarter;

/**
 * The Ranked List Of Files
 */
public class RankDirectoryTree extends DirectoryTree
{
	RankPad myRankPad;
	
	RankDirectoryTree(RankPad rankPad, FileFilter filter, boolean countHiddenDirectories)
	{
		super(rankPad, filter, countHiddenDirectories);
		super.setCountTargetFiles(false);
		super.addMouseListener(new MouseAdpt());
		this.myRankPad = rankPad;
	}
	
	void fileDeleted(File file)
	{
		String parentPath = file.getParentFile().getAbsolutePath();
		fileDeleted(file, this.getRootNode(), parentPath);
	}
	
	boolean fileDeleted(File file, DirectoryTreeNode node, String parentPath)
	{
		 Enumeration<?> kids = node.children();
		 while(kids.hasMoreElements())
		 {
			 DirectoryTreeNode child = (DirectoryTreeNode)kids.nextElement();
			 if (parentPath.startsWith(child.getFile().getAbsolutePath()))
			 {
				 if (parentPath.equals(child.getFile().getAbsolutePath()))
				 {
					 child.decrementFileCount(file.length());
					 return true;
				 }
				 if (fileDeleted(file, child, parentPath))
				 {
					 return true;
				 }
			 }
		 }
		 return false;
	}
	
	// --------------------------------------------------------------------------
	// Inner Class: MouseAdapter
	// --------------------------------------------------------------------------
	class MouseAdpt extends MouseAdapter
	{
		TreePath			selPath	= null;

		@Override
		public void mouseReleased(MouseEvent e)
		{
			int selRow = RankDirectoryTree.this.getRowForLocation(e.getX(), e.getY());
			if (selRow < 0)
			{
				return;
			}

			// We only set the tree on the first click since the model
			// may shift before the second click comes in.
			if (e.getClickCount() == 1)
			{
				this.selPath = RankDirectoryTree.this.getPathForLocation(e.getX(), e.getY());
			}

			// Single right click is a pop up.
			if (((e.getModifiers() & InputEvent.BUTTON2_MASK) != 0)
					|| ((e.getModifiers() & InputEvent.BUTTON3_MASK) != 0))
			{
				// We worry about where the click is because the selection will
				// not change as a result of picking the item.
				new TreePopUp(this.selPath).show((Component) e.getSource(), e.getX(), e.getY());
				return;
			}
			super.mouseReleased(e);
		}
	}

	private class TreePopUp
   extends JPopupMenu
	{
	   DirectoryTreeNode myDirectoryNode = null;
		protected TreePath myTreePath = null;
	
	   // --------------------------------------------------------------------------
	   //                           Constructor
	   // --------------------------------------------------------------------------
	   /**
	    * @param sortScreen popped up on
	    * @param path path
	    */
	   public TreePopUp(TreePath path)
	   {
	      super();
	      this.myTreePath = path;
	
	      System.out.println("FileTreePopUp Path: " + path);
	
	      Object[] selPath = this.myTreePath.getPath();
	      Object node = selPath[selPath.length - 1];
	      if (node instanceof DirectoryTreeNode)
	      {
	         this.myDirectoryNode = (DirectoryTreeNode)node;
	      }
	
	      JPanel center = new JPanel(new FlowLayout(FlowLayout.CENTER));
	
	      if (this.myDirectoryNode == null)
	      {
	         center.add(new JLabel(AppStarter.getString("FileTreePopUp.0")), 0); //$NON-NLS-1$
	      }
	      else
	      {
	         center.add(new JLabel(this.myDirectoryNode.getName()), 0);
	      }
	
	      insert(center, 0);
	      addSeparator();
	
	      JMenuItem moveFile = add("Move current file to here");
	      this.addSeparator();
	      JMenuItem reloadTree = add(AppStarter.getString("FileTreePopUp.4")); //$NON-NLS-1$
	      this.addSeparator();
	      JMenuItem newDirItem = add(AppStarter.getString("FileTreePopUp.5")); //$NON-NLS-1$
	      JMenuItem renameDirItem = add(AppStarter.getString("FileTreePopUp.6")); //$NON-NLS-1$
	      JMenuItem deleteDirItem = add(AppStarter.getString("FileTreePopUp.7")); //$NON-NLS-1$
	      JMenuItem deleteEmptyDirectories = add("Delete EmptyDirectories");
	      deleteDirItem.setToolTipText("Deletes all empyt directorys found in this tree");
	
	      if (this.myDirectoryNode == null)
	      {
	      	moveFile.setEnabled(false);
	         newDirItem.setEnabled(false);
	         renameDirItem.setEnabled(false);
	         deleteDirItem.setEnabled(false);
	         deleteEmptyDirectories.setEnabled(false);
	         return;
	      }
	
	      reloadTree.setEnabled(true);
	
	      File[] kids = this.myDirectoryNode.getFile().listFiles();
	      if (kids != null && kids.length > 0)
	      {
	         deleteDirItem.setEnabled(false);
	         deleteDirItem.setToolTipText(
	            AppStarter.getString("FileTreePopUp.11")); //$NON-NLS-1$
	      }      	
	
	      reloadTree.addActionListener(
	         new ActionListener()
	      {
	         public void actionPerformed(ActionEvent event)
	         {
	         	TreePopUp.this.myDirectoryNode.reloadNodes();
	         }
	      });
	
	      moveFile.addActionListener(
		         new ActionListener()
			      {
			         public void actionPerformed(ActionEvent event)
			         {
			         	moveFile(TreePopUp.this.myDirectoryNode);
			         }
			      });
	
	      newDirItem.addActionListener(
	         new ActionListener()
	      {
	         public void actionPerformed(ActionEvent event)
	         {
	            TreePopUp.this.myDirectoryNode.newSubDir();
	         }
	      });
	
	      renameDirItem.addActionListener(
	         new ActionListener()
	      {
	         public void actionPerformed(ActionEvent event)
	         {
	            TreePopUp.this.myDirectoryNode.renameDir();
	         }
	      });
	
	      deleteDirItem.addActionListener(
	         new ActionListener()
	      {
	         public void actionPerformed(ActionEvent event)
	         {
	            TreePopUp.this.myDirectoryNode.deleteDir();
	         }
	      });
	      
	      deleteEmptyDirectories.addActionListener(
	            new ActionListener()
	            {
	               public void actionPerformed(ActionEvent event)
	               {
	                  TreePopUp.this.myDirectoryNode.deleteEmptyDirectories();
	               }
	            });
	   }
	}

	void moveFile(DirectoryTreeNode directoryNode)
	{
		if (directoryNode == null)
		{
			return;
		}
		System.out.println("moveFile pressed");
		File moveMe = this.myRankPad.rankList.getSelectedFile();
		if (moveMe.exists())
		{
			try
			{
				System.out.println("moving: " + moveMe);
				File newFile = FileUtil.moveFile(directoryNode.getDirectory(), moveMe);			
				System.out.println("to: " + newFile);
				
				// Change the item in the list and save.
				RankFile item = (RankFile)this.myRankPad.rankList.getSelectedValue();
				System.out.println("item: " + item);
				System.out.println("newFile.getAbsolutePath(): " + newFile.getAbsolutePath());
				item.setFile(newFile);
				this.myRankPad.rankList.saveFile();

				// Update recent file
				this.myRankPad.iAmAdjustingComboBoxSelection = true;
				this.myRankPad.myRecentFiles.removeElement(new RecentFile(moveMe,0));	
				this.myRankPad.myRecentFiles.insertElementAt(new RecentFile(newFile,0), 0);		
				this.myRankPad.iAmAdjustingComboBoxSelection = false;
				
				// Remove from tree.
				this.myRankPad.myTree.fileDeleted(moveMe);
				
				// Add to this node.
				directoryNode.incrementFileCount(moveMe.length());
			}
			catch (Exception ex)
			{
				ex.printStackTrace();
			}
		}
	}
}


