/*
 * Copyright (c) 2010 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.gerhardb.jibs.textPad.rankPad;

import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.*;
import java.util.prefs.Preferences;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.*;
import javax.swing.text.html.HTMLEditorKit;
import javax.swing.text.rtf.RTFEditorKit;

import org.gerhardb.jibs.Jibs;
import org.gerhardb.jibs.viewer.ViewerPreferences;
import org.gerhardb.lib.dirtree.*;
import org.gerhardb.lib.dirtree.rdp.ILoad;
import org.gerhardb.lib.dirtree.rdp.PathManager;
import org.gerhardb.lib.io.FileFileFilter;
import org.gerhardb.lib.io.FileUtil;
import org.gerhardb.lib.io.FilenameFileFilter;
import org.gerhardb.lib.swing.JFileChooserExtra;
import org.gerhardb.lib.swing.SwingUtils;
import org.gerhardb.lib.swing.UIMenu;
import org.gerhardb.lib.util.ActionHelpers;
import org.gerhardb.lib.util.Icons;
import org.gerhardb.lib.util.app.InfoPlugins;
import org.gerhardb.lib.util.startup.ActiveActions;
import org.gerhardb.lib.util.startup.AppPluginStartup;
import org.gerhardb.lib.util.startup.AppStarter;
import org.gerhardb.lib.util.startup.Loading;

/**
 * RankPad
 */
public class RankPad extends JFrame implements InfoPlugins, ILoad
{
   /** Should be kept in sync with javax.swing.text.html.FormView counterpart. */
   //final static String PostDataProperty = "javax.swing.JEditorPane.postdata";

	final static int URL = 0;
	final static int HTML = 1;
	final static int TEXT = 2;
	
	static final Preferences clsPrefs = Preferences.userRoot().node(
	"/org/gerhardb/jibs/textPad/RankPad");  //$NON-NLS-1$

	private static final String[] FILE_NAME_ENDINGS = new String[]{".txt",".htm",".html"};	
	static final String COUNT_TEXT_FILES = "count_text_files";
	static final String DELETE_DIRECTORY = "delete_directory";
	
	private static final String FILE_PANEL_SIZE = "ListPicturePanelDividerLocation";  //$NON-NLS-1$
	private static final String TREE_PANEL_SIZE = "TreePanelSize";  //$NON-NLS-1$
	private static final String FONT_SIZE = "font_size";  //$NON-NLS-1$
	static final String RANK_FILE = "rank_file";  //$NON-NLS-1$
	private static final String TREE_DIRECTORY = "tree_directory";  //$NON-NLS-1$
	public static final int TREE_MIN = 100;
	public static final int FILE_MIN = 400;
	
	private JSplitPane myListEditorTreePanel = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);	
	private JSplitPane myEditorTreePanel = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);
	RankPadActions myActions;
	JToolBar myToolBar = new JToolBar();
	UIMenu myUImenu = new UIMenu(this, getAppIDandTargetType());
	private JPanel myCenterPanel;
	private JComboBox myFontSize = new JComboBox(new String[]{"10", "12", "14", "16", "18", "20", "22", "24", "26", "28", "32", "48", "64", "72", "80", "96", "128"});
	JLabel myStatusCount = new JLabel();
	JTextField myStatusFileName = new JTextField();
	boolean iExitOnClose;
	RankDirectoryTree myTree;
	TreeManager myTreeManager;
	File myDeleteDirectory;
	DefaultEditorKit myDeafultEditorKit = new DefaultEditorKit();
	HTMLEditorKit myHTMLEditorKit = new HTMLEditorKit();
	RTFEditorKit myRTFEditorKit = new RTFEditorKit();
	//boolean iCanDelete = true;
	DTNReaderWriter myTreeReaderWriter;
	JScrollPane myEditorScrollPane;
	Action deleteMenuAction;
	Action deleteToolbarAction;
	JTabbedPane rankAndFile = new JTabbedPane();
	RankList rankList = new RankList(this);
	DirectoryList myDirectoryList = new DirectoryList(this);
	
	// Used because you could get two exit messages from the exit hooks.
	boolean iNeedToFinalize = true; 

	//JEditorPane myEditorPane = new JEditorPane();
	JTextPane myEditorPane = new JTextPane();

	boolean iAmAdjustingComboBoxSelection = false;
	DefaultComboBoxModel myRecentFiles = new DefaultComboBoxModel();
	
	/*
	 * =========================================================================
	 *                         Constructor
	 * =========================================================================
	 */

	/**
	 * Prints basic Information Report.
	 */
	public RankPad(boolean exitOnClose)
	{
		this.iExitOnClose = exitOnClose;
		Loading loading = new Loading(this, "Rank Pad");		
		
		this.myActions = new RankPadActions(this);	
		
		File rootDirectoryForTree = new File(clsPrefs.get(TREE_DIRECTORY, System.getProperty("user.home"))); 
		setTree(rootDirectoryForTree, loading);
		
		this.myDeleteDirectory = getStoredDeleteDirectory();
		System.out.println("myDeleteDirectory " + this.myDeleteDirectory);
		this.rankList.populate(clsPrefs.get(RANK_FILE, ""));
		
		buildAndDisplayInterface();
		
		// Get previously viewed files
		RankPad.this.iAmAdjustingComboBoxSelection = true;
		try
		{
			BufferedReader buffRead = new BufferedReader(new FileReader(this.rankList.viewHistoryFile));
		   String aLine = buffRead.readLine();
			while (aLine != null)
			{
				//System.out.println("Read in: " + aLine);
				String[] splitRF = aLine.split("\t");
				if (splitRF.length > 1)
				{
					try
					{
						int y = Integer.parseInt(splitRF[0]);
						RecentFile rf = new RecentFile(new File(splitRF[1]), y);
						//System.out.println("RF: " + rf);
						this.myRecentFiles.addElement(rf);
					}
					catch(Exception ex)
					{
						ex.printStackTrace();
					}
				}
				aLine = buffRead.readLine();
			}
			buffRead.close();
		}
		catch(Exception ex)
		{
			ex.printStackTrace();
		}
		RankPad.this.iAmAdjustingComboBoxSelection = false;
				
		RecentFile rfToShow = null;
		if (this.myRecentFiles.getSize() > 0)
		{
			rfToShow = (RecentFile)this.myRecentFiles.getElementAt(0);
		}
		this.rankList.showRecnetFile(rfToShow);
		
		// In case I want to thread later.
		loading.dispose();
		System.out.println("\n" //$NON-NLS-1$
				+ "**************************************************** \n" //$NON-NLS-1$
				+ "*         JIBS Background Population Ended         * \n" //$NON-NLS-1$
				+ "**************************************************** \n"); //$NON-NLS-1$
	}
	
	// ==========================================================================
	// InfoPlugins
	// ==========================================================================
	public void info(StringBuffer sb)
	{		
		sb.append("From RankPad"); 
	}
	
	// ==========================================================================
	// IListShowTree
	// ==========================================================================
		public String getAppIDandTargetType()
		{
			return "Text";
		}
		
		public JFrame getTopFrame()
		{
			return this;
		}
		
		public FilenameFileFilter getFilenameFileFilter()
		{
			return new FilenameFileFilter(FILE_NAME_ENDINGS);
		}
		
		public boolean getCountTargetFiles()
		{
			return clsPrefs.getBoolean(COUNT_TEXT_FILES, true);
		}
		
		public File getStoredDeleteDirectory()
		{
			String defaultDir = AppStarter.getMyDocumentsDir() + ViewerPreferences.JIBS_DIR_NAME + "/" + "RankPadDelete";
			String storedDir = clsPrefs.get(DELETE_DIRECTORY, defaultDir);
			File rtnMe = new File(storedDir); 
			if (rtnMe.exists() && rtnMe.isDirectory())
			{
				return rtnMe;
			}
			else if (defaultDir.equals(storedDir))
			{
				try
				{
					if (rtnMe.mkdir())
					{
						return rtnMe;
					}
				}
				catch(Exception ex)
				{
					// 
				}
			}
			return null;
		}
		
	// equal to showImage() in Viewer.
	public void showFile(File file, final int scrollDocumentTextTo)
	{
		if (file == null)
		{ 
			return;
		}

		if (!file.exists())
		{ 
			this.myEditorPane.setText("Could not find file: " + file.getAbsolutePath());
			return;
		}
		
		this.myStatusFileName.setText(file.getAbsolutePath());
		
		// Remove from present position in list if any.
		RecentFile rf = new RecentFile(file, scrollDocumentTextTo);
		this.myRecentFiles.removeElement(rf);
		// Now remember new one.
		this.myRecentFiles.insertElementAt(rf, 0);
		// Update because we updated model from underneath...
		// But don't trigger event because we are in middle of processing a 
		// selection change.
		this.iAmAdjustingComboBoxSelection = true;
		this.myRecentFiles.setSelectedItem(rf);
		this.iAmAdjustingComboBoxSelection = false;
		
		String url = file.toURI().toString();
		
		//System.out.println("RankPad File: " + file);
		//System.out.println("RankPad URL: " + url);
		
		int type = URL;
		String lowerCase = url.toLowerCase();
		if (lowerCase.endsWith("htm") || lowerCase.endsWith("html"))
		{
			type = HTML;
		}
		else if (lowerCase.endsWith("txt"))
		{
			type = TEXT;
		}
		
		try
		{
			switch (type)
			{
			case URL:
				//this.iCanDelete = false;
				setDelete(false);
				// This works with formatting, but keeps file open so you can't delete.
				this.myEditorPane.setPage(url);
				break;
			case HTML:
				//this.iCanDelete = true;
				setDelete(true);
				// This works fine, but no formatting for text documents.
				BufferedInputStream in = new BufferedInputStream(
						new FileInputStream(file));
				// No joy for text doc formatting, html fine.
				this.myEditorPane.setEditorKit(JEditorPane.createEditorKitForContentType("text/html")); //$NON-NLS-1$
				this.myEditorPane.read(in, this.myHTMLEditorKit.createDefaultDocument());
				// No joy for text doc formating, html doesn't even show!
				//this.myEditorPane.read(in, this.myRTFEditorKit.createDefaultDocument());
				// No joy for text doc formating, html doesn't even show!
				//this.myEditorPane.read(in, this.myDeafultEditorKit.createDefaultDocument());
				//this.myEditorPane.read(in, null);
				in.close();
				break;
			case TEXT:
				//this.iCanDelete = true;
				setDelete(true);
				this.myEditorPane.setEditorKit(JEditorPane.createEditorKitForContentType("text/html")); //$NON-NLS-1$
				String theText = "foo";
				theText = FileUtil.getStringFromFile(file);
				Style def = StyleContext.getDefaultStyleContext().getStyle(StyleContext.DEFAULT_STYLE);
				//StyledDocument doc = new DefaultStyledDocument(); //javax.swing.text.DefaultStyledDocument cannot be cast to javax.swing.text.html.HTMLDocument
				StyledDocument doc = this.myEditorPane.getStyledDocument();
				//StyledDocument doc = this.myHTMLEditorKit.createDefaultDocument();
				doc.remove(0, doc.getLength());
				doc.addStyle("regular", def);
				StyleConstants.setFontFamily(def, "SansSerif");
				this.myEditorPane.setDocument(doc);
				try
				{
					doc.insertString(
							doc.getLength(), 
							theText, 
							doc.getStyle("regular"));
				}
				catch (BadLocationException ble)
				{
					System.err
							.println("Couldn't insert initial text into text pane.");
				}
				break;
			}
			this.myEditorPane.setCaretPosition(0);
			
			this.myEditorScrollPane.getViewport().setViewPosition(new Point(0, scrollDocumentTextTo));		
		     EventQueue.invokeLater(new Runnable()
		      {
		         public void run()
		         {
		      		RankPad.this.myEditorScrollPane.getViewport().setViewPosition(new Point(0, scrollDocumentTextTo));		
		         }
		      });
		}
		catch (Exception ex)
		{
			System.out.println("Can't find file: " + url);
			ex.printStackTrace();
		}
	}

	/**
	 * Put setVisable on the AWT Event Queue to avoid following error message:
	 * Exception in thread "AWT-EventQueue-0" java.lang.NullPointerException
	 * at javax.swing.plaf.basic.BasicSliderUI.calculateTrackBuffer(Unknown Source)
	 *
	 */
	public void awtComplete()
	{
		// Needed for ILoad.
	}

	// ==========================================================================
	// BUILD THE DISPLAY
	// ==========================================================================
	private void buildAndDisplayInterface()
	{		
		addWindowListener(new WindowAdapter()
		{
			@Override
			public void windowClosing(WindowEvent e)
			{
				exit();
			}
		});
		this.setIconImage(Icons.getIcon(Icons.JIBS_16).getImage());		
		
		setupToolBar();
		
		Dimension size = new Dimension(700, 400);
		setSize(size);

		setTitle("JIBS RankPad"); //$NON-NLS-1$

		this.myEditorPane.setEditable(false);
		this.myEditorPane.setDisabledTextColor(Color.black);

		this.myEditorScrollPane = new JScrollPane(this.myEditorPane,
				ScrollPaneConstants.VERTICAL_SCROLLBAR_AS_NEEDED,
				ScrollPaneConstants.HORIZONTAL_SCROLLBAR_NEVER);
		this.myEditorScrollPane.getViewport().addChangeListener(new ChangeListener()
		{
			public void stateChanged(ChangeEvent e)
			{
				if (!RankPad.this.isActive())
				{
					return;
				}
				// Adjust recent files
				if (RankPad.this.myRecentFiles.getSize() > 0)
				{
					Point p = RankPad.this.myEditorScrollPane.getViewport().getViewPosition();
					RecentFile f = (RecentFile)RankPad.this.myRecentFiles.getElementAt(0);		
					f.y = p.y;
				}
			}
		});

		this.myEditorTreePanel.setLeftComponent(this.myEditorScrollPane);
		this.myEditorTreePanel.setRightComponent(new JScrollPane(this.myTree));		
		
		this.rankAndFile.addTab("Rank", new JScrollPane(this.rankList));
		this.rankAndFile.addTab("Files", new JScrollPane(this.myDirectoryList));
				
		this.myListEditorTreePanel.setLeftComponent(this.rankAndFile);
		this.myListEditorTreePanel.setRightComponent(this.myEditorTreePanel);

		JPanel statusPanel = new JPanel(new BorderLayout());
		statusPanel.add(this.myStatusCount, BorderLayout.WEST);
		statusPanel.add(this.myStatusFileName, BorderLayout.CENTER);
		this.myStatusFileName.setEditable(false);
		
		this.myCenterPanel = new JPanel(new BorderLayout());
		getContentPane().add(this.myToolBar, BorderLayout.NORTH);
		this.myCenterPanel.add(this.myListEditorTreePanel, BorderLayout.CENTER);
		this.myCenterPanel.add(statusPanel, BorderLayout.SOUTH);		
		getContentPane().add(this.myCenterPanel);		
		
		this.setJMenuBar(this.makeMenus());
		this.myUImenu.initLookAndFeel();
		
		this.updateFont();

		// Size to 75% of screen.
		SwingUtils.sizeScreen(this, .75f);
		
 		doLayout();
		
		// Based on sizing, so do afterwards
		int storedSize = clsPrefs.getInt(FILE_PANEL_SIZE, FILE_MIN);
		if (storedSize < FILE_MIN)
		{
			storedSize = FILE_MIN;
		}
		this.myListEditorTreePanel.setDividerLocation(storedSize);
		doLayout();
		
		// Had to use center panel because myListPictureTreePanel has a width at
		// startup different than at shutdown.
		int w = this.myCenterPanel.getWidth() - clsPrefs.getInt(TREE_PANEL_SIZE, TREE_MIN);
		//System.out.println("list: " + this.myListEditorTreePanel.getDividerLocation());
		//System.out.println("width: " + this.myCenterPanel.getWidth());
		//System.out.println("tree: " + this.myEditorTreePanel.getDividerLocation());
		//System.out.println("TREE_PANEL_SIZE: " + w);
		if (w < TREE_MIN)
		{
			w = TREE_MIN;
		}
		//System.out.println("TREE_PANEL_SIZE: " + w);
		this.myEditorTreePanel.setDividerLocation(w);

		doLayout();
		
		SwingUtils.centerOnScreen(this); // seems to need to be here also
		
		//System.out.println("Rank Pad setVisable");
		super.setVisible(true);		
		//System.out.println("Rank Pad setVisable done");
	}
		
	void setupToolBar()
	{
		//myToolBar.add(myActions.makeSaveButton());
		//myToolBar.add(myActions.getToolBarButton("file", "saveas")); //$NON-NLS-1$ //$NON-NLS-2$
		//myToolBar.add(myActions.getToolBarButton("file", "rename")); //$NON-NLS-1$ //$NON-NLS-2$
		//myToolBar.addSeparator();

		//this.myToolBar.add(this.myLST.getRDPmanager().getMoveManager().getUndoButton()); 
		//this.myToolBar.add(this.myLST.getRDPmanager().getMoveManager().getRedoButton());
		//this.myToolBar.addSeparator();

		//this.myLST.getFileListManager().addButtonsToToolBar(this.myToolBar, this.myActions);
		this.myToolBar.add(this.myActions.getToolBarButton("view", "reload")); //$NON-NLS-1$ //$NON-NLS-2$
		this.myToolBar.addSeparator();

		this.myToolBar.add(new JLabel("Font Size  ")); 
		//myFontSize.setEditable(false);
		this.myFontSize.setSelectedItem(clsPrefs.get(FONT_SIZE, "12"));
		this.myFontSize.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent ae)
			{
				RankPad.this.updateFont();
			}
		});
		this.myFontSize.setMaximumSize(new Dimension(20,20));
		this.myToolBar.add(this.myFontSize); 
		this.myToolBar.addSeparator();		
		
		final JComboBox recentFiles = new JComboBox();
		recentFiles.setModel(this.myRecentFiles);
		recentFiles.setMinimumSize(new Dimension(100, 5));
		recentFiles.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent ae)
			{
				if (RankPad.this.iAmAdjustingComboBoxSelection)
				{
					return;
				}
				//System.out.println("JCombo box called");
				RecentFile rfToShow = (RecentFile)recentFiles.getSelectedItem();
				RankPad.this.rankList.showRecnetFile(rfToShow);
			}
		});
		this.myToolBar.add(recentFiles);
		
		// BIG GAP - to stuff on the right
		// Separator is needed when the toolbar is undocked.
		this.myToolBar.addSeparator();
		this.myToolBar.add(javax.swing.Box.createHorizontalGlue());

		this.myToolBar.add(makeDeleteButton(this.myActions));
		this.myToolBar.addSeparator();

		this.myToolBar.add(this.myActions.getToolBarButton("help", "about")); //$NON-NLS-1$ //$NON-NLS-2$
		this.myToolBar.add(this.myActions.getToolBarButton("help", "help")); //$NON-NLS-1$ //$NON-NLS-2$
	}
	
	void exit()
	{
		if (this.iNeedToFinalize)
		{
			this.iNeedToFinalize = false;
			
			// Remember Tree
			try
			{
				this.myTreeReaderWriter.getWriter().store(this.myTree.getRootNode());
			}
			catch(Exception ex)
			{
				ex.printStackTrace();
			}
			
			// Remember recent files
			try
			{
				PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(this.rankList.viewHistoryFile)));
				int size = this.myRecentFiles.getSize();
				for(int i=0; i<size && i<20; i++)
				{
					RecentFile rf = (RecentFile)this.myRecentFiles.getElementAt(i);
					//System.out.println("Remembering: " + rf);
					out.print(rf.y);
					out.print("\t");
					out.println(rf.filePath);
				}
				out.close();				
			}
			catch(Exception ex)
			{
				ex.printStackTrace();
			}
			
			clsPrefs.putInt(FILE_PANEL_SIZE, this.myListEditorTreePanel.getDividerLocation());
			
			// Had to use center panel because myListPictureTreePanel has a width at
			// startup different than at shutdown.
			int w = this.myCenterPanel.getWidth() - this.myEditorTreePanel.getDividerLocation();
			//System.out.println("list: " + this.myListEditorTreePanel.getDividerLocation());
			//System.out.println("width: " + this.myCenterPanel.getWidth());
			//System.out.println("tree: " + this.myEditorTreePanel.getDividerLocation());
			//System.out.println("saving: " + w);
			clsPrefs.putInt(TREE_PANEL_SIZE, w);			
			clsPrefs.put(FONT_SIZE, (String)this.myFontSize.getSelectedItem());
			try
			{
				clsPrefs.flush();
			}
			catch(Exception ex)
			{
				ex.printStackTrace();
			}
			
			System.out.println("" //$NON-NLS-1$
					+ "**************************************************** \n" //$NON-NLS-1$
					+ "*           JIBS Text Pad Exited Normally          * \n" //$NON-NLS-1$
					+ "**************************************************** \n"); //$NON-NLS-1$
			System.out.println(AppPluginStartup.JIBS_OUT); //$NON-NLS-1$
			
			AppStarter.closeDownApp();
			if (this.iExitOnClose)
			{
				System.exit(0);
			}
			else
			{
				// May want to kill any running worker thread.
				// Would need to track in here and manage from the actual workers
				// adding an interupt reutine to the workers.
				this.setVisible(false);
				dispose();
			}
		}
	}
	
	void openRankings()
	{
		String fileName = clsPrefs.get(RANK_FILE, null); 
		JFileChooserExtra chooser = new JFileChooserExtra(clsPrefs.get(RANK_FILE, null));
		chooser.setSaveName(getAppIDandTargetType(), fileName); 
		chooser.setDialogTitle("Ranking File"); 
		chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
		chooser.setMultiSelectionEnabled(false);
		//chooser.addChoosableFileFilter(new JibsTreeFileFilter());  

		int returnVal = chooser.showOpenDialog(this);
		if (returnVal == JFileChooser.APPROVE_OPTION)
		{
			File picked = chooser.getSelectedFile();
			setRankingFile(picked);
		}
	}
	
	void setRankingFile(File rankingFile)
	{
		if (rankingFile == null || !rankingFile.exists())
		{
			return;
		}					
		int maxListSize = 10;
   	PathManager.storeUniqueDirectories(RankPadActions.RECENT_VIEW, rankingFile.getAbsolutePath(), maxListSize, clsPrefs);
   	try
		{
			clsPrefs.put(RANK_FILE, rankingFile.toString());
			clsPrefs.flush();
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
		}			
		this.rankList.populate(clsPrefs.get(RANK_FILE, ""));		
	}
	
	void openTree()
	{
		String dirName = clsPrefs.get(TREE_DIRECTORY, System.getProperty("user.home")); 
		dirName = FileUtil.lookupDir("Tree Directory", dirName, this); //$NON-NLS-1$
      if (dirName != null)
      {
      	File dirFile = new File(dirName);
      	setTreeDirectory(dirFile);
      }
	}

	void setTreeDirectory(File treeDirectory)
	{
   	if (treeDirectory.exists() && treeDirectory.isDirectory())
   	{
			int maxListSize = 10;
      	PathManager.storeUniqueDirectories(RankPadActions.RECENT_TREE, treeDirectory.getAbsolutePath(), maxListSize, clsPrefs);
      	
      	System.out.println("Saving: " + treeDirectory);
      	clsPrefs.put(TREE_DIRECTORY, treeDirectory.getAbsolutePath()); 
      	try
      	{
      		clsPrefs.flush();
      	}
      	catch(Exception ex)
      	{
      		ex.printStackTrace();
      	}
   		Loading loading = new Loading(this, "Rank Pad");		
      	setTree(treeDirectory, loading);
   		loading.dispose();
   	}		
	}
	
	void setTree(File rootDirectoryForTree, Loading loading)
	{
		File remeberTreeNodesFile = new File(rootDirectoryForTree.getAbsolutePath()+ "/JibsRankPad.jibs");
		this.myTreeReaderWriter = new DTNReaderWriter("RankPad", getCountTargetFiles(), remeberTreeNodesFile); 
		this.myTree = new RankDirectoryTree(this, new FileFileFilter(), false);
		boolean rememberDirectories = true;
		this.myTreeManager = new TreeManager(this, this.myTree, this.myTreeReaderWriter, rememberDirectories);		
		this.myTreeManager.showViewTree(
				rootDirectoryForTree, 
				loading);
		this.myEditorTreePanel.setRightComponent(new JScrollPane(this.myTree));
	}
 
	JMenuBar makeMenus()
	{
		JMenu toolsMenu = new JMenu("Tools");
		class OptionsAction extends AbstractAction
		{
			public void actionPerformed(ActionEvent e)
			{
				new RankPadOptionsDialog(RankPad.this);
			}
		}
		JMenuItem options = new JMenuItem("Options");
		toolsMenu.add(options);
		options.addActionListener(new OptionsAction());
		
		JMenuBar menuBar = new JMenuBar();
		menuBar.add(this.myActions.getFileMenu());
		menuBar.add(toolsMenu);
		menuBar.add(this.myUImenu.populateUIMenu(ActionHelpers.makeMenu("ui"))); //$NON-NLS-1$
		menuBar.add( Box.createHorizontalGlue() );
		menuBar.add( this.myActions.getHelpMenu() ); 
		return menuBar;
	}	

	void updateFont()
	{
		int fontSize = 12;
		try
		{
			fontSize = Integer.parseInt((String)this.myFontSize.getSelectedItem());
			Font font = new Font("Serif", Font.PLAIN, fontSize);
			this.myEditorPane.setFont(font);
		}
		catch(Exception ex)
		{
			this.myFontSize.setSelectedItem("12");
		}
	}
	
	public JButton makeDeleteButton(ActiveActions aa)
	{
		this.deleteMenuAction = aa.getAction("file", "trash"); //$NON-NLS-1$ //$NON-NLS-2$
		this.deleteToolbarAction = aa.getAction("file", "trash");
		this.deleteToolbarAction.putValue(Action.NAME, this.deleteMenuAction.getValue(Action.NAME));
		this.deleteToolbarAction.putValue(Action.SHORT_DESCRIPTION, this.deleteMenuAction.getValue(Action.SHORT_DESCRIPTION));
		return ActiveActions.makeToolBarButton(this.deleteToolbarAction);
	}
	
	private void setDelete(boolean delete)
	{
		this.deleteMenuAction.setEnabled(delete);
		this.deleteToolbarAction.setEnabled(delete);
	}
	
	void deletePressed()
	{
		if (this.myDeleteDirectory != null && this.myDeleteDirectory.exists())
		{
			File deleteMe = this.rankList.getSelectedFile();
			if (deleteMe.exists())
			{
				try
				{
					FileUtil.moveFile(this.myDeleteDirectory, deleteMe);			
					
					// Remove from display and save the removal to disk.
					int selectedIndex = this.rankList.getSelectedIndex();
					this.rankList.listModel.remove(selectedIndex);
					this.rankList.saveFile();

					// Clear display
					this.myEditorPane.setText("");			
					
					// Forget this as a recent file
					this.iAmAdjustingComboBoxSelection = true;
					this.myRecentFiles.removeElement(new RecentFile(deleteMe,0));		
					this.iAmAdjustingComboBoxSelection = false;
					
					// Remove from tree.
					this.myTree.fileDeleted(deleteMe);
					
					// Set up to display the next file.
					if (selectedIndex >= this.rankList.listModel.size())
					{
						selectedIndex = this.rankList.listModel.size() - 1;
					}
					if (selectedIndex > -1)
					{
						this.rankList.setSelectedIndex(selectedIndex);	
						this.rankList.ensureIndexIsVisible(selectedIndex);
					}
				}
				catch (Exception ex)
				{
					ex.printStackTrace();
				}
			}
		}
		else
		{
         JOptionPane.showMessageDialog(
         	this,
            "No Trash directory is set.",
             Jibs.getString("problem"),
             JOptionPane.ERROR_MESSAGE );  			
		}
	}
	
	// ==========================================================================
	// Main
	// ==========================================================================
	public static void main(String[] args)
	{		
		// "org.gerhardb.jibs.Jibs" is correct for bundleName,
		// it gets expanded out to: org.gerhardb.jibs.Jibs.properties
		// or whatever the locale version is (DE for Germany etc.)
		AppStarter.startUpApp(args, "org.gerhardb.jibs.Jibs", true);

		new RankPad(true);
	}
	
}
