/*
 * Copyright (c) 2007 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.gerhardb.jibs.textPad.ranker;

import java.awt.BorderLayout;
import java.awt.Cursor;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.prefs.Preferences;

import javax.swing.*;

import org.gerhardb.jibs.Jibs;
import org.gerhardb.lib.swing.JFileChooserExtra;
import org.gerhardb.lib.swing.JPanelRows;

/**
 *
 * @author  Gerhard Beck
 */
public class CountAndRankPanel extends JPanelRows
{
	static final Preferences clsPrefs = Preferences.userRoot().node("/org/gerhardb/jibs/textPad/RankUI"); //$NON-NLS-1$
	private static final String ROOT_DIR = "ROOT_DIR";
	private static final String SCORING_FILE = "SCORING_FILE";
	private static final String WORD_COUNT_FILE = "WORD_COUNT_FILE";
	private static final String OUTPUT_FILE = "OUTPUT_FILE";
	private static final String OUTPUT_HTML = "OUTPUT_HTML";

	JProgressBar myProgress = new JProgressBar(); 
	WordCount myWordCount = new WordCount(this.myProgress);
	RankByWordCounts myRankByWordCounts = new RankByWordCounts(this.myProgress);

	JTextField myRootDir = new JTextField(50);
	JTextField scoringFileFld = new JTextField(50);
	JTextField wordCountFileFld = new JTextField(50);
	JTextField outputFileFld = new JTextField(50);
	JCheckBox outputHTML = new JCheckBox("Output HTML Rankings");
	JButton stopCountingBtn = new JButton("Stop Counting");
	JButton stopRankingBtn = new JButton("Stop Ranking");
	JButton goCountBtn = new JButton("Count Words");
	JButton goRankBtn = new JButton("Do Ranking");

	//===========================================================================
	//                              Constructor
	//===========================================================================
	public CountAndRankPanel()
	{
		layoutComponents();
	}

	void exit()
	{
		try
		{
			clsPrefs.put(ROOT_DIR, this.myRootDir.getText());
			clsPrefs.put(SCORING_FILE, this.scoringFileFld.getText());
			clsPrefs.put(WORD_COUNT_FILE, this.wordCountFileFld.getText());
			clsPrefs.put(OUTPUT_FILE, this.outputFileFld.getText());
			clsPrefs.putBoolean(OUTPUT_HTML, this.outputHTML.isSelected());
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
		}
	}

	void layoutComponents()
	{
		try
		{
			this.myRootDir.setText(clsPrefs.get(ROOT_DIR, System
					.getProperty("user.home")));
			this.wordCountFileFld.setText(clsPrefs.get(WORD_COUNT_FILE, System
					.getProperty("user.home")));
			this.scoringFileFld.setText(clsPrefs.get(SCORING_FILE, System
					.getProperty("user.home")));
			this.outputFileFld.setText(clsPrefs.get(OUTPUT_FILE, System
					.getProperty("user.home")));
			this.outputHTML.setSelected(clsPrefs.getBoolean(OUTPUT_HTML, true));
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
		}

		this.myProgress.setMinimum(0);
		this.myProgress.setIndeterminate(false);
		this.myProgress.setStringPainted(true);
		this.myProgress.setString(""); //$NON-NLS-1$
		
		JButton rootBtn = new JButton("..."); //$NON-NLS-1$
		rootBtn.addActionListener(new java.awt.event.ActionListener()
		{
			public void actionPerformed(java.awt.event.ActionEvent evt)
			{
				selectRoot();
			}
		});

		this.goCountBtn.addActionListener(new java.awt.event.ActionListener()
		{
			public void actionPerformed(java.awt.event.ActionEvent evt)
			{
				Runnable runMe = new Runnable()
				{
					public void run()
					{
						countWords(); 
					}
				};
				new Thread(runMe).start();	
			}
		});

		this.goRankBtn.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent ae)
			{
				Runnable runMe = new Runnable()
				{
					public void run()
					{
						doRank();
					}
				};
				new Thread(runMe).start();	
			}
		});

		this.stopCountingBtn.addActionListener(new java.awt.event.ActionListener()
		{
			public void actionPerformed(java.awt.event.ActionEvent evt)
			{
				CountAndRankPanel.this.myRankByWordCounts.stop();
			}
		});

		this.stopRankingBtn.addActionListener(new java.awt.event.ActionListener()
		{
			public void actionPerformed(java.awt.event.ActionEvent evt)
			{
				CountAndRankPanel.this.myRankByWordCounts.stop();
			}
		});
		
		this.stopCountingBtn.setEnabled(false);
		this.stopRankingBtn.setEnabled(false);

		JPanel nextRow = this.topRow();
		nextRow.add(new JLabel("Directory: "));
		nextRow.add(this.myRootDir);
		nextRow.add(rootBtn);

		nextRow = this.nextRow();
		nextRow.add(new JLabel("Scoring file: "));
		nextRow.add(this.scoringFileFld);
		nextRow.add(makeFileButton(this.scoringFileFld, SCORING_FILE));

		nextRow = this.nextRow(FlowLayout.CENTER);
		nextRow.add(this.goCountBtn);
		nextRow.add(this.stopCountingBtn);

		nextRow = this.nextRow();
		nextRow.add(new JLabel("Word Count file: "));
		nextRow.add(this.wordCountFileFld);
		nextRow.add(makeFileButton(this.wordCountFileFld, WORD_COUNT_FILE));

		nextRow = this.nextRow(FlowLayout.CENTER);
		nextRow.add(this.goRankBtn);
		nextRow.add(this.stopRankingBtn);

		nextRow = this.nextRow();
		nextRow.add(new JLabel("Output file: "));
		nextRow.add(this.outputFileFld);
		nextRow.add(makeFileButton(this.outputFileFld, OUTPUT_FILE));

		nextRow = this.nextRow();
		nextRow.add(this.outputHTML);

		nextRow = this.nextRow(new BorderLayout());
		nextRow.add(this.myProgress, BorderLayout.CENTER);
	}

	JButton makeFileButton(final JTextField field, final String prefsLabel)
	{
		JButton rtnMe = new JButton("..."); //$NON-NLS-1$
		rtnMe.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent ae)
			{
				String file = field.getText();
				if (file == null)
				{
					file = System.getProperty("user.home"); //$NON-NLS-1$
				}
				file = lookupFile("File with word counts", file);
				field.setText(file);
				try
				{
					clsPrefs.put(prefsLabel, file);
				}
				catch (Exception ex)
				{
					ex.printStackTrace();
				}
			}
		});
		return rtnMe;
	}

	String lookupFile(String msg, String prior)
	{
		JFileChooser chooser = new JFileChooser();
		chooser.setDialogTitle(msg);
		chooser.setFileSelectionMode(JFileChooser.FILES_AND_DIRECTORIES);
		
		File priorFile = new File(prior);
		if (priorFile.isFile() || !priorFile.exists())
		{
			priorFile = priorFile.getParentFile();
		}
		chooser.setCurrentDirectory(priorFile);

		int returnVal = chooser.showOpenDialog(this);
		if (returnVal == JFileChooser.APPROVE_OPTION)
		{
			try
			{
				return chooser.getSelectedFile().getCanonicalPath();
			}
			catch (IOException ex)
			{
				ex.printStackTrace();
			}
		}
		return null;
	}

	void selectRoot()
	{
		JFileChooserExtra chooser = new JFileChooserExtra(this.myRootDir.getText());
		chooser.setApproveButtonText("Select Root");
		chooser.setDialogTitle("Select root directory to do word count in");
		chooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
		chooser.setMultiSelectionEnabled(false);

		int returnVal = chooser.showOpenDialog(this);
		if (returnVal == JFileChooser.APPROVE_OPTION)
		{
			File picked = chooser.getSelectedFile();
			if (picked != null)
			{
				this.myRootDir.setText(picked.toString());
				try
				{
					clsPrefs.put(ROOT_DIR, picked.toString());
					clsPrefs.flush();
				}
				catch (Exception ex)
				{
					ex.printStackTrace();
				}
			}
		}
	}

	void countWords()
	{
		this.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
		try
		{
			File scoringFile = new File(this.scoringFileFld.getText());
			File outputFile = new File(this.wordCountFileFld.getText());

			if (scoringFile.equals(outputFile)) { throw new Exception(
					"Output files must be different than scorking file."); }
			
			String rootString = this.myRootDir.getText();
			File rootDir = new File(rootString);

			String[] words = getWordListFromScoreFile(scoringFile);
			
			this.goRankBtn.setEnabled(false);
			this.stopCountingBtn.setEnabled(true);
			this.myWordCount.count(words, rootDir, outputFile);
			this.stopCountingBtn.setEnabled(false);
			this.goRankBtn.setEnabled(true);
		}
		catch (Exception ex)
		{
			JOptionPane.showMessageDialog(this, ex.getMessage(), "Problem",
					JOptionPane.ERROR_MESSAGE);
			return;
		}
		this.setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
	}
	
	@SuppressWarnings("null")
	String[] getWordListFromScoreFile(File scoringFile) 
	{
		if (scoringFile.exists() && scoringFile.isFile())
		{
			String errMsg = null;
			HashMap<String, String> wordList = new HashMap<String, String>(50);
			System.out.println("\n---------------------------------------------------");
			System.out.println("Scanning scoring file: " + scoringFile);
			
			BufferedReader buffRead = null;
			try
			{
				// First read in the file names.
				buffRead = new BufferedReader(new FileReader(scoringFile));
				
				String aLine = buffRead.readLine();
				while (aLine != null)
				{
					System.out.println(aLine);
					aLine = aLine.trim();
					if (!aLine.startsWith("#") && !aLine.startsWith("@") && aLine.length()>0)
					{
						String[] tokens = aLine.split(":");
						if (tokens.length > 3)
						{
							throw new Exception("This line has more than two \":\":" + aLine);				
						}
						
						String words = null;
						if (tokens.length == 3)
						{
							words = tokens[1].trim();
						}
						else if (tokens.length == 2)
						{
							words = tokens[0].trim();
						}
						
						if (words == null || words.length() == 0)
						{
							throw new Exception("No words found in:" + aLine);				
						}
						
						if (words.startsWith("(") && words.endsWith(")"))
						{
							words = words.substring(1, words.length()-1);
						}
						
						String[] wordsInThisLine = words.split(",");						
						for(int i=0;i<wordsInThisLine.length;i++)
						{
							String aWord = wordsInThisLine[i].trim();
							wordList.put(aWord, aWord);
						}
					}
					aLine = buffRead.readLine();
				}
				buffRead.close();
			}
			catch(Exception ex)
			{
				errMsg = ex.getMessage();			
				ex.printStackTrace();
			}
			finally
			{
				try
				{
					buffRead.close();
				}
				catch(Exception ex)
				{
					ex.printStackTrace();
				}
			}	
			if (errMsg != null)
			{
	         JOptionPane.showMessageDialog(
		            this,
		            errMsg,
		            "Problem",
		            JOptionPane.ERROR_MESSAGE);
	         return new String[0];
			}

			String[] allWords = wordList.values().toArray(new String[wordList.values().size()]);
			Arrays.sort(allWords);
			return allWords;
		}
		JOptionPane.showMessageDialog(
		   this,
		   "Could not find file: " + scoringFile,
		   "Problem",
		   JOptionPane.ERROR_MESSAGE);	
		return new String[0];
	}

	void doRank()
	{
		try
		{
			this.goCountBtn.setEnabled(false);
			this.stopRankingBtn.setEnabled(true);
			this.myRankByWordCounts.rank(this.wordCountFileFld.getText(),
					this.scoringFileFld.getText(), this.outputFileFld.getText(), this.outputHTML.isSelected());
			this.stopRankingBtn.setEnabled(false);
			this.goCountBtn.setEnabled(true);
		}
		catch (Exception ex)
		{
			JOptionPane.showMessageDialog(CountAndRankPanel.this, ex
					.getMessage(), Jibs.getString("problem"), //$NON-NLS-1$
					JOptionPane.ERROR_MESSAGE);
			ex.printStackTrace();
		}		
	}
}
