/*
 * Copyright (c) 2010 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.gerhardb.jibs.textPad.ranker;

import java.util.*;

class ScoringFactors
{
	RankByWordCounts ranker;

	String myLabel; // a or jibs-row#
	boolean iAmAGroup = false;
	String[] myScoringWordsOrGroups; // sub,submarine		
	// myScoring will be sorted to be in order even if not entered that way.
	private ScoreForCount[] myScoring; // 0>300,5>500,10>700,20>1000

	// Turned to false as this is added to evaluation array.
	boolean iNeedArranging = true;

	ScoringFactors(RankByWordCounts rankerIn, int count, String aLine)
			throws Exception
	{
		this.ranker = rankerIn;
		RankByWordCounts.logln("\nLine: " + count);
		RankByWordCounts.logln("Input:  " + aLine);

		String[] tokens = aLine.split(":");
		if (tokens.length > 3) { throw new Exception(
				"This line has more than two \":\":" + aLine); }

		String words = null;
		String scores = null;
		if (tokens.length == 3)
		{
			this.myLabel = tokens[0].trim();
			words = tokens[1].trim();
			scores = tokens[2].trim();
		}
		else if (tokens.length == 2)
		{
			this.myLabel = "jibs-" + count;
			words = tokens[0].trim();
			scores = tokens[1].trim();
		}

		if (words == null || words.length() == 0) { throw new Exception(
				"No words found in:" + aLine); }

		if (words.startsWith("(") && words.endsWith(")"))
		{
			this.iAmAGroup = true;
			words = words.substring(1, words.length() - 1);
			this.myScoringWordsOrGroups = words.split(",");
		}
		else
		{
			this.iAmAGroup = false;
			this.myScoringWordsOrGroups = words.split(",");
		}

		for (int i = 0; i < this.myScoringWordsOrGroups.length; i++)
		{
			this.myScoringWordsOrGroups[i] = this.myScoringWordsOrGroups[i].trim();
		}

		if (scores == null || scores.length() == 0) { throw new Exception(
				"No scores found in:" + aLine); }
		String[] scoreTokens = scores.split(",");
		this.myScoring = new ScoreForCount[scoreTokens.length];
		for (int i = 0; i < scoreTokens.length; i++)
		{
			this.myScoring[i] = new ScoreForCount(scoreTokens[i]);
		}
		Arrays.sort(this.myScoring);

		RankByWordCounts.logln("Output: " + this);
	}

	/**
	 * This must be called so that all individual words are processed before
	 * any groups and that there are no cyclical group issues.
	 * @param wordCounts
	 * @param scoresByFactor
	 * @return
	 * @throws Exception
	 */
	int score(int[] wordCounts, HashMap<String, LabelAndCount> scoresByFactor) throws Exception
	{
		if (this.iAmAGroup) { return scoreAGroup(scoresByFactor); }
		return scoreAWord(wordCounts, scoresByFactor);
	}

	int scoreAWord(int[] wordCounts, HashMap<String, LabelAndCount> scoresByFactor) throws Exception
	{
		int wordCount = 0;
		// Add up score for all words in this factor before returning.
		for (int word = 0; word < this.myScoringWordsOrGroups.length; word++)
		{
			String scoringWord = this.myScoringWordsOrGroups[word];
			// Check each word in the count file until a match is found.
			for (int tokenCounter = 1; tokenCounter < this.ranker.myTokenList.length; tokenCounter++)
			{
				String tokenWord = this.ranker.myTokenList[tokenCounter];
				RankByWordCounts.logln("Checking: " + scoringWord + " against "
						+ tokenWord);
				if (tokenWord.equals(scoringWord))
				{
					wordCount += wordCounts[tokenCounter];
					RankByWordCounts.logln("FOUND COUNT OF: "
							+ wordCounts[tokenCounter] + " for " + tokenWord);
				}
			}
		}
		int score = 0;
		for (int scoring = 0; scoring < this.myScoring.length; scoring++)
		{
			if (wordCount > this.myScoring[scoring].myCount)
			{
				score = this.myScoring[scoring].myScore;
				break;
			}
		}
		LabelAndCount was = new LabelAndCount(this.myLabel, wordCount);
		scoresByFactor.put(was.myLabel, was);
		RankByWordCounts.logln("Factor: " + this.myLabel + "  Word Count: "
				+ wordCount + " Score: " + score);
		return score;
	}

	int scoreAGroup(HashMap<String, LabelAndCount> scoresByFactor) throws Exception
	{
		int wordInstancesForGroup = 0;
		// Groups will always match the first token and 
		// will ALWAYS return a value with the first token.
		for (int word = 0; word < this.myScoringWordsOrGroups.length; word++)
		{
			LabelAndCount was = scoresByFactor
					.get(this.myScoringWordsOrGroups[word]);
			if (was == null) { throw new Exception(
					"Could not find score for label: "
							+ this.myScoringWordsOrGroups[word]); }
			if (was.myCount > 0)
			{
				wordInstancesForGroup++;
			}
			RankByWordCounts.logln("For word: "
					+ this.myScoringWordsOrGroups[word] + " found: " + was
					+ "  cummulative word count: " + wordInstancesForGroup);
		}

		// Look up the score
		int groupScore = 0;
		for (int scoring = 0; scoring < this.myScoring.length; scoring++)
		{
			if (wordInstancesForGroup > this.myScoring[scoring].myCount)
			{
				groupScore = this.myScoring[scoring].myScore;
				break;
			}
		}

		LabelAndCount was = new LabelAndCount(this.myLabel, wordInstancesForGroup);
		scoresByFactor.put(was.myLabel, was);

		RankByWordCounts.logln("Group: " + this.myLabel + " has "
				+ wordInstancesForGroup + " words which scored: " + groupScore);
		return groupScore;
	}

	@Override
	public boolean equals(Object obj)
	{
		if (obj instanceof ScoringFactors)
		{
			ScoringFactors other = (ScoringFactors) obj;
			if (other.myLabel == this.myLabel) { return true; }
		}
		return false;
	}

	@Override
	public int hashCode()
	{
		return this.myLabel.hashCode();
	}

	@Override
	public String toString()
	{
		StringBuffer buff = new StringBuffer(100);
		buff.append(this.myLabel);
		buff.append(":");
		if (this.iAmAGroup)
		{
			buff.append("(");
		}
		for (int i = 0; i < this.myScoringWordsOrGroups.length; i++)
		{
			if (i > 0)
			{
				buff.append(",");
			}
			buff.append(this.myScoringWordsOrGroups[i]);
		}
		if (this.iAmAGroup)
		{
			buff.append(")");
		}
		buff.append(":");
		for (int i = 0; i < this.myScoring.length; i++)
		{
			if (i > 0)
			{
				buff.append(",");
			}
			buff.append(this.myScoring[i]);
		}
		return buff.toString();
	}
}
