/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
package org.gerhardb.jibs.util;

import java.awt.*;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.*;
import java.text.DecimalFormat;
import java.util.*;
import java.util.prefs.Preferences;
import java.util.zip.*;

import javax.swing.*;
import org.gerhardb.jibs.Jibs;
import org.gerhardb.lib.io.DirectoriesOnlyFileFilter;
import org.gerhardb.lib.io.EndingFileFilter;
import org.gerhardb.lib.io.EzLogger;
import org.gerhardb.lib.io.*;
import org.gerhardb.lib.swing.JFileChooserExtra;
import org.gerhardb.lib.swing.JPanelRows;
import org.gerhardb.lib.swing.SwingUtils;
import org.gerhardb.lib.util.Icons;
import org.gerhardb.lib.util.StopCheck;

/**
 */

public class FileRenameWithDirName extends JFrame implements StopCheck, Runnable
{
	private static final int BUFFER = 2048;
	private static final String ZIP_FILE = "ZipFile";  //$NON-NLS-1$
	private static final String SUB_DIR = "SubFile";  //$NON-NLS-1$
	private static final String MOVE_FILES = "MoveFile";  //$NON-NLS-1$
	
	private static final String LAST_ROOT = "LastRoot";  //$NON-NLS-1$

	private static final Preferences clsPrefs = Preferences.userRoot().node(
			"/org/gerhardb/jibs/util/FileRenameWithDirName");  //$NON-NLS-1$

	JTextField myRootDir = new JTextField(60);
	JProgressBar myProgressBar = new JProgressBar(); //$NON-NLS-1$

	boolean iStop = false;
	JButton myStopBtn = new JButton("Stop");  
	String myRootString;
	
	JCheckBox myZipFiles = new JCheckBox("Unzip Zip Files (might not always process zipped directories)");
	JCheckBox mySubDirectories = new JCheckBox("Rename files in subdirectories");
	JCheckBox myMoveFilesToRootDirectory = new JCheckBox("Move renamed files to root directory");
   boolean iExitOnClose;
   EzLogger myLogger;
	
	/**
	 * Zip functionality ready to be wired in.
	 */
	public FileRenameWithDirName(boolean exitOnClose)
	{
		super("Rename Files with Directory Name"); 
		
		this.iExitOnClose = exitOnClose;
		if (this.iExitOnClose)
		{
			this.addWindowListener(new WindowAdapter()
			{
				@Override
				public void windowClosing(WindowEvent evt)
				{
					System.exit(0);
				}
			});
		}

		layoutComponents();
		this.myRootDir.setText(clsPrefs.get(LAST_ROOT, null));

		this.setIconImage(Icons.getIcon(Icons.JIBS_16).getImage());
		
		this.myZipFiles.setSelected(clsPrefs.getBoolean(ZIP_FILE, false));
		this.mySubDirectories.setSelected(clsPrefs.getBoolean(SUB_DIR, true));
		this.myMoveFilesToRootDirectory.setSelected(clsPrefs.getBoolean(MOVE_FILES, true));

		EventQueue.invokeLater(new Runnable()
		{
			public void run()
			{
				FileRenameWithDirName.this.pack();
				SwingUtils.centerOnScreen(FileRenameWithDirName.this);
				FileRenameWithDirName.this.setVisible(true);
			}
		});
		
	}

	public boolean isStopped(){return this.iStop;}
	
	private void layoutComponents()
	{
		// Set up application
		this.setSize(new Dimension(600, 600));
		
		this.myProgressBar.setStringPainted(true);

		JButton goBtn = new JButton("Rename");  
		goBtn.addActionListener(new java.awt.event.ActionListener()
		{
			public void actionPerformed(java.awt.event.ActionEvent evt)
			{
	        Thread t = new Thread(FileRenameWithDirName.this);
           t.start();
			}
		});

      this.myStopBtn.addActionListener(new java.awt.event.ActionListener()
	    {
	       public void actionPerformed(java.awt.event.ActionEvent evt)
	       {
	       	FileRenameWithDirName.this.iStop = true;
	       }
	    });
            
   	
		JButton rootBtn = new JButton("...");  //$NON-NLS-1$
		rootBtn.addActionListener(new java.awt.event.ActionListener()
		{
			public void actionPerformed(java.awt.event.ActionEvent evt)
			{
				selectRoot();
			}
		});

		JPanelRows topPanel = new JPanelRows();
		JPanel aRow = topPanel.topRow(FlowLayout.CENTER);
		
      aRow.add(new JLabel("Root Directory: "));   
      aRow.add(this.myRootDir);
      aRow.add(rootBtn);

      aRow = topPanel.nextRow();
      aRow.add(this.myZipFiles);
      
      aRow = topPanel.nextRow();
      aRow.add(this.mySubDirectories);
      
      aRow = topPanel.nextRow();
      aRow.add(this.myMoveFilesToRootDirectory);

      aRow = topPanel.nextRow();
      aRow.add(goBtn);
      aRow.add(this.myStopBtn);

      //aRow = topPanel.nextRow();
      //aRow.add(myProgressBar);
	      
      JPanel content = new JPanel(new BorderLayout());
      content.add(topPanel, BorderLayout.CENTER);
      content.add(this.myProgressBar, BorderLayout.SOUTH);
      
		this.setContentPane(content);
	}

	void selectRoot()
	{
		JFileChooserExtra chooser = new JFileChooserExtra(clsPrefs.get(LAST_ROOT, null));
		chooser.setSaveName("FileRenameWithDirName", "Select Root Directory");  
		chooser.setApproveButtonText("Select Root Directory"); 
		chooser.setDialogTitle("Root");  
		chooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
		chooser.setMultiSelectionEnabled(false);

		int returnVal = chooser.showOpenDialog(this);
		if (returnVal == JFileChooser.APPROVE_OPTION)
		{
			File picked = chooser.getSelectedFile();
			if (picked != null)
			{
				this.myRootDir.setText(picked.toString());
				try
				{
					clsPrefs.put(LAST_ROOT, picked.toString());
					clsPrefs.flush();
				}
				catch (Exception ex)
				{
					ex.printStackTrace();
				}
			}
		}
	}

	public void run()
	{
		clsPrefs.putBoolean(ZIP_FILE, this.myZipFiles.isSelected());
		clsPrefs.putBoolean(SUB_DIR, this.mySubDirectories.isSelected());
		clsPrefs.putBoolean(MOVE_FILES, this.myMoveFilesToRootDirectory.isSelected());

		this.myRootString = this.myRootDir.getText();
		File rootFile = new File(this.myRootString);
		if (!rootFile.exists() || !rootFile.isDirectory())
		{
	         JOptionPane.showMessageDialog(
	            this,
	            "Root directory must be a directory", 
	            "Problem",  
	            JOptionPane.ERROR_MESSAGE);
	         return;
		}
		//System.out.println("Root Directory: " + rootFile);
		
		try
		{
			this.myLogger = EzLogger.makeEzLogger(rootFile);
		}
		catch(Exception ex)
		{
	         JOptionPane.showMessageDialog(
	            this,
	            "Could not open log file in:\n " + rootFile, 
	            "Problem",  
	            JOptionPane.ERROR_MESSAGE);
	         return;			
		}
		
		setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
		
		try
		{
			if (this.myZipFiles.isSelected())
			{
				//System.out.println("myZipFiles.isSelected()");
				//throw new Exception("Zip not yet implemented");
				File[] dirs = rootFile.listFiles(new EndingFileFilter("zip"));
				this.myProgressBar.setValue(0);
				this.myProgressBar.setMaximum(dirs.length);
				this.myProgressBar.setIndeterminate(false);	
							
				Arrays.sort(dirs, new FileNameComparatorInsensative() );
				for(int i=0; i<dirs.length; i++)
				{
					System.out.println("Subdirectory: " + dirs[i] + "    " + dirs[i].getName());
					this.myLogger.logLine(EzLogger.NEW_LINE + "Unzipping: " + dirs[i] + EzLogger.NEW_LINE);
					this.myProgressBar.setString("Processing subdirectory: " + dirs[i].getName());
					this.myProgressBar.setValue(i);
					
					try
					{
						ArrayList<String> nameList = new ArrayList<String>(100);
						ZipFile zipFile = new ZipFile(dirs[i]);
						Enumeration<?> entries = zipFile.entries();
						while (entries.hasMoreElements())
						{
							ZipEntry entry = (ZipEntry)entries.nextElement();
							if (!entry.isDirectory())
							{
								nameList.add(entry.getName());
							}
						}
						String[] names = new String[nameList.size()];
						names = nameList.toArray(names);
						Arrays.sort(names, new FileNameStringComparatorInsensative() );
						for(int j=0; j<names.length; j++)
						{
							ZipEntry entry = zipFile.getEntry(names[j]);
							//System.out.println(entry.getName());
				         BufferedInputStream is = new BufferedInputStream
			              (zipFile.getInputStream(entry));
			            int count;
			            byte data[] = new byte[BUFFER];
			            
			            String nameWithRoot = this.myRootString + "/" + entry.getName();
			            File entryFile = new File(nameWithRoot);
			            File parent = entryFile.getParentFile();
			            if (!parent.exists())
		            	{
		            		parent.mkdirs();
		            	}	            
							this.myLogger.logLine("    Inflating: " + nameWithRoot);
							FileOutputStream fos = new 
			              FileOutputStream(nameWithRoot);
			            BufferedOutputStream dest = new 
			              BufferedOutputStream(fos, BUFFER);
			            while ((count = is.read(data, 0, BUFFER)) != -1) 
			            {
			               dest.write(data, 0, count);
			            }
			            dest.flush();
			            dest.close();
			            is.close();
						}
					}
					catch(Exception ex)
					{
						System.out.println(ex);
						ex.printStackTrace();
						this.myLogger.logLine(ex.getMessage());
					}
				}
				this.myProgressBar.setValue(dirs.length);
			}
			if (this.mySubDirectories.isSelected())
			{
				//System.out.println("mySubDirectories.isSelected()");
				File[] dirs = rootFile.listFiles(DirectoriesOnlyFileFilter.DIRECTORIES_ONLY);
				this.myProgressBar.setValue(0);
				this.myProgressBar.setMaximum(dirs.length);
				this.myProgressBar.setIndeterminate(false);	
							
				Arrays.sort(dirs, new FileNameComparatorInsensative() );
				for(int i=0; i<dirs.length; i++)
				{
					//System.out.println("Subdirectory: " + dirs[i] + "    " + dirs[i].getName());
					this.myLogger.logLine(EzLogger.NEW_LINE + EzLogger.NEW_LINE + "Processing Directory: " + dirs[i].getName() + EzLogger.NEW_LINE);
					this.myProgressBar.setString("Processing subdirectory: " + dirs[i].getName());
					this.myProgressBar.setValue(i);
					File[] changedFiles = doNameChange(dirs[i]);
					if (changedFiles != null && this.myMoveFilesToRootDirectory.isSelected())
					{				
						for (int j = 0; j < changedFiles.length; j++)
						{
							if (changedFiles[j] != null)
							{
								FileUtil.moveFile(rootFile, changedFiles[j], this);
								this.myLogger.logLine("     Moving: " + changedFiles[j].getName() + " to " + rootFile);
							}
						}
						// If there is nothing in the directory, delete it.
						if (dirs[i].listFiles().length == 0)
						{
							this.myLogger.logLine(EzLogger.NEW_LINE + EzLogger.NEW_LINE + "Removing directory: " + dirs[i] + EzLogger.NEW_LINE  + EzLogger.NEW_LINE);
							dirs[i].delete();
						}
					}
				}	
				this.myProgressBar.setValue(dirs.length);
			}
			this.myLogger.close();
		}
		catch(Exception ex)
		{
	         JOptionPane.showMessageDialog(
	            this,
	            ex.getMessage(), 
	            "Problem Encountered", 
	            JOptionPane.ERROR_MESSAGE);
		}
		setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
		this.myProgressBar.setString("Done");
	}
	
	/**
	 * Copied from: FileSeveralNameChangeDialog
	 */
	File[] doNameChange(File renameDir)
	{
		String newBase = renameDir.getName();
		File[] inputFileList = renameDir.listFiles(FilesOnlyFileFilter.FILES_ONLY);
		File[] outputFileList = new File[inputFileList.length];
		String failureMsg = "Could not rename";
		boolean failures = false;
		setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
		
		// Make sure files are in alphabetical order
		Arrays.sort(inputFileList);
		
		try
		{
			// First pass we will get the new names and see if there are any
			// duplicates.
			for (int i = 0; i < inputFileList.length; i++)
			{
				if (inputFileList[i] != null)
				{
					//System.out.println("Looking at: " + inputFileList[i]);
					try
					{
						DecimalFormat formatter = FileRename.getDecimalFormat(inputFileList.length);
						String dir = inputFileList[i].getParentFile().getCanonicalPath();
						String newName = dir + "/" //$NON-NLS-1$
								+ getNewName(
										inputFileList[i].getName(), 
										i + 1, newBase, formatter);	
						outputFileList[i] = new File(newName);
						//System.out.println("Renamed to: " + outputFileList[i]);
						if (outputFileList[i].exists())
						{
							failures = true;
							failureMsg =   "There is already a file named:\n" + outputFileList[i];
							break;
						}
					}
					catch (Exception ex)
					{
						System.out.println("FileRenameWithDirName A: " + ex.getMessage()); //$NON-NLS-1$
						failures = true;
					}
				}
			}
			
			// Now we do the actual rename.
			if (failures)
			{
				return null;
			}
			
			for (int i = 0; i < inputFileList.length; i++)
			{
				if (inputFileList[i] != null)
				{
					try
					{
						String entry = "Renaming: " + inputFileList[i] + EzLogger.NEW_LINE + "to: " + outputFileList[i] + EzLogger.NEW_LINE;
						//System.out.println(entry);
						this.myLogger.logLine(entry);
						inputFileList[i].renameTo(outputFileList[i]);
					}
					catch (Exception ex)
					{
						System.out.println("FileRenameWithDirName B: " + ex.getMessage()); //$NON-NLS-1$
						failures = true;
					}
				}
			}
		}
		finally
		{
			setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
		}

		if (failures)
		{
			JOptionPane.showMessageDialog(this, failureMsg, 
					Jibs.getString("FileList.11"), //$NON-NLS-1$
					JOptionPane.WARNING_MESSAGE);
		}
		return outputFileList;
	}
	
   public String getNewName(String oldName, int i, String newBase, DecimalFormat formatter)
	{
		String baseName = newBase + "-" + formatter.format(i);

		// OK, we don't have an ending to use, so let's compute one.
		if (oldName == null) { return baseName; }

		int lastPeriod = oldName.lastIndexOf('.');
		
		// We don't care if the file has no ending.
		// If the file has more than one ending, take the first one.
		// And we need a front and a back.
		// Also, we don't care in the impossible case of a name of '.'.
		if (lastPeriod < 0 || oldName.length() == 1) { return baseName; }

		String oldEnding = oldName.substring(lastPeriod + 1).toLowerCase();
		//System.out.println("ending: " + ending);

		return baseName + "." + oldEnding; //$NON-NLS-1$
	}

	// ==========================================================================
	// Other display functions
	// ==========================================================================
		
   /*
	class ZipFileFilter extends FileFilter
	{
		public  boolean accept(File f)
		{
			if (f == null){return false;}
			if (f.isDirectory()){return true;}
			String name = f.getName().toLowerCase();
			if (name.endsWith(".zip")) //$NON-NLS-1$
			{
				return true;
			}
			return false;
		}
		
		public  String getDescription()
		{
			return "Zip File Filter"; 
		}
	}	
	*/
   
	// ==========================================================================
	// Main
	// ==========================================================================
	public static void main(String[] args)
	{
		new FileRenameWithDirName(true);
	}
}
