/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
package org.gerhardb.jibs.util;

import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.EventQueue;
import java.awt.FlowLayout;
import java.io.File;
import java.util.Enumeration;
import java.util.prefs.Preferences;

import javax.swing.JButton;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.filechooser.FileFilter;

import org.gerhardb.jibs.Jibs;
import org.gerhardb.jibs.viewer.ViewerPreferences;
import org.gerhardb.lib.dirtree.DTNReader;
import org.gerhardb.lib.dirtree.DTNReaderWriter;
import org.gerhardb.lib.dirtree.DirectoryTree;
import org.gerhardb.lib.dirtree.DirectoryTreeNode;
import org.gerhardb.lib.dirtree.rdp.PathManager;
import org.gerhardb.lib.swing.JFileChooserExtra;
import org.gerhardb.lib.swing.JPanelRows;
import org.gerhardb.lib.swing.SwingUtils;
import org.gerhardb.lib.util.Icons;
import org.gerhardb.lib.util.StopCheck;
import org.gerhardb.lib.util.startup.ILoadingMessage;

/**
 */

public class RecreateDirectories extends JFrame implements StopCheck
{
	private static final String LAST_FILE = "LastFile";  //$NON-NLS-1$
	private static final String LAST_ROOT = "LastRoot";  //$NON-NLS-1$
	private static final String APP_NAME = "RecreateDirectories";

	private static final Preferences clsPrefs = Preferences.userRoot().node(
			"/org/gerhardb/jibs/util/RecreateDirectories");  //$NON-NLS-1$

	JTextField myFileName = new JTextField(60);
	JTextField myRootDir = new JTextField(60);
	JLabel myProgress = new JLabel("                                                                                       "); //$NON-NLS-1$
	DTNReaderWriter myReaderWriter;
	
	boolean iStop = false;
	JButton myStopBtn = new JButton(Jibs.getString("RecreateDirectories.4"));  //$NON-NLS-1$
	
	public RecreateDirectories(String targetType)
	{
		super(Jibs.getString("RecreateDirectories.5"));  //$NON-NLS-1$		
		this.myFileName.setText(clsPrefs.get(LAST_FILE, null));
		this.myRootDir.setText(clsPrefs.get(LAST_ROOT, null));

		this.myReaderWriter = new DTNReaderWriter(targetType, ViewerPreferences.countImageFilesInNodes(), PathManager.getStartingJibsFile());

		layoutComponents();
		this.setIconImage(Icons.getIcon(Icons.JIBS_16).getImage());

		EventQueue.invokeLater(new Runnable()
		{
			public void run()
			{
				RecreateDirectories.this.pack();
				SwingUtils.centerOnScreen(RecreateDirectories.this);
				RecreateDirectories.this.setVisible(true);
			}
		});
	}

	public boolean isStopped(){return this.iStop;}
	
	private void layoutComponents()
	{
		// Set up application
		this.setSize(new Dimension(600, 600));

		JButton goBtn = new JButton(Jibs.getString("RecreateDirectories.6"));  //$NON-NLS-1$
		goBtn.addActionListener(new java.awt.event.ActionListener()
		{
			public void actionPerformed(java.awt.event.ActionEvent evt)
			{
				go();
			}
		});

      this.myStopBtn.addActionListener(new java.awt.event.ActionListener()
	    {
	       public void actionPerformed(java.awt.event.ActionEvent evt)
	       {
	       	RecreateDirectories.this.iStop = true;
	       }
	    });
            
   	
		JButton fileBtn = new JButton("...");  //$NON-NLS-1$
		fileBtn.addActionListener(new java.awt.event.ActionListener()
		{
			public void actionPerformed(java.awt.event.ActionEvent evt)
			{
				selectFile();
			}
		});

		JButton rootBtn = new JButton("...");  //$NON-NLS-1$
		rootBtn.addActionListener(new java.awt.event.ActionListener()
		{
			public void actionPerformed(java.awt.event.ActionEvent evt)
			{
				selectRoot();
			}
		});

		JPanelRows topPanel = new JPanelRows(FlowLayout.CENTER);
		JPanel aRow = topPanel.topRow();
		aRow.add(new JLabel(Jibs.getString("RecreateDirectories.9")));  //$NON-NLS-1$

		aRow = topPanel.nextRow();
		aRow.add(new JLabel(Jibs.getString("RecreateDirectories.10")));  //$NON-NLS-1$

	    aRow = topPanel.nextRow();
		aRow.add(new JLabel(Jibs.getString("RecreateDirectories.11")));  //$NON-NLS-1$
		aRow.add(new JLabel(Jibs.getString("RecreateDirectories.12")));  //$NON-NLS-1$
	      
	      aRow = topPanel.nextRow();
	      aRow.add(new JLabel(" "));   //$NON-NLS-1$
	      
	      aRow = topPanel.nextRow();
	      aRow.add(new JLabel(Jibs.getString("RecreateDirectories.14")));   //$NON-NLS-1$

	      aRow = topPanel.nextRow();
	      aRow.add(new JLabel(this.myReaderWriter.getFileName() + Jibs.getString("colon") + " "));   //$NON-NLS-1$ //$NON-NLS-2$
	      aRow.add(this.myFileName);
	      aRow.add(fileBtn);
	      
	      aRow = topPanel.nextRow();
	      aRow.add(new JLabel(" "));   //$NON-NLS-1$
	      
	      aRow = topPanel.nextRow();
	      aRow.add(new JLabel(Jibs.getString("RecreateDirectories.17")));   //$NON-NLS-1$

	      aRow = topPanel.nextRow();
	      aRow.add(new JLabel(Jibs.getString("RecreateDirectories.18") + Jibs.getString("colon") + " "));   //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
	      aRow.add(this.myRootDir);
	      aRow.add(rootBtn);

	      aRow = topPanel.nextRow();
	      aRow.add(goBtn);
	      aRow.add(this.myStopBtn);

	      aRow = topPanel.nextRow();
	      aRow.add(this.myProgress);
	      
		this.setContentPane(topPanel);
	}

	void selectFile()
	{
		JFileChooserExtra chooser = new JFileChooserExtra(clsPrefs.get(LAST_FILE, null));
		chooser.setSaveName(APP_NAME, this.myReaderWriter.getFileName()); 
		chooser.setDialogTitle(Jibs.getString("RecreateDirectories.20") + this.myReaderWriter.getFileName());  //$NON-NLS-1$
		chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
		chooser.setMultiSelectionEnabled(false);
		chooser.addChoosableFileFilter(new JibsTreeFileFilter());

		int returnVal = chooser.showOpenDialog(this);
		if (returnVal == JFileChooser.APPROVE_OPTION)
		{
			File picked = chooser.getSelectedFile();
			if (picked != null)
			{
				this.myFileName.setText(picked.toString());
				try
				{
					clsPrefs.put(LAST_FILE, picked.toString());
					clsPrefs.flush();
				}
				catch (Exception ex)
				{
					ex.printStackTrace();
				}
			}
		}
	}

	void selectRoot()
	{
		JFileChooserExtra chooser = new JFileChooserExtra(clsPrefs.get(LAST_ROOT, null));
		chooser.setSaveName(APP_NAME, Jibs.getString("RecreateDirectories.21"));  //$NON-NLS-1$
		chooser.setApproveButtonText(Jibs.getString("RecreateDirectories.22")); //$NON-NLS-1$
		chooser.setDialogTitle(Jibs.getString("RecreateDirectories.23"));  //$NON-NLS-1$
		chooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
		chooser.setMultiSelectionEnabled(false);

		int returnVal = chooser.showOpenDialog(this);
		if (returnVal == JFileChooser.APPROVE_OPTION)
		{
			File picked = chooser.getSelectedFile();
			if (picked != null)
			{
				this.myRootDir.setText(picked.toString());
				try
				{
					clsPrefs.put(LAST_ROOT, picked.toString());
					clsPrefs.flush();
				}
				catch (Exception ex)
				{
					ex.printStackTrace();
				}
			}
		}
	}

	void go()
	{
		String fileName = this.myFileName.getText();
		//System.out.println("RecreateDirectories Selected File: " + fileName);
		File treeFile = new File(fileName);
		if (!treeFile.exists())
		{
	         JOptionPane.showMessageDialog(
	            this,
	            Jibs.getString("RecreateDirectories.24"),  //$NON-NLS-1$
	            Jibs.getString("RecreateDirectories.25"),  //$NON-NLS-1$
	            JOptionPane.ERROR_MESSAGE);
	         return;
		}
		
		String rootString = this.myRootDir.getText();
		File rootFile = new File(rootString);
		if (!rootFile.exists())
		{
	         JOptionPane.showMessageDialog(
	            this,
	            Jibs.getString("RecreateDirectories.26"),  //$NON-NLS-1$
	            Jibs.getString("RecreateDirectories.27"),  //$NON-NLS-1$
	            JOptionPane.ERROR_MESSAGE);
	         return;
		}
		
		//System.out.println("RootString: " + rootString);
		
		if (this.iStop){return;}
		setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
		try
		{
			DirectoryTree tree = null;
			DTNReader reader = this.myReaderWriter.getReader(treeFile, tree, new Message());
			DirectoryTreeNode rootNode = reader.getRootNode();
			makeChildern(rootNode, rootString);
		}
		catch(Exception ex)
		{
			ex.printStackTrace();
	         JOptionPane.showMessageDialog(
	            this,
	            ex.getMessage(), 
	            Jibs.getString("RecreateDirectories.28"),  //$NON-NLS-1$
	            JOptionPane.ERROR_MESSAGE);
		}
		setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
		this.myProgress.setText(Jibs.getString("RecreateDirectories.29")); //$NON-NLS-1$
	}

	/**
	 * Recursively make all the child directories.
	 * @param node
	 * @param strip
	 * @throws Exception
	 */
	private void makeChildern(DirectoryTreeNode node, String rootString)
	throws Exception
	{
		if (this.iStop){return;}
		
		if (!rootString.endsWith("/") && !rootString.endsWith("\\")) //$NON-NLS-1$ //$NON-NLS-2$
		{
			rootString += File.separator;
		}
		
		Enumeration<?> kids = node.children();
		while (kids.hasMoreElements())
		{
			DirectoryTreeNode aChild = (DirectoryTreeNode)kids.nextElement();
			String name = aChild.getName();
			String dirName = rootString + name;
			//System.out.println("Name: " + name + " -- " + dirName);
			File dir = new File(dirName);
			dir.mkdirs();
			makeChildern(aChild, dirName);
		}    
	}

	// ===========================================================================
	// Other display functions
	// ===========================================================================
	class JibsTreeFileFilter extends FileFilter
	{
		@Override
		public  boolean accept(File f)
		{
			if (f == null){return false;}
			if (f.isDirectory()){return true;}
			String name = f.getName();
			if (name.startsWith("JibsTree", 0)) //$NON-NLS-1$
			{
				return true;
			}
			return false;
		}
		
		@Override
		public  String getDescription()
		{
			return "JibsTree"; //$NON-NLS-1$
		}
	}
	
	class Message implements ILoadingMessage
	{
	   private int myIncrement = 1;
	   public void setText(String msg)
	   {
		   RecreateDirectories.this.myProgress.setText(msg);
	   }
	   public void setMessage(String msg)
	   {
			// We don't care
	   }
	   public int getNextIncrement(){return this.myIncrement++;}
	}
}

// Add one for backslash.
// If myCurrentDirName == C:\gbroot\DCGS-N
// you need to strip for C:\gbroot\DCGS-N\

