/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
package org.gerhardb.jibs.viewer;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JEditorPane;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTabbedPane;
import javax.swing.ScrollPaneConstants;

import org.gerhardb.jibs.Jibs;
import org.gerhardb.jibs.viewer.frame.SortScreen;
import org.gerhardb.lib.dirtree.rdp.StatusBarManager.AutocopyFilenamePanel;
import org.gerhardb.lib.swing.JPanelRows;
import org.gerhardb.lib.swing.SwingUtils;

/**
 *
 * @author  Gerhard Beck
 */
public class ViewerPreferencesDialog
   extends javax.swing.JDialog
{
   SortScreen mySortScreen;
   VPDMainPanel myMainPanel;
   TreePanel myTreePanel;
   AutocopyFilenamePanel myAutocopyFilenamePanel;
   ExifPanel myExifPanel;

   //===========================================================================
   //                              Constructor
   //===========================================================================
   public ViewerPreferencesDialog(SortScreen owner, int index)
   {
      super(owner, Jibs.getString("ViewerPreferencesDialog.12"), true); //$NON-NLS-1$
      this.mySortScreen = owner;
      
      this.myMainPanel = new VPDMainPanel(this.mySortScreen);
      this.myTreePanel = new TreePanel();
      this.myAutocopyFilenamePanel = this.mySortScreen.getLST().getStatusBarManager().getAutocopyFilenamePanel();
      this.myExifPanel = new ExifPanel();
      
      JTabbedPane jTabbed = new JTabbedPane();
      jTabbed.addTab(Jibs.getString("ViewerPreferencesDialog.14"), this.myMainPanel.setUpScreen()); //$NON-NLS-1$
      jTabbed.addTab(Jibs.getString("ViewerPreferencesDialog.15"), this.mySortScreen.getPathManager().directoryPanel()); //$NON-NLS-1$

      // Set up content pane
      JPanel thePanel = new JPanel(new BorderLayout());
      thePanel.add(jTabbed, BorderLayout.CENTER);
      thePanel.add(getOkCancelPanel(), BorderLayout.SOUTH);
      this.setContentPane(thePanel);     
      
      this.pack();

      // Add these later to make sure it does not increase the size.
      jTabbed.addTab(Jibs.getString("ViewerPreferencesDialog.11"), this.myTreePanel);  //$NON-NLS-1$
      jTabbed.addTab("Other", this.myAutocopyFilenamePanel);  
      jTabbed.addTab("EXIF", this.myExifPanel); //$NON-NLS-1$

      jTabbed.setSelectedIndex(index);
      
      SwingUtils.centerOnScreen(this);
      this.setVisible(true);
   }

   void save()
   {
      this.myMainPanel.save();
      this.myExifPanel.save();  
      this.myAutocopyFilenamePanel.save();
      // Do tree panel last in case user bails during reload of nodes.
      this.myTreePanel.save();
      ViewerPreferences.flush();
   	super.setVisible(false);
      super.dispose();   	
      this.mySortScreen.repaint();
   }
   
   void cancel()
   {
   	super.setVisible(false);
      super.dispose();   	
   }
   
  	private JPanel getOkCancelPanel()
	{
      JButton btnSave = new JButton(Jibs.getString("save")); //$NON-NLS-1$
      btnSave.addActionListener(
         new java.awt.event.ActionListener()
      {
         public void actionPerformed(java.awt.event.ActionEvent evt)
         {
         	ViewerPreferencesDialog.this.save();
         }
      });

      JButton btnCancel = new JButton(Jibs.getString("cancel")); //$NON-NLS-1$
      btnCancel.addActionListener(
         new java.awt.event.ActionListener()
      {
         public void actionPerformed(java.awt.event.ActionEvent evt)
         {
            ViewerPreferencesDialog.this.cancel();
         }
      });

      JPanel okPanel = new JPanel();
      okPanel.add(btnSave);
      okPanel.add(btnCancel);
      return okPanel;
   }

   //===========================================================================
   //                          Tree Panel
   //===========================================================================

   class TreePanel extends JPanelRows
   {
      JCheckBox myCountImagesBox = new JCheckBox(
      		Jibs.getString("ViewerPreferencesDialog.10")); //$NON-NLS-1$
      boolean priorCountTarget = ViewerPreferences.countImageFilesInNodes();
     
		TreePanel()
   	{  
	      this.myCountImagesBox.setSelected(ViewerPreferences.countImageFilesInNodes());
	      
	      this.myCountImagesBox.addActionListener
         (
            new ActionListener()
            {
               public void actionPerformed(ActionEvent ae)
               {
               	// First pass - make sure user wants to do this
               	if (TreePanel.this.myCountImagesBox.isSelected())
               	{
             			// Only warn if a change was made.
            			if (ViewerPreferences.countImageFilesInNodes() != true)
            			{           			
	              			  int picked = JOptionPane.showConfirmDialog(TreePanel.this,
	              					Jibs.getString("ViewerPreferencesDialog.73"),  //$NON-NLS-1$
	              					Jibs.getString("ViewerPreferencesDialog.60"), JOptionPane.OK_CANCEL_OPTION); //$NON-NLS-1$
	            	    	  if (picked == JOptionPane.CANCEL_OPTION)
	            	    	  {
	            	    		  // Turn check box off.  
	            	    		  TreePanel.this.myCountImagesBox.setSelected(false);
	            	    	  }      		
            			}
               	}               	
               }
            }
         );
	      
	      JPanel aRow = super.topRow();
	      aRow.add(this.myCountImagesBox);
   	}
   	
		void save()
		{
			ViewerPreferences.setCountImageFilesInNodes(this.myCountImagesBox.isSelected());
			ViewerPreferences.flush();
			if (this.priorCountTarget != this.myCountImagesBox.isSelected())
			{
				ViewerPreferencesDialog.this.mySortScreen.getLST().getTree().setCountTargetFiles(this.myCountImagesBox.isSelected());
				ViewerPreferencesDialog.this.mySortScreen.getTreeManager().reloadAllNodes();
			}
		}
   }
   
   //===========================================================================
   //                          Other Panel
   //===========================================================================

   /*
   class ShowTextPanel extends JPanelRows
   {
       JCheckBox myShowText = new JCheckBox("Show text files"); 
      
       ShowTextPanel()
   	{
      	myShowText.setSelected(ImageFactory.getImageFactory().isTextFilesShowingPreferenceSet());
	      
	      //JPanel aRow = super.topRow();
		      
	      // **ShowText** Attempt
	      //aRow = super.nextRow();
	      //aRow.add(myShowText);
   	}
   	
		void save()
		{
      	// Update preferences and then make sure they show up.
      	ImageFactory.getImageFactory().setTextFilesShowingPreference(myShowText.isSelected());   
   		mySortScreen.getScroller().reloadScroller();   	
		}
   }
   */

   //===========================================================================
	//                          EXIF Panel
	//===========================================================================

	class ExifPanel extends JPanel
	{
		JCheckBox showExifPanelChkBox = new JCheckBox(
				"Show EXIF information on left of picture on main screen (restart JIBS to see change)"); //$NON-NLS-1$
		
		JCheckBox showExifWarningChkBox = new JCheckBox(
		"Show EXIF warnings on save"); //$NON-NLS-1$

		ExifPanel()
		{
			super(new BorderLayout());
			this.showExifPanelChkBox.setSelected(ViewerPreferences.isExifPanelShowing());
			this.showExifWarningChkBox.setSelected(ViewerPreferences.showExifWarning());	

			JEditorPane editorPane = new JEditorPane();
			editorPane.setEditable(false);
			editorPane.setDisabledTextColor(Color.black);
			editorPane.setEditorKit(JEditorPane.createEditorKitForContentType(
					Jibs.getString("ViewerPreferencesDialog.96"))); //$NON-NLS-1$
			editorPane.setText(getInfo());

			JScrollPane center = new JScrollPane(editorPane,
					ScrollPaneConstants.VERTICAL_SCROLLBAR_AS_NEEDED,
					ScrollPaneConstants.HORIZONTAL_SCROLLBAR_NEVER);

			super.add(this.showExifPanelChkBox, BorderLayout.NORTH);
			super.add(center, BorderLayout.CENTER);
			super.add(this.showExifWarningChkBox, BorderLayout.SOUTH);
		}

		void save()
		{
			ViewerPreferences.setExifPanelShowing(this.showExifPanelChkBox.isSelected());
			ViewerPreferences.setShowExifWarning(
					this.showExifWarningChkBox.isSelected());
		}

		private String getInfo()
		{
			StringBuffer myResults = new StringBuffer(3000);
			myResults.append("<html>"); //$NON-NLS-1$
			myResults.append("<b><i>What's the problem?</i></b><br>"); //$NON-NLS-1$
			myResults
					.append("The current save mechanism loses all picture metadata. " //$NON-NLS-1$
							+ "This will be fixed in an upcoming release; " //$NON-NLS-1$
							+ "however, its timing is dependent on finding a java image library " //$NON-NLS-1$
							+ "which will support saving EXIF data.<p>"); //$NON-NLS-1$
			myResults.append("<b><i>What's metadata?</i></b><br>"); //$NON-NLS-1$
			myResults
					.append("Camera's record extra information with pictures like the model of camera and the date and time of the picture. That is all metadata. Specifically, EXIF metadata.<br><br>"); //$NON-NLS-1$
			myResults
					.append("<b><i>Does losing the metadata affect the quality of my pictures?</i></b><br>"); //$NON-NLS-1$
			myResults.append("No.<p>"); //$NON-NLS-1$
			myResults
					.append("<b><i>Is there anything else I should know?</i></b><br>"); //$NON-NLS-1$
			myResults
					.append("Be sure do any renaming your pictures based on the EXIF metadata prior to saving.<p>"); //$NON-NLS-1$
			myResults
					.append("If you change pictures from the main viewer (rotate, grow, shrink), and then save the picture, the metadata (EXIF) information which your camera stored "); //$NON-NLS-1$
			myResults
					.append("with the picture will be lost.  This does not affect the "); //$NON-NLS-1$
			myResults
					.append("image quality, but could make it harder for some image "); //$NON-NLS-1$
			myResults
					.append("manipulation programs to automatically enhance your picture's "); //$NON-NLS-1$
			myResults
					.append("quality.  It also make it impossible to rename the file using "); //$NON-NLS-1$
			myResults
					.append("the EXIF information.  So do renames before doing rotations.<p>"); //$NON-NLS-1$
			myResults.append("</html>"); //$NON-NLS-1$
			return myResults.toString();
		}
	}   
}
