/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package org.gerhardb.jibs.viewer.contact;

import java.awt.*;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;
import java.util.ArrayList;
import java.util.prefs.Preferences;

import javax.swing.*;
import org.gerhardb.jibs.Jibs;
import org.gerhardb.lib.image.ImageFactory;
import org.gerhardb.lib.io.NoFiles;
import org.gerhardb.lib.print.*;
import org.gerhardb.lib.print.contactSheet.*;
import org.gerhardb.lib.scroller.Scroller;
import org.gerhardb.lib.swing.SwingUtils;
import org.gerhardb.lib.util.Icons;
import org.gerhardb.lib.util.startup.AppStarter;

/**
 * ContactSheetDisplay
 */
public class ContactSheetDisplay
{
	public static final int SHOW_DIALOG = 0;
	public static final int SHOW_PREVIEW = 1;
	//public static final int SHOW_PRINT = 2;
	
	public static final int COLOR_FOR_PRINTING = 0;
	public static final int COLOR_FOR_FILE_SAVE = 1;
	
   static final String DEFAULT_TITLE = "Contact Sheet/Thumbnails"; //$NON-NLS-1$
   
	File[] mySelectedFiles; 
	boolean iShowOnlySelectedFiles;
   
   //JCheckBox myRepaginate = new JCheckBox(Jibs.getString("ContactSheetOptions.15")); //$NON-NLS-1$
   JTabbedPane myTabs = new JTabbedPane();
        
   JFrame myParentFrame;
   JFrame myFrame;
   JButton myCloseBtn;
   
   private ColorPanel colorPanel;
   private FontsPanel fontsPanel;
   private OptionsPanel optionsPanel;
   private RowColPanel rowColPanel;
   FilePanel filePanel;
   OtherPanel otherPanel;
     
	Scroller myScroller;

   Preferences myPrefs;

   /**
    */
   public ContactSheetDisplay(JFrame frame, File[] selectedFiles, boolean showOnlySelectedFiles, int showOnTop, Scroller scroller)
   {
   	this.iShowOnlySelectedFiles = showOnlySelectedFiles;
      this.myParentFrame = frame; 
      this.myScroller = scroller;
      this.myPrefs = Preferences.userRoot().node("/org/gerhardb/jibs/contactsheetdisplay");
      
      this.myFrame = new JFrame(DEFAULT_TITLE);
      this.myFrame.setIconImage(Icons.getIcon(Icons.JIBS_16).getImage());
      
      this.myCloseBtn = new JButton(Jibs.getString("close")); //$NON-NLS-1$
      this.myCloseBtn.addActionListener(new java.awt.event.ActionListener()
      {
         public void actionPerformed(java.awt.event.ActionEvent ae)
         {
            exit();
         }
      });
            
       this.myFrame.addWindowListener(new WindowAdapter()
         {
            @Override
				public void windowClosing(WindowEvent e)
            {
               exit();
            }
         });
      
      if (selectedFiles == null)
      {
       	this.mySelectedFiles = new File[0];   	   	      	
      }
      else
      {
       	this.mySelectedFiles = selectedFiles;   	   	     	
      }
      
   	this.colorPanel = new ColorPanel(this);
   	this.fontsPanel = new FontsPanel(this);
   	this.rowColPanel = new RowColPanel(this);
   	this.filePanel = new FilePanel(this);
   	this.otherPanel = new OtherPanel(this);
   	this.optionsPanel = new OptionsPanel(this);
   	
     	// If we already have files, there is no files tab and we really don't care.
	   this.myTabs.addTab(Jibs.getString("ContactSheetOptions.26"), this.filePanel);      //$NON-NLS-1$
      this.myTabs.addTab(Jibs.getString("ContactSheetOptions.27"), this.rowColPanel);      //$NON-NLS-1$
      this.myTabs.addTab(Jibs.getString("ContactSheetOptions.28"), this.optionsPanel);          //$NON-NLS-1$
      this.myTabs.addTab(Jibs.getString("ContactSheetOptions.29"), this.fontsPanel);       //$NON-NLS-1$
      this.myTabs.addTab(Jibs.getString("ContactSheetOptions.30"), this.colorPanel);    //$NON-NLS-1$
      this.myTabs.addTab("Other", this.otherPanel);   
      
      JPanel mainPanel = new JPanel(new BorderLayout());
      mainPanel.add(this.myTabs, BorderLayout.CENTER);      
      mainPanel.add(this.myCloseBtn, BorderLayout.SOUTH);   	
      this.myFrame.setContentPane(mainPanel);
      this.myFrame.pack();
      SwingUtils.centerOnScreen(this.myFrame);
      
      // Populate and show
     	getOptions();   	
      this.myFrame.setVisible(true);   	
   	   	
   	switch(showOnTop)
   	{
   	case SHOW_PREVIEW:
			PageLayout pl = PageLayout.makeDefault();
	   	ContactSheetPageable csp = makePageable(pl, true, true);
			new PageablePreviewer(csp,	false, this.filePanel.savePanel.makePDFbutton());
   		break;
   	}
   }
   
   // --------------------------------------------------------------------------
   // CSO
   // --------------------------------------------------------------------------
   
   
   File[] getFiles()
   throws Exception
	{
  		if (this.filePanel.myUseSelectedFiles.isSelected())
		{
			//System.out.println("Using Selected");
			return this.mySelectedFiles;
		}
		else if (this.filePanel.myUseAllFilesCurrentlyShownInBrowser.isSelected())
		{
			//System.out.println("Using All");
			return this.myScroller.getPictureFiles();
		}
		else if (this.filePanel.myUseDirectory.isSelected())
		{
			//System.out.println("File Recompute");
			this.myFrame.setCursor(Cursor
					.getPredefinedCursor(Cursor.WAIT_CURSOR));
			//System.out.println("================== Recomputing Files =========================");
			
			File treeRoot = new File(this.filePanel.myShowDirFld.getText());
			if (!treeRoot.exists())
			{
				String treeRootName = this.filePanel.myShowDirFld.getText();
				String msg = Jibs.getString("ContactSheetOptions.22") + Jibs.getString("colon") + " " //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
				+ treeRootName; //$NON-NLS-1$
				if (treeRootName == null || treeRootName.trim().length() == 0)
				{
					msg = "You need to select a directory to scan.";
				}
				this.myFrame.setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
	  			throw new Exception(msg);	  	
	  			/*
				JOptionPane.showMessageDialog(this.myFrame, msg, Jibs.getString("problem"), JOptionPane.ERROR_MESSAGE);
				this.myTabs.setSelectedIndex(0);
				return;
				*/
			}

			File[] rtnMe = null;
			if (treeRoot.isFile())
			{
				// Handle edge case where user selected a file not a directory.
				rtnMe =  new File[]{treeRoot};
			} 
			else
			{			
				if (this.filePanel.myWalkTree.isSelected())
				{
					ArrayList<File> directories = new ArrayList<File>(1000);	
					directories.add(treeRoot);
					addDirectoriesRecursive(directories, treeRoot);
					rtnMe = new File[directories.size()];
					directories.toArray(rtnMe);
				} 
				else
				{
					rtnMe = treeRoot.listFiles(ImageFactory.getImageFactory().getFilter());
				}
			}			
			this.myFrame.setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
			return rtnMe;
		}
		else
		{
			this.myFrame.setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
			throw new Exception("Internal Error: ContactSheetDisplay can't find list of files.");
		}
	}
   	
	HeaderFooterInfo getHeaderFooterInfo(boolean printNotSave)
	{
		HeaderFooterInfo rtnMe = new HeaderFooterInfo();
		
		rtnMe.headerLeft = this.optionsPanel.myHeaderLeft.getText();
		rtnMe.headerCenter = this.optionsPanel.myHeader.getText();
		rtnMe.headerRight = this.optionsPanel.myHeaderRight.getText();	
		
		rtnMe.footerLeft = this.optionsPanel.myFooterLeft.getText();
		rtnMe.footerCenter = this.optionsPanel.myFooter.getText();
		rtnMe.footerRight = this.optionsPanel.myFooterRight.getText();	
		
		rtnMe.headerFontSize = this.fontsPanel.getHeaderFontSize();
		rtnMe.footerFontSize = this.fontsPanel.getFooterFontSize();
		rtnMe.headerFontFamily = this.fontsPanel.myHeaderFont.getModel().getSelectedItem().toString();
		rtnMe.footerFontFamily = this.fontsPanel.myFooterFont.getModel().getSelectedItem().toString();
		rtnMe.showLines = this.optionsPanel.myShowLines.isSelected();
		
		if (printNotSave)
		{
			rtnMe.textColor = this.colorPanel.myPrintTextColor;
		}
		else
		{
			rtnMe.textColor = this.colorPanel.mySaveTextColor;
		}
		
		return rtnMe;
	}

	ContactSheetInfo getContactSheetInfo(boolean printNotSave)
	{
		ContactSheetInfo rtnMe = new ContactSheetInfo();

		rtnMe.rows = this.rowColPanel.myRowSlider.getValue();
		rtnMe.cols = this.rowColPanel.myColSlider.getValue();
		rtnMe.showPictureName = this.rowColPanel.myShowFileName.isSelected();
		rtnMe.showPictureNumber = this.rowColPanel.myShowPicCount.isSelected();
		rtnMe.showPictureSize = this.rowColPanel.myShowFileSize.isSelected();
		rtnMe.frameSize = this.rowColPanel.myFrameSlider.getValue();
		rtnMe.shadowSize = this.rowColPanel.myShadowSlider.getValue();
		rtnMe.resizeType = (ContactSheetInfo.ResizeType)this.rowColPanel.myResizeType.getSelectedItem();
		rtnMe.resizeDPI = (ContactSheetInfo.ResizeDPI)this.rowColPanel.myResizeDPI.getSelectedItem();
		
		// Fonts Tab
		rtnMe.pictureFontSize = this.fontsPanel.getPictureFontSize();
		rtnMe.pictureFontFamily = this.fontsPanel.myPictureFont.getModel().getSelectedItem().toString();
		
		// Colors Tab
		if (printNotSave)
		{
			rtnMe.pictureTextColor = this.colorPanel.myPrintTextColor;
			rtnMe.foregroundColor = this.colorPanel.myPrintTextColor;
			rtnMe.backgroundColor = this.colorPanel.myPrintBackgroundColor;
		}
		else
		{
			rtnMe.pictureTextColor = this.colorPanel.mySaveTextColor;
			rtnMe.foregroundColor = this.colorPanel.mySaveTextColor;
			rtnMe.backgroundColor = this.colorPanel.mySaveBackgroundColor;
		}
		
		return rtnMe;
	}   
	
   //public Preferences getPreferences(){return this.myPrefs;}
	
   // --------------------------------------------------------------------------
   // Private methods
   // --------------------------------------------------------------------------
  
   void addDirectoriesRecursive(ArrayList<File> directories, File root)
   {
	   File[] someDirs = root.listFiles(NoFiles.NO_FILES);
	   // First add the directories
	   for(int i=0; i<someDirs.length; i++)
	   {
		   directories.add(someDirs[i]);
	   }
	   // Then look for subdirectories
	   for(int i=0; i<someDirs.length; i++)
	   {
	   	addDirectoriesRecursive(directories, someDirs[i]);
	   }   	
   }   

   void getOptions()
   {
    	this.colorPanel.lookupPrefences();
   	this.fontsPanel.lookupPrefences();
    	this.rowColPanel.lookupPrefences();
    	this.filePanel.lookupPrefences();
    	this.optionsPanel.lookupPrefences();
    	this.otherPanel.lookupPrefences();
   }
   
   void saveOptions()
   {
     	this.colorPanel.savePreferences();
    	this.fontsPanel.savePreferences();
    	this.rowColPanel.savePreferences();
    	this.filePanel.savePreferences();
    	this.optionsPanel.savePreferences();
    	this.otherPanel.savePreferences();
    }

   void exit()
   {
	   saveOptions();
	   try{this.myPrefs.flush();}catch(Exception ex){ex.printStackTrace();}
       this.myFrame.dispose();
      if (this.myParentFrame == null)
      {
         System.exit(0);
      }
   }
       
   ContactSheetPageable makePageable(PageLayout pl, boolean printNotSave, 
   		boolean getImagesInBackground)
   {
   	ContactSheetPageable rtnMe = null;
     	try
     	{
     	  	DisplayInfo info = new DisplayInfo(pl, printNotSave, getImagesInBackground);
     	  	rtnMe = new ContactSheetPageable(
    			info.fileList, 
    			info.showFilesNotDirectories, 
    			info.pageLayout, 
    			info.headerFooterInfo, 
    			info.contactSheetInfo, 
    			info.saveInfo,
    			info.useBackgroundThead);
     		rtnMe.setProgressBar(this.filePanel.myPicBar);
     	}
     	catch(Exception ex)
     	{
     		ex.printStackTrace();
			JOptionPane.showMessageDialog(this.myFrame, ex.getMessage(), Jibs.getString("problem"), JOptionPane.ERROR_MESSAGE);
     	}  	
     	return rtnMe;
   }
   
   /**
    * Returns saved information.
    * @param pl
    * @param printNotSave
    * @param getImagesInBackground
    * @return
    * @throws Exception
    */
   DisplayInfo getDisplayInfo(PageLayout pl, boolean printNotSave, boolean getImagesInBackground)
   throws Exception
   {
   	return new DisplayInfo(pl, printNotSave, getImagesInBackground);
   }
   
   class DisplayInfo
   {
     	boolean showFilesNotDirectories; 
     	File[] fileList;
 		PageLayout pageLayout;
		HeaderFooterInfo headerFooterInfo;
		ContactSheetInfo contactSheetInfo;
		SaveInfo saveInfo;
		boolean useBackgroundThead;
		
	  	DisplayInfo(PageLayout pl, boolean printNotSave, boolean getImagesInBackground)
   	throws Exception
	   {  		
	 		this.pageLayout = pl;
			this.headerFooterInfo = getHeaderFooterInfo(printNotSave);
			this.contactSheetInfo = getContactSheetInfo(printNotSave);
			this.saveInfo = ContactSheetDisplay.this.filePanel.savePanel.getSaveInfo();
			this.useBackgroundThead = getImagesInBackground;
	   	//---------------------------------------------------------------------
			if (ContactSheetDisplay.this.iShowOnlySelectedFiles)
	  		{
	  			this.showFilesNotDirectories = true;
	  			this.fileList = ContactSheetDisplay.this.mySelectedFiles; 
	  			if (this.fileList.length == 1)
	  			{
	  				this.contactSheetInfo.rows = 1;
	  				this.contactSheetInfo.cols = 1;
	  			}
	  		}
	  		else
	  		{
		  		this.showFilesNotDirectories = ContactSheetDisplay.this.filePanel.showFilesNotDirectories();   	
		   	// If we aren't showing files, we will try to show directories.
		   	// But, if we don't walk the directories, we are just getting
		   	// a file list from the root directory meaning we are really 
		   	// showing files.
		   	if (!this.showFilesNotDirectories && !ContactSheetDisplay.this.filePanel.myWalkTree.isSelected())
		   	{
		   		this.showFilesNotDirectories = true;
		   	}	   		
		   	this.fileList = ContactSheetDisplay.this.getFiles();
	  		}
	   }
   }
   
   File getFileToSavePDFto()
   {
   	String file = this.filePanel.savePanel.mySavePdfFld.getText();
   	File rtnMe = new File(file);
   	if (rtnMe.getParentFile().exists())
   	{
   		return rtnMe;
   	}
   	return null;
   }
   
   // **************************************************************************
   // MAIN
   // **************************************************************************
   public static void main(String[] args)
   {
		AppStarter.startUpApp(args, "org.gerhardb.jibs.Jibs", true);
   	try
   	{
   		File dir = new File("D:/testpics/Fixed Files/contactSheet/a"); //$NON-NLS-1$
   		new ContactSheetDisplay(null, dir.listFiles(), true, ContactSheetDisplay.SHOW_DIALOG, new Scroller());
   		//new ContactSheetDisplay(null, dir.listFiles(),ContactSheetDisplay.SHOW_PREVIEW, new Scroller());
   	}
   	catch(Exception ex)
   	{
   		ex.printStackTrace();
   	}
   }
}
