/*
 * Copyright (c) 2007 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.gerhardb.jibs.viewer.frame;

import ij.IJ;
import ij.ImageJ;

import java.awt.Cursor;
import java.awt.event.*;
import java.io.File;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JList;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import org.gerhardb.jibs.Jibs;
import org.gerhardb.jibs.LicenseBox;
import org.gerhardb.jibs.camera.Camera;
import org.gerhardb.jibs.dirtools.DirectoryCopy;
import org.gerhardb.jibs.dirtools.DirectoryDedup;
import org.gerhardb.jibs.diskUsage.JibsDiskUsage;
import org.gerhardb.jibs.games.missingTile.MissingTile;
import org.gerhardb.jibs.optimizer.Optimizer;
import org.gerhardb.jibs.textCompare.TextCompareGUI;
import org.gerhardb.jibs.textPad.TextPad;
import org.gerhardb.jibs.textPad.rankPad.RankPad;
import org.gerhardb.jibs.textPad.ranker.AppRanker;
import org.gerhardb.jibs.util.*;
import org.gerhardb.jibs.viewer.IFrame;
import org.gerhardb.jibs.viewer.ViewerPreferences;
import org.gerhardb.jibs.viewer.contact.*;
import org.gerhardb.lib.print.PageableUtils;
import org.gerhardb.lib.scroller.ScrollerUtils;
import org.gerhardb.lib.util.ActionHelpers;
import org.gerhardb.lib.util.Icons;
import org.gerhardb.lib.util.app.AboutBox;
import org.gerhardb.lib.util.app.Info;
import org.gerhardb.lib.util.startup.*;

/**
 * Trying to remove clutter from main SortScreen and support
 * internationalized accelerated iconized menus.
 */
public class ViewerActions extends ActiveActions
{
	SortScreen mySortScreen;
	SaverActions mySaverActions;

	public ViewerActions(SortScreen ss)
	{
		this.mySortScreen = ss;
		this.mySaverActions = new SaverActions(this.mySortScreen);
		super.initialize();
	}
	

	@Override
	public Action getAction(String menu, String item)
	{
		// Menus orginally set up this way...
		menu = "SortScreen.menu" + "." + menu; //$NON-NLS-1$
		return super.getAction(menu, item);
	}
	
	/**
	 * Points to class in here with factory default options.
	 * Viewer can change these, then revert to factory defaults.
	 * @see org.gerhardb.lib.util.startup.ActiveActions#getStartupActions()
	 */
	@Override
	protected FactoryDefaultActions getFactoryDefaultActions()
	{
		return new ViewerStartupActions();
	}
	
	public SaverActions getSaverActions()
	{
		return this.mySaverActions;
	}
	
	// ==========================================================================
	// Starter Actions and Defaults
	// ==========================================================================
	
	/**
	 * Factory Default options.
	 */
	private class ViewerStartupActions extends FactoryDefaultActions
	{
		public ViewerStartupActions()
		{
			// These just click on the toolbar but have the menu accelerators
			addStartupAction("SortScreen.menu","file", "repeat", new RepeatAction(), Icons.getIcon(Icons.REPEAT)); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","file", "park", new ParkAction(), Icons.getIcon(Icons.PARK)); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","file", "trash", new DeleteAction(), Icons.getIcon(Icons.DELETE)); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","file", "directoryList", new DirectoryListAction(), Icons.getIcon(Icons.DIRECTORY_LIST)); //$NON-NLS-1$ //$NON-NLS-2$
			
			addStartupAction("SortScreen.menu","contact", "dir", new ShowContactSheetDirAction(), null); //$NON-NLS-1$ //$NON-NLS-2$
			
			addStartupAction("SortScreen.menu","file", "openview", new ViewDirectoryAction(), null); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","file", "opentree", new TreeDirectoryAction(), null); //$NON-NLS-1$ //$NON-NLS-2$
	      try
	      {
	      	if (AppStarter.isJava6Enabled())
	      	{
		      	Action defaultFileViewerAction = ScrollerUtils.getDefaultFileViewerAction(ViewerActions.this.mySortScreen.getScroller());
		      	if (defaultFileViewerAction != null)
		      	{
		      		addStartupAction("SortScreen.menu","file", "defaultFileView", defaultFileViewerAction, null); //$NON-NLS-1$ //$NON-NLS-2$
		      	}
	      	}
	      }
	      catch(NoClassDefFoundError ex)
	      {
	      	ex.printStackTrace();
	      }			
			addStartupAction("SortScreen.menu","file", "empty", new EmptyTrashAction(), null); //$NON-NLS-1$ //$NON-NLS-2$	
				
			addStartupAction("SortScreen.menu","file", "imagej", new ImageJAction(), null); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","file", "save", ViewerActions.this.mySaverActions.getSavePictureAction(), Icons.getIcon(Icons.SAVE)); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","file", "saveas", ViewerActions.this.mySaverActions.getSaveAsPictureAction(), Icons	.getIcon(Icons.SAVE_AS)); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","file", "rename", new RenameAction(), Icons.getIcon(Icons.RENAME)); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","file", "preview", new PreviewAction(), null); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","file", "exit", new ExitAction(), null); //$NON-NLS-1$ //$NON-NLS-2$

			addStartupAction("SortScreen.menu","edit", "undo", new UndoAction(), Icons.getIcon(Icons.UNDO)); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","edit", "redo", new RedoAction(), Icons.getIcon(Icons.REDO)); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","edit", "copy", new CopyAction(), null); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","edit", "all", new SelectAllAction(), null); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","edit", "clear", new ClearSelectionAction(), Icons.getIcon(Icons.CLEAR)); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","edit", "right", ViewerActions.this.mySaverActions.getRotateRightAction(), Icons.getIcon(Icons.ROTATE_RIGHT)); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","edit", "left", ViewerActions.this.mySaverActions.getRotateLeftAction(), Icons.getIcon(Icons.ROTATE_LEFT)); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","edit", "shrink", ViewerActions.this.mySaverActions.getShrinkAction(), Icons.getIcon(Icons.SHRINK)); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","edit", "grow", ViewerActions.this.mySaverActions.getGrowAction(), Icons.getIcon(Icons.GROW)); //$NON-NLS-1$ //$NON-NLS-2$

			addStartupAction("SortScreen.menu","view", "reload", new ReloadAction(), Icons.getIcon(Icons.REFRESH)); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","view", "back", new BackAction(), Icons.getIcon(Icons.BACK)); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","view", "forward", new ForwardAction(), Icons.getIcon(Icons.FORWARD)); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","view", "fullmovie", new FullMovieAction(), Icons.getIcon(Icons.FULL_SCREEN_MOVIE)); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","view", "movie", new MovieAction(), Icons.getIcon(Icons.MOVIE)); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","view", "full", new FullAction(), Icons.getIcon(Icons.FULL_SCREEN)); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","view", "comic", new ComicAction(), Icons.getIcon(Icons.COMIC_SCREEN)); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","view", "sorter", new SorterAction(), null); //$NON-NLS-1$ //$NON-NLS-2$
						
			addStartupAction("SortScreen.menu","tools", "group", new ShowOptimizerAction(), null); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","tools", "camera", new ShowCameraAction(), null); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","tools", "splitter", new ShowSplitterAction(), null); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","tools", "concatenater", new ShowConcatenatorAction(), null); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","tools", "autoconcat", new ShowAutoConcatenatorAction(), null); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","tools", "recreateDirs", new RecreateDirsAction(), null); //$NON-NLS-1$ //$NON-NLS-2$ 
			addStartupAction("SortScreen.menu","tools", "checkSumChecker", new CheckSumCheckerAction(), null); //$NON-NLS-1$ //$NON-NLS-2$ 
			addStartupAction("SortScreen.menu","tools", "dirTreeList", new ListDirectoryTreeAction(), null); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","tools", "diskUsage", new DiskUsageAction(), null); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","tools", "options", new ShowSetupOptionsAction(), null); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","tools", "menu", new ShowSetupMenuAction(), null); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","tools", "treestats", new ShowTreeStatsAction(), null); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","tools", "dirCopy", new DirCopyAction(), null); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","tools", "dirDedup", new DirDedupAction(), null); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","tools", "renameSubdirsNames", new RenameSubdirsNamesAction(), null); //$NON-NLS-1$ //$NON-NLS-2$ 
			
			addStartupAction("SortScreen.menu","text", "textPad", new TextPadAction(), null); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","text", "rankPad", new RankPadAction(), null); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","text", "textRank", new TextRankerAction(), null); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","text", "textDedup", new TextDedupAction(), null); //$NON-NLS-1$ //$NON-NLS-2$
			
			addStartupAction("SortScreen.menu","accelerators", "focus", new SwapFocusAction(), null); //$NON-NLS-1$ //$NON-NLS-2$
			
			addStartupAction("SortScreen.menu","toys", "tile", new ShowTileGameAction(), null); //$NON-NLS-1$ //$NON-NLS-2$
			
			addStartupAction("SortScreen.menu","help", "help", new HelpAction(), Icons.getIcon(Icons.HELP)); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","help", "about", new AboutAction(), Icons.getIcon(Icons.ABOUT)); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","help", "license", new LicenseAction(), null); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","help", "info", new InfoAction(), null); //$NON-NLS-1$ //$NON-NLS-2$
		}
	}
	
	class DeleteAction extends AbstractAction
	{
		DeleteAction()
		{
			super(null, Icons.getIcon(Icons.DELETE));
		}

		public void actionPerformed(ActionEvent ae)
		{
			ViewerActions.this.mySortScreen.getRDPmanager().clickTrash();
		}
	}

	class ParkAction extends AbstractAction
	{
		ParkAction()
		{
			super(null, Icons.getIcon(Icons.PARK));
		}

		public void actionPerformed(ActionEvent ae)
		{
			ViewerActions.this.mySortScreen.getRDPmanager().clickPark();
		}
	}

	class RepeatAction extends AbstractAction
	{
		RepeatAction()
		{
			super(null, Icons.getIcon(Icons.REPEAT));
		}

		public void actionPerformed(ActionEvent ae)
		{
			ViewerActions.this.mySortScreen.getRDPmanager().clickRepeat();			
		}
	}

	class DirectoryListAction extends AbstractAction
	{
		DirectoryListAction()
		{
			super(null, Icons.getIcon(Icons.DIRECTORY_LIST));
		}

		public void actionPerformed(ActionEvent ae)
		{
			ViewerActions.this.mySortScreen.getRDPmanager().clickDirectoryList();			
		}
	}

	// ==========================================================================
	// Helper Methods
	// ==========================================================================
		
	public JButton makeSaveButton()
	{
		JButton rtnMe = getToolBarButton("file", "save"); //$NON-NLS-1$ //$NON-NLS-2$
		rtnMe.setToolTipText(Jibs.getString("Actions.94")); //$NON-NLS-1$
		return rtnMe;
	}
	
	@Override
	public JButton getToolBarButton(String menu, String item)
	{
		JButton rtnMe = getAccelleratedToolBarButton(menu, item);
		return rtnMe;
	}

	// ==========================================================================
	// FILE Menu
	// ==========================================================================
	JMenu getFileMenu()
	{
		// Set up sub menus
		JMenu move = ActionHelpers.makeMenu("move"); //$NON-NLS-1$
		move.add(makeMenuItem("file", "repeat")); //$NON-NLS-1$ //$NON-NLS-2$
		move.add(makeMenuItem("file", "park")); //$NON-NLS-1$ //$NON-NLS-2$
		move.add(makeMenuItem("file", "trash")); //$NON-NLS-1$ //$NON-NLS-2$
		move.add(makeMenuItem("file", "directoryList")); //$NON-NLS-1$ //$NON-NLS-2$
		
		JMenu editor = new JMenu("Edit Picture");
		Action action = getAction("file", "defaultFileView");
		if (action != null)
		{
			editor.add(new JMenuItem(action));
		}
		editor.add(makeMenuItem("file", "imagej")); //$NON-NLS-1$ //$NON-NLS-2$
		
		// Get down to work
		JMenu menu = ActionHelpers.makeMenu("file"); //$NON-NLS-1$
		menu.add(makeMenuItem("file", "openview")); //$NON-NLS-1$ //$NON-NLS-2$
		menu.add(this.mySortScreen.getPathManager().getRecentDirMenu());
		menu.add(makeMenuItem("file", "opentree")); //$NON-NLS-1$ //$NON-NLS-2$
		menu.add(this.mySortScreen.getPathManager().getRecentTreeMenu());
		menu.add(editor);		
		menu.addSeparator();
		menu.add(move);
		menu.add(makeMenuItem("file", "empty")); //$NON-NLS-1$ //$NON-NLS-2$
		menu.add(makeMenuItem("contact", "dir")); //contact); //$NON-NLS-1$ //$NON-NLS-2$
		menu.addSeparator();
		menu.add(makeMenuItem("file", "save")); //$NON-NLS-1$ //$NON-NLS-2$
		menu.add(makeMenuItem("file", "saveas")); //$NON-NLS-1$ //$NON-NLS-2$
		menu.add(makeMenuItem("file", "rename")); //$NON-NLS-1$ //$NON-NLS-2$
		menu.addSeparator();
		menu.add(makeMenuItem("file", "preview")); //$NON-NLS-1$ //$NON-NLS-2$
		menu.addSeparator();
		menu.add(makeMenuItem("file", "exit")); //$NON-NLS-1$ //$NON-NLS-2$
		return menu;
	}
	class ViewDirectoryAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent e)
		{
			ViewerActions.this.mySortScreen.getFileListManager().askUserForViewDirectory(ViewerActions.this.mySortScreen);
		}
	}

	class TreeDirectoryAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent e)
		{
			ViewerActions.this.mySortScreen.getTreeManager().setTreeDirectory();
		}
	}
	
	class EmptyTrashAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent e)
		{
			ViewerActions.this.mySortScreen.getRDPmanager().emptyTrash();
		}
	}

	class PreviewAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent e)
		{
			ViewerActions.this.mySortScreen.myShow.getSaver().pageableAction(PageableUtils.PREVIEW);
		}
	}
	
	class ShowContactSheetDirAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent e)
		{
		    new ContactSheetDisplay(ViewerActions.this.mySortScreen, ViewerActions.this.mySortScreen.getFileList().getSelectedFiles(), 
		   		 false, ContactSheetDisplay.SHOW_DIALOG, ViewerActions.this.mySortScreen.getScroller());
		}
	}
	
	class ExitAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent e)
		{
			ViewerActions.this.mySortScreen.exit();
		}
	}

	class RecreateDirsAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent e)
		{
			new RecreateDirectories(ViewerActions.this.mySortScreen.getAppIDandTargetType());
		}
	}
	
	class TextRankerAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent e)
		{
			new AppRanker(false);
		}
	}
	
	class TextPadAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent e)
		{
			// Have to get this out of the AWTThread so the loading box won't hang.
			Runnable runMe = new Runnable()
			{
				public void run()
				{
					new TextPad(false);
				}
			};
			new Thread(runMe).start();	
		}
	}
	
	class RankPadAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent e)
		{
			// Have to get this out of the AWTThread so the loading box won't hang.
			Runnable runMe = new Runnable()
			{
				public void run()
				{
					new RankPad(false);
				}
			};
			new Thread(runMe).start();	
		}
	}
	
	class TextDedupAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent e)
		{
			new TextCompareGUI(false);
		}
	}
	
	class RenameSubdirsNamesAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent e)
		{
			new FileRenameWithDirName(false);
		}
	}

	class CheckSumCheckerAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent e)
		{
			new CheckSumChecker();
		}
	}

	class ListDirectoryTreeAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent e)
		{
			new ListDirectoryContents();
		}
	}
	
	class DiskUsageAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent e)
		{
			try
			{
				new JibsDiskUsage(false, ViewerActions.this.mySortScreen.getScroller());
			}
			catch(NoClassDefFoundError ex)
			{
	           JOptionPane.showMessageDialog(
	         		  ViewerActions.this.mySortScreen,
	                 Jibs.getString("Actions.91"), //$NON-NLS-1$
	                 Jibs.getString("Actions.90"), //$NON-NLS-1$
	                 JOptionPane.INFORMATION_MESSAGE );				
			}
			catch(UnsupportedClassVersionError ex)
			{
	           JOptionPane.showMessageDialog(
	         		  ViewerActions.this.mySortScreen,
	                 Jibs.getString("Actions.89"), //$NON-NLS-1$
	                 Jibs.getString("Actions.88"), //$NON-NLS-1$
	                 JOptionPane.INFORMATION_MESSAGE );				
			}
		}
	}

	// ==========================================================================
	// EDIT Menu
	// ==========================================================================

	JMenu getEditMenu()
	{
		JMenu menu = ActionHelpers.makeMenu("edit"); //$NON-NLS-1$
		menu.add(makeMenuItem("edit", "undo")); //$NON-NLS-1$ //$NON-NLS-2$
		menu.add(makeMenuItem("edit", "redo")); //$NON-NLS-1$ //$NON-NLS-2$
		menu.addSeparator();
		menu.add(makeMenuItem("edit", "copy")); //$NON-NLS-1$ //$NON-NLS-2$
		menu.add(makeMenuItem("edit", "all")); //$NON-NLS-1$ //$NON-NLS-2$
		menu.add(makeMenuItem("edit", "clear")); //$NON-NLS-1$ //$NON-NLS-2$
		menu.addSeparator();
		menu.add(makeMenuItem("edit", "right")); //$NON-NLS-1$ //$NON-NLS-2$
		menu.add(makeMenuItem("edit", "left")); //$NON-NLS-1$ //$NON-NLS-2$
		menu.add(makeMenuItem("edit", "shrink")); //$NON-NLS-1$ //$NON-NLS-2$
		menu.add(makeMenuItem("edit", "grow")); //$NON-NLS-1$ //$NON-NLS-2$
		return menu;
	}

	class UndoAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent ae)
		{
			ViewerActions.this.mySortScreen.getRDPmanager().getMoveManager().getUndoManager().undo();
			ViewerActions.this.mySortScreen.getRDPmanager().getMoveManager().updateUndoRedoButtons();
		}
	}

	class RedoAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent ae)
		{
			ViewerActions.this.mySortScreen.getRDPmanager().getMoveManager().getUndoManager().redo();
			ViewerActions.this.mySortScreen.getRDPmanager().getMoveManager().updateUndoRedoButtons();
		}
	}
	
	class CopyAction extends AbstractAction
	{

		public void actionPerformed(ActionEvent e)
		{
			ViewerActions.this.mySortScreen.myShow.getSaver().copy();
		}
	}	
	
	class SelectAllAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent e)
		{
			JList list = ViewerActions.this.mySortScreen.getFileList();
			list.clearSelection();
			list.addSelectionInterval(0, list.getModel().getSize() - 1);
			ViewerActions.this.mySortScreen.myShow.showPageFromScroller();
		}
	}

	class ClearSelectionAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent e)
		{
			JList list = ViewerActions.this.mySortScreen.getFileList();
			list.clearSelection();
			ViewerActions.this.mySortScreen.myShow.showPageFromScroller();
		}
	}

	// ==========================================================================
	// VIEW Menu
	// ==========================================================================
	JMenu getViewMenu(IFrame frame)
	{
		JCheckBox sorter = makeCheckBox("view", "sorter"); //$NON-NLS-1$ //$NON-NLS-2$
		sorter.setSelected(ViewerPreferences.isShowSorterShowing());
		
		JMenu menu = ActionHelpers.makeMenu("view"); //$NON-NLS-1$
		menu.add(makeMenuItem("view", "reload")); //$NON-NLS-1$ //$NON-NLS-2$
		menu.add(makeMenuItem("view", "back")); //$NON-NLS-1$ //$NON-NLS-2$
		menu.add(makeMenuItem("view", "forward")); //$NON-NLS-1$ //$NON-NLS-2$
		menu.addSeparator();
		menu.add(makeMenuItem("view", "full")); //$NON-NLS-1$ //$NON-NLS-2$
		menu.add(makeMenuItem("view", "fullmovie")); //$NON-NLS-1$ //$NON-NLS-2$
		menu.add(makeMenuItem("view", "movie")); //$NON-NLS-1$ //$NON-NLS-2$
		menu.add(makeMenuItem("view", "comic")); //$NON-NLS-1$ //$NON-NLS-2$
		menu.addSeparator();
		ViewerPreferences.makeViewMenu(frame, menu);
		menu.addSeparator();
		menu.add(sorter); 
		menu.addSeparator();
		return menu;
	}

	class ReloadAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent ae)
		{
			ViewerActions.this.mySortScreen.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
			ViewerActions.this.mySortScreen.myShow.reloadScroller();
			ViewerActions.this.mySortScreen.setCursor(Cursor.getDefaultCursor());
		}
	}

	class BackAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent ae)
		{
			ViewerActions.this.mySortScreen.getFileListManager().back();
		}
	}

	class ForwardAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent ae)
		{
			ViewerActions.this.mySortScreen.getFileListManager().forward();
		}
	}

	class FullAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent ae)
		{
			ViewerActions.this.mySortScreen.gotoFullScreen(ViewerPreferences.immediateShow());
		}
	}

	class FullMovieAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent ae)
		{
			System.out.println("FULL MOVIE ACTION PRESSED");
			ViewerActions.this.mySortScreen.gotoFullScreen(true);
		}
	}

	class MovieAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent ae)
		{
			ViewerActions.this.mySortScreen.mySlideShowManager.flipSlideShow(); 
		}
	}
	
	class ComicAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent ae)
		{
			ViewerActions.this.mySortScreen.gotoComicScreen();
		}
	}

	class SorterAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent ae)
		{
			ViewerActions.this.mySortScreen.mySorterPanel.flipSorter();
		}
	}
	
	// ==========================================================================
	// TOOLS Menu
	// ==========================================================================
	JMenu getToolsMenu()
	{
		JMenu textMenu = ActionHelpers.makeMenu("text"); //$NON-NLS-1$ //$NON-NLS-2$
		textMenu.add(makeMenuItem("text", "textPad")); //$NON-NLS-1$ //$NON-NLS-2$
		textMenu.add(makeMenuItem("text", "textDedup")); //$NON-NLS-1$ //$NON-NLS-2$
		textMenu.add(makeMenuItem("text", "textRank")); //$NON-NLS-1$ //$NON-NLS-2$
		textMenu.add(makeMenuItem("text", "rankPad")); //$NON-NLS-1$ //$NON-NLS-2$

		JMenu randomMenu = ActionHelpers.makeMenu("random"); //$NON-NLS-1$ //$NON-NLS-2$
		randomMenu.add(makeMenuItem("tools", "recreateDirs")); //$NON-NLS-1$ //$NON-NLS-2$
		randomMenu.add(makeMenuItem("tools", "dirCopy")); //$NON-NLS-1$ //$NON-NLS-2$
		randomMenu.add(makeMenuItem("tools", "dirDedup")); //$NON-NLS-1$ //$NON-NLS-2$
		randomMenu.add(makeMenuItem("tools", "dirTreeList")); //$NON-NLS-1$ //$NON-NLS-2$

		JMenu otherMenu = ActionHelpers.makeMenu("other"); //$NON-NLS-1$ //$NON-NLS-2$
		otherMenu.add(textMenu);
		otherMenu.add(randomMenu);
		otherMenu.add(makeMenuItem("tools", "diskUsage")); //$NON-NLS-1$ //$NON-NLS-2$
		otherMenu.add(makeMenuItem("tools", "checkSumChecker")); //$NON-NLS-1$ //$NON-NLS-2$
		otherMenu.add(makeMenuItem("tools", "renameSubdirsNames")); //$NON-NLS-1$ //$NON-NLS-2$

		JMenu setupMenu = ActionHelpers.makeMenu("setup"); //$NON-NLS-1$ //$NON-NLS-2$
		setupMenu.add(makeMenuItem("tools", "options")); //$NON-NLS-1$ //$NON-NLS-2$
		setupMenu.add(makeMenuItem("tools", "menu")); //$NON-NLS-1$ //$NON-NLS-2$
		
		JMenu splitMenu = ActionHelpers.makeMenu("split"); //$NON-NLS-1$ //$NON-NLS-2$
		splitMenu.add(makeMenuItem("tools", "splitter")); //$NON-NLS-1$ //$NON-NLS-2$
		splitMenu.add(makeMenuItem("tools", "concatenater")); //$NON-NLS-1$ //$NON-NLS-2$
		splitMenu.add(makeMenuItem("tools", "autoconcat")); //$NON-NLS-1$ //$NON-NLS-2$
		
		JMenu menu = ActionHelpers.makeMenu("tools"); //$NON-NLS-1$
		menu.add(makeMenuItem("contact", "dir")); //contact); //$NON-NLS-1$ //$NON-NLS-2$
		menu.add(makeMenuItem("tools", "group")); //$NON-NLS-1$ //$NON-NLS-2$
		menu.add(makeMenuItem("tools", "camera")); //$NON-NLS-1$ //$NON-NLS-2$
		menu.add(makeMenuItem("tools", "treestats")); //$NON-NLS-1$ //$NON-NLS-2$
		menu.add(splitMenu);
		menu.add(otherMenu);
		menu.add(setupMenu); 
		return menu;
	}

	JMenu getToysMenu()
	{
		JMenu toysMenu = ActionHelpers.makeMenu("toys"); //$NON-NLS-1$ //$NON-NLS-2$
		toysMenu.add(makeMenuItem("toys", "tile")); //$NON-NLS-1$ //$NON-NLS-2$		
		return toysMenu;
	}

	static ImageJ imageJ;
	class ImageJAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent e)
		{
				if (imageJ == null)
				{
					imageJ = new ImageJ();
				}
				File image = ViewerActions.this.mySortScreen.getScroller().getCurrentFile();
				IJ.open(image.getAbsolutePath());	
		}
	}

	class RenameAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent e)
		{
			ViewerActions.this.mySortScreen.getFileList().renameSomeFiles();
		}
	}

	class ShowOptimizerAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent e)
		{
			new Optimizer(false, ViewerActions.this.mySortScreen.getRDPplugins());
		}
	}

	class ShowCameraAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent e)
		{
			new Camera(false);
		}
	}

	class ShowSplitterAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent e)
		{
			new org.gerhardb.jibs.concatenater.Splitter(false);
		}
	}
	
	class ShowConcatenatorAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent e)
		{
			new org.gerhardb.jibs.concatenater.Concatenater(false);
		}
	}

	class ShowAutoConcatenatorAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent e)
		{
			new org.gerhardb.jibs.concatenater.AutoConcatenate(false);
		}
	}
	
	class ShowSetupOptionsAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent e)
		{
			// Show with tab 0 showing.
			ViewerActions.this.mySortScreen.myPopups.showSetup(0);
		}
	}
	
	class ShowSetupMenuAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent e)
		{
			// Java 6 option
			try
			{
	      	if (AppStarter.isJava6Enabled())
	      	{
	      		new KeyChangerDialog(ViewerActions.this.mySortScreen);
	      	}
			}
			catch(NoClassDefFoundError ex)
			{
	           JOptionPane.showMessageDialog(
	         		  ViewerActions.this.mySortScreen,
	                 Jibs.getString("Actions.85"), //$NON-NLS-1$
	                 Jibs.getString("Actions.84"), //$NON-NLS-1$
	                 JOptionPane.INFORMATION_MESSAGE );				
			}
			catch(UnsupportedClassVersionError ex)
			{
	           JOptionPane.showMessageDialog(
	         		  ViewerActions.this.mySortScreen,
	                 Jibs.getString("Actions.83"), //$NON-NLS-1$
	                 Jibs.getString("Actions.82"), //$NON-NLS-1$
	                 JOptionPane.INFORMATION_MESSAGE );				
			}
		}
	}

	class ShowTreeStatsAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent e)
		{
			ViewerActions.this.mySortScreen.getRDPmanager().showStats(ViewerActions.this.mySortScreen.getTree().getRootNode(), ViewerPreferences.countImageFilesInNodes());
		}
	}
	
	class DirCopyAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent e)
		{
			new DirectoryCopy(false);
		}
	}
	
	class DirDedupAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent e)
		{
			new DirectoryDedup(false);
		}
	}
	
	class ShowTileGameAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent e)
		{
			try
			{
				new MissingTile(false);
			}
			catch(NoClassDefFoundError ex)
			{
	           JOptionPane.showMessageDialog(
	         		  ViewerActions.this.mySortScreen,
	                 Jibs.getString("Actions.91"), //$NON-NLS-1$
	                 Jibs.getString("Actions.90"), //$NON-NLS-1$
	                 JOptionPane.INFORMATION_MESSAGE );				
			}
			catch(UnsupportedClassVersionError ex)
			{
	           JOptionPane.showMessageDialog(
	         		  ViewerActions.this.mySortScreen,
	                 Jibs.getString("Actions.89"), //$NON-NLS-1$
	                 Jibs.getString("Actions.88"), //$NON-NLS-1$
	                 JOptionPane.INFORMATION_MESSAGE );				
			}
		}
	}

	// ==========================================================================
	// Accelerators Menu
	// ==========================================================================
	class SwapFocusAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent e)
		{
			ViewerActions.this.mySortScreen.myShow.swapPictureFocus();
		}
	}
	
	// ==========================================================================
	// HELP Menu
	// ==========================================================================

	JMenu getHelpMenu()
	{
		JMenu acceleratorsMenu = ActionHelpers.makeMenu("accelerators"); //$NON-NLS-1$ //$NON-NLS-2$
		acceleratorsMenu.add(makeMenuItem("accelerators", "focus")); //$NON-NLS-1$ //$NON-NLS-2$

		JMenu menu = ActionHelpers.makeMenu("help"); //$NON-NLS-1$
		menu.add(makeMenuItem("help", "help")); //$NON-NLS-1$ //$NON-NLS-2$
		menu.add(makeMenuItem("help", "about")); //$NON-NLS-1$ //$NON-NLS-2$
		menu.add(acceleratorsMenu);
		menu.add(makeMenuItem("help", "license")); //$NON-NLS-1$ //$NON-NLS-2$
		menu.add(makeMenuItem("help", "info")); //$NON-NLS-1$ //$NON-NLS-2$
		return menu;
	}

	class HelpAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent e)
		{
			ViewerActions.this.mySortScreen.showHelp();
		}
	}

	class AboutAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent e)
		{
			new AboutBox(ViewerActions.this.mySortScreen);
		}
	}

	class InfoAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent e)
		{
			new Info(ViewerActions.this.mySortScreen, ViewerActions.this.mySortScreen.getShow());
		}
	}

	class LicenseAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent e)
		{
			new LicenseBox(ViewerActions.this.mySortScreen);
		}
	}
	
	// ==========================================================================
	// Main
	// ==========================================================================
	/*
	public static void main(String[] args)
	{
		String key = "A"; //$NON-NLS-1$
		int keyCode = getKeyCode(key);
		char keyChar = getKeyChar(keyCode);
		String keyString = getKeyString(keyCode);
		
		System.out.println("keyCode: " + keyCode); //$NON-NLS-1$
		System.out.println("keyChar: " + keyChar); //$NON-NLS-1$
		System.out.println("keyString: " + keyString); //$NON-NLS-1$
		System.out.println("back to code from char: " + getKeyCode(keyChar)); //$NON-NLS-1$
		for(int i = 0; i < 128; i++) 
		{
         //System.out.println(i + " = " + (char)i);
      }
		
		KeyStroke.getKeyStroke(keyChar, java.awt.event.InputEvent.SHIFT_MASK).toString();
		KeyStroke.getKeyStroke(keyChar, java.awt.event.InputEvent.ALT_MASK).toString();
		KeyStroke.getKeyStroke(keyChar, java.awt.event.InputEvent.CTRL_MASK).toString();
		KeyStroke.getKeyStroke(keyChar, java.awt.event.InputEvent.META_MASK).toString();
		
	}	
	*/
}
