/*
 * Copyright (c) 2007 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.gerhardb.jibs.viewer.shows.classic;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.image.BufferedImage;
import java.io.File;

import javax.swing.*;

import org.gerhardb.jibs.viewer.IFrame;
import org.gerhardb.jibs.viewer.sorter.TilePanel;

/**
 * JPanel implementation of showing a single image.
 */
public class SingleNoResizeScrollControl extends SingleNoResize
{
	int myOffsetWidth;
	int myOffsetHeight;

	private JScrollPane myScrollPane = 
		new JScrollPane(this, 
				ScrollPaneConstants.VERTICAL_SCROLLBAR_AS_NEEDED, //.VERTICAL_SCROLLBAR_ALWAYS, 
				ScrollPaneConstants.HORIZONTAL_SCROLLBAR_AS_NEEDED);
	
	// ==========================================================================
	// Constructor
	// ==========================================================================
	public SingleNoResizeScrollControl(IFrame mf)
	{
		super(mf);
	}

	// ==========================================================================
	// Overridden methods
	// ==========================================================================
	@Override
	public JComponent getImageDisplay()
	{
		return this.myScrollPane;
	}

	
	//public void paint(Graphics g)
	@Override
	public void showImage(Graphics g)
 	{
   	// Apparently, background color can only be set once per draw without
		// confusing SWING.
		Color backgroundColor = OK_COLOR; // DEFAULT COLOR

		BufferedImage showMe = this.mySaver.getCurrentImage();
		if (showMe == null)
		{
			showMe = this.myViewerFrame.getScroller().getBeyondBounds();
			if (showMe == null)
			{
				super.showNoPicture(g);
				return;
			}
			backgroundColor = EMPTY_COLOR;
		}

		//System.out.println("======================== SingleNoResizeScrollControl.paint() ============================");
		//System.out.println("Panel w h: " + super.getWidth() + " " + super.getHeight());
		//System.out.println("Raw Picture w h: " + showMe.getWidth() + " " + showMe.getHeight());

		// Remember for pic info dialog
		this.myShownWidth = showMe.getWidth();
		this.myShownHeight = showMe.getHeight();
		
		Graphics2D g2 = (Graphics2D) g;

		// Initial position for when there are no scroll bars.
		this.myOffsetWidth = (int) ((SingleNoResizeScrollControl.this.getWidth() - this.myShownWidth) * this.myOffsetWidthPercentage);
		this.myOffsetHeight = (int) ((SingleNoResizeScrollControl.this.getHeight() - this.myShownHeight) * this.myOffsetHeightPercentage);

		Dimension size = new Dimension(this.myShownWidth, this.myShownHeight);		
      // Update client's preferred size because the area taken up by the graphics has
      // gotten larger or smaller (if cleared).
		this.setPreferredSize(size);
      // Let the scroll pane know to update itself and its scrollbars.
		this.revalidate();
				
		// Now the the correct background color has been established,
		// use it.
		super.setBackground(backgroundColor);
		super.paintComponent(g); // First, fill with background color.

		// Draw the new image
		g2.drawImage(showMe, null, this.myOffsetWidth, this.myOffsetHeight);

		if (this.iNeedToRepositionScrollBars)
		{
			SwingUtilities.invokeLater(new Runnable() { public void run() { positionPicture(); } });
			this.iNeedToRepositionScrollBars = false;
		}
		
		if (this.myViewerFrame.getScroller().getCurrentIOImage() != null)
		{
	      File file = this.myViewerFrame.getScroller().getCurrentIOImage().getFile();
			TilePanel.clsTilePanel.clip(file, g2, super.getWidth(), super.getHeight());
		}
 
		this.myViewerFrame.statusCurrentPage();		
		//super.setBackground(OK_COLOR);

	}

	// ==========================================================================
	// IShow Implemenation
	// ==========================================================================


	void positionPicture()
	{								
		JViewport vp = this.myScrollPane.getViewport();
		Rectangle vpRect = vp.getViewRect();
		Dimension vpExtent = vp.getExtentSize();
		Dimension vpViewSize = vp.getViewSize();
		
		JScrollBar hBar = this.myScrollPane.getHorizontalScrollBar();
		JScrollBar vBar = this.myScrollPane.getVerticalScrollBar();	

		//System.out.println("======================== SingleNoResizeScrollControl.positionPicture() ============================");
		
		if (vpExtent.width == vpViewSize.width)
		{
			// If there are NO bars
			//System.out.println("NO width bars " + myShownWidth);
			//offsetWidth = (int) ((SingleNoResizeScrollControl.this.getWidth() - myShownWidth) * myOffsetWidthPercentage);
			//myOffsetWidth = (int)((vpRect.getWidth() - myShownWidth) * myOffsetWidthPercentage);
		}
		else
		{
			// If there ARE bars
			//System.out.println("WIDTH bars");
			this.myOffsetWidth = (int) ((hBar.getMaximum() - (int)vpRect.getWidth()) * this.myOffsetWidthPercentage);
		}
		
		if (vpExtent.height == vpViewSize.height)
		{
			// If there are NO bars
			//System.out.println("NO height bars");
			//myOffsetHeight = (int) ((vpRect.getHeight() - myShownHeight) * myOffsetHeightPercentage);
		}
		else
		{
			// If there ARE bars
			//System.out.println("HEIGHT bars");
			this.myOffsetHeight = (int) ((vBar.getMaximum() - (int)vpRect.getHeight()) * this.myOffsetHeightPercentage);
		}
		
		Point newPosition = new Point(this.myOffsetWidth, this.myOffsetHeight);
		vp.setViewPosition(newPosition);
		
		/*
		System.out.println("vpRect: " + vpRect);
		System.out.println("Rect x/w: " + vpRect.x + "/" + vpRect.width + " hBar: " +  hBar.getMaximum() + " offset: " + myOffsetWidth);
		System.out.println("Rect y/h: " + vpRect.y + "/" + vpRect.height + " vBar: " +  vBar.getMaximum() + " offset: " + myOffsetHeight);
		
		System.out.println("Extent: " + vp.getExtentSize());
		System.out.println("View Size: " + vp.getViewSize());
		*/
	}
	
	// ==========================================================================
	// Inner Classes
	// ==========================================================================

	@Override
	KeyListener getKeyListener()
	{
		return new KeyActions();
	}
	
	private class KeyActions implements KeyListener
	{
		protected KeyActions(){ /* warning supression re synthetic */}
		
		public void keyTyped(KeyEvent event)
		{
			//System.out.println("keyTyped");
		}


      public void keyPressed(KeyEvent event)
       {
				//System.out.println("event.getKeyCode(): " + event.getKeyCode());
				//System.out.println("KEY: " + event.getKeyText(event.getKeyCode()));
				switch (event.getKeyCode())
				{
				case KeyEvent.VK_ENTER:
					if ((event.getModifiersEx() & InputEvent.CTRL_DOWN_MASK) == InputEvent.CTRL_DOWN_MASK)
					{
						revert();
					}
					if ((event.getModifiersEx() & InputEvent.ALT_DOWN_MASK) == InputEvent.ALT_DOWN_MASK)
					{
						revert();
					}
					event.consume();
					return;
				}
       }
      
		public void keyReleased(KeyEvent event)
		{
			//System.out.println("keyReleased");
			if (!event.isConsumed())
			{
				if (event.isAltDown())
				{
					processNumPad(event);
				}
				else
				{
					processFuncitionKeys(event);
				}
			}
		}

		private void processFuncitionKeys(KeyEvent event)
		{
				switch (event.getKeyCode())
			{
			case KeyEvent.VK_F2:
				SingleNoResizeScrollControl.this.myViewerFrame.gotoFullScreen(false);
				event.consume();
				return;
			case KeyEvent.VK_F4:
				SingleNoResizeScrollControl.this.myOptions.setFullSizeTileTo(NoResizeOptions.FULL_SIZE_CENTER_CENTER);
				showImage();
				event.consume();
				return;
			}
		}
	}
}
