/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package org.gerhardb.jibs.viewer.shows.full;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Cursor;
import java.awt.EventQueue;
import java.awt.Dimension;
import java.awt.DisplayMode;
import java.awt.GraphicsDevice;
import java.awt.GraphicsEnvironment;
import java.awt.Rectangle;
import java.awt.event.InputEvent;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.image.BufferedImage;
import javax.swing.*;
import javax.swing.JPanel;
import javax.swing.JToolBar;
import javax.swing.SwingConstants;

import org.gerhardb.jibs.viewer.IFrame;
import org.gerhardb.jibs.viewer.IRevert;
import org.gerhardb.jibs.viewer.PicInfoDialog;
import org.gerhardb.jibs.viewer.ViewerPreferences;
import org.gerhardb.lib.image.ImageChangeUtil;
import org.gerhardb.lib.scroller.ScrollerChangeEvent;
import org.gerhardb.lib.scroller.ScrollerListener;
import org.gerhardb.lib.scroller.ScrollerSlider;
import org.gerhardb.lib.util.Misc;

/**
 * Full Screen Display!
 * ALWAYS SHRINKS!  Tried to do otherwise, but failed.
 * Don't think you EVER need full screen with scroll bars.
 */
public class FullScreen implements ScrollerListener, IRevert
{
   // --------------- Things passed into me ---------------------------
   IFrame myFrame;

   // --------------- Things I create ---------------------------
   JFrame mySurface;
   JImage myImagePanel = new JImage();
   private JPanel myTopPanel = new JPanel(new BorderLayout());
   private JPanel myBottomPanel = new JPanel(new BorderLayout());
   private GraphicsEnvironment myGE =
      GraphicsEnvironment.getLocalGraphicsEnvironment();
   GraphicsDevice myGD = this.myGE.getDefaultScreenDevice();

   private boolean iBlowUp = ViewerPreferences.blowUp();
   //private boolean iShrink = ViewerPreferences.shrink();
   private int maxBlow = ViewerPreferences.getMaxBlow();
   
   //==========================================================================
   //                         Constructor
   //==========================================================================
   public FullScreen(IFrame mf, boolean startNow)
   {
   	this.myFrame = mf;
   	this.myFrame.getScroller().setEndMessages(true);
   	FullScreenKeyAdapter fska = new FullScreenKeyAdapter();
   	// Make sure we eat all events to stop bad stuff on full screen.
		this.myFrame.getScroller().setShowViewKeyListener(fska);

      // Set slider for initial show to user
      this.myBottomPanel.setVisible(false);

      this.myImagePanel.setFocusable(true);

      this.mySurface = new JFrame(this.myGD.getDefaultConfiguration());
      this.mySurface.setUndecorated(true);
      this.mySurface.setFocusableWindowState(true);
      this.mySurface.setFocusable(true);
      this.mySurface.setCursor(Misc.getTransparentCursor());

      JToolBar bottomRight = new JToolBar();
      bottomRight.setFloatable(false);
      bottomRight.add(this.myFrame.getActions().makeSaveButton());
      bottomRight.add(this.myFrame.getActions().getAction("file", "saveas")); //$NON-NLS-1$ //$NON-NLS-2$
      bottomRight.add(this.myFrame.getActions().getAction("edit", "left")); //$NON-NLS-1$ //$NON-NLS-2$
      bottomRight.add(this.myFrame.getActions().getAction("edit", "right")); //$NON-NLS-1$ //$NON-NLS-2$
      bottomRight.add(this.myFrame.getActions().getToolBarButton("file", "trash")); //$NON-NLS-1$ //$NON-NLS-2$
      bottomRight.add(this.myFrame.getActions().getToolBarButton("file", "park")); //$NON-NLS-1$ //$NON-NLS-2$
      bottomRight.add(this.myFrame.getActions().getToolBarButton("file", "repeat")); //$NON-NLS-1$ //$NON-NLS-2$

      ScrollerSlider slider = new ScrollerSlider(SwingConstants.HORIZONTAL, this.myFrame.getScroller());
      this.myBottomPanel.add(slider, BorderLayout.CENTER);
      this.myBottomPanel.add(bottomRight, BorderLayout.EAST);

      // Set the surface
      //myTopPanel.add(picScrollPane, BorderLayout.CENTER);
      this.myTopPanel.add(this.myImagePanel, BorderLayout.CENTER);
      this.myTopPanel.add(this.myBottomPanel, BorderLayout.SOUTH);
      this.mySurface.setContentPane(this.myTopPanel);

      this.mySurface.addKeyListener(this.myFrame.getScroller().getScrollerKeyListener());
      this.mySurface.addKeyListener(fska);

      this.mySurface.addMouseListener(new MouseAdapter()
      {
         @Override
			public void mousePressed(MouseEvent event)
         {
            if (event.isPopupTrigger()
               || event.getButton() == MouseEvent.BUTTON2
               || event.getButton() == MouseEvent.BUTTON3)
            {
               if (FullScreen.this.myFrame.getScroller().isBeyond())
               {
                  return;
               }
               // Java no longer likes my tricky work around...
               //popupDialog();  // pooh
            }
            else
            {
               FullScreen.this.myFrame.getScroller().down();
            }
         }
      });

      try
      {
    	  if ( ViewerPreferences.getGDFullScreen() )
    	  {
    		  this.myGD.setFullScreenWindow(this.mySurface);
    	  }
    	  else
    	  {
    	      GraphicsEnvironment ge =
    	          GraphicsEnvironment.getLocalGraphicsEnvironment();
    	       GraphicsDevice gd = ge.getDefaultScreenDevice();
    	       DisplayMode dm = gd.getDisplayMode();
    		  Dimension dim = new Dimension(dm.getWidth(), dm.getHeight());
    		  this.myImagePanel.setPreferredSize(dim);
    		  this.mySurface.pack();
    		  this.mySurface.setVisible(true);
    	  }
         //

         // Actually show the current picture
         scrollerChanged(null);

         this.myFrame.getScroller().setAutoFocus(this.mySurface);
         this.myFrame.getScroller().addScrollerListener(this);
         this.myImagePanel.requestFocus(true);
         this.mySurface.requestFocus();
         if (startNow)
         {
            startShow();
         }
      }
      catch (Exception ex)
      {
         ex.printStackTrace();
      }
   }

   public JFrame getFrame()
   {
   	return this.mySurface;
   }
   
   //==========================================================================
   //                         ScrollerListener Implemenation
   //==========================================================================
   public void scrollerChanged(ScrollerChangeEvent ce)
   {
      if (!this.myFrame.getScroller().getValueIsAdjusting())
      {
   		BufferedImage image = null;
   		try
   		{
   			image = this.myFrame.getScroller().getCurrentImage();
   		}
   		catch(Exception ex)
   		{
   			ex.printStackTrace();
   		}
   		
          if (image == null)
         {
         	// If there is no image to show, try to show the past the end
         	// picture.  In any event, return since there is no picture.
        	 image = this.myFrame.getScroller().getBeyondBounds();
            if (image != null)
            {
               this.myImagePanel.setImage(image);
             }
            return;
         }

         if (this.iBlowUp)// || iShrink)
         {
            // Let's see if we should do an on the fly resize.
            Rectangle clip = this.myImagePanel.getVisibleRect();
            double clipWidth = clip.getWidth();
            double clipHeight = clip.getHeight();

            if (clipWidth > 0 && clipHeight > 0)
            {
               int imgWidth = image.getWidth();
               int imgHeight = image.getHeight();

               //System.out.println("Full Screen Image: " + myFrame.getScroller().getCurrentFile());
               //System.out.println("Visible Region: w: " + clipWidth + " h: " + clipHeight );
               //System.out.println("Image Real: w: " + imgWidth + " h: " + imgHeight );
               float thetaDown =
                  ImageChangeUtil.calcThetaDown(
                  imgWidth, imgHeight, clipWidth, clipHeight);
               if (thetaDown > 0 && thetaDown < 1.0)
               {
                  //System.out.println("SHRINKING");
                  //System.out.println("thetaDown: " + thetaDown);
                  image = ImageChangeUtil.scale(image, thetaDown);
               }
               else
               {
                  if (this.iBlowUp)
                  {
                     //System.out.println("GROWING");
                     float thetaUp =
                        ImageChangeUtil.calcThetaUp(
                        imgWidth, imgHeight, clipWidth, clipHeight);
                     //System.out.println("thetaUp: " + thetaUp);

                     if (thetaUp > this.maxBlow)
                     {
                        image = ImageChangeUtil.scale(image, this.maxBlow);
                        //System.out.println("using max blow: " + maxBlow);
                     }
                     else if (thetaUp > 0)
                     {
                        image = ImageChangeUtil.scale(image, thetaUp);
                     }
                  }
               }
            }
         }
         //System.out.println("Image Adjusted: w: " + image.getWidth() + " h: " + image.getHeight() );
         this.myImagePanel.setImage(image);
      }
   }

   //==========================================================================
	//                        IRevert
	//==========================================================================
	public void revert()
	{
		this.myFrame.getScroller().stopSlideShow();
		this.myFrame.getScroller().removeScrollerListener(FullScreen.this);
		this.myFrame.getScroller().setShowViewKeyListener(null);
		
		// Clear screen, set to gray, and start wait for slow switch in Linux.
		this.myImagePanel.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
		// Need to set color before changing image.
		this.myImagePanel.setBackground(Color.darkGray);		
		this.myImagePanel.setImage(null);
		
		// If you run this without putting on thread,
		// the myGD.setFullScreenWindow executes before clear starts.
		Runnable runSwitch = new Runnable()
		{
			public void run()
			{
				//try{Thread.sleep(500);}catch(Exception ex){}
				FullScreen.this.myGD.setFullScreenWindow(null);
				FullScreen.this.mySurface.setVisible(false);
				FullScreen.this.mySurface.dispose();
				FullScreen.this.myFrame.setWaitCursor(true);
				FullScreen.this.myFrame.gotoRegularScreen();
				FullScreen.this.myFrame.setWaitCursor(false);
			}		
		};		
		EventQueue.invokeLater(runSwitch);
	}

   public boolean isFullScreen()
   {
      return true;
   }

   //==========================================================================
   //                        Methods
   //==========================================================================

   public void startShow()
   {
      this.myFrame.getScroller().startSlideShow(
         ViewerPreferences.continuousShow());
   }

   void toggleBottom()
   {
      if (this.myBottomPanel.isShowing())
      {
         this.myBottomPanel.setVisible(false);
         this.mySurface.setCursor(Misc.getTransparentCursor());
         this.myTopPanel.revalidate();
      }
      else
      {
         //Robot robot = new Robot();
         //robot.mouseMove(x, y);
         this.myBottomPanel.setVisible(true);
         this.mySurface.setCursor(Cursor.getDefaultCursor());
         this.myTopPanel.revalidate();
      }
   }
   
   void popupDialog()
   {
		int width = -1;
		int height = -1;
		try
		{
			BufferedImage image = FullScreen.this.myFrame.getScroller().getCurrentImage();
			width = image.getWidth();
			height = image.getHeight();
		}
		catch(Exception ex)
		{
			ex.printStackTrace();
		}
   	
   	/*
   	 * http://weblogs.java.net/blog/mickleness/archive/2011/04/06/implementing-fake-modal-dialogs-full-screen-mode
   	 * suggested this.  But not worth the effort.
   	 */
   	/*
      this.mySurface.setCursor(Cursor.DEFAULT_CURSOR);
   	JInternalFrame iFrame = new JInternalFrame("Test", true, true);
   	iFrame.add(new JLabel("hi"));
   	iFrame.pack();
   	iFrame.setVisible(true);
   	this.myTopPanel.add(iFrame);
   	*/
		
   	System.out.println("Popping up dialog...");
		new PicInfoDialog(
			this.myFrame.getScroller().getCurrentFile(),
		   this.mySurface,
		   this.myImagePanel,
		   this,
		   this.myFrame.getActions().getToolBarButton("file","trash"), //$NON-NLS-1$ //$NON-NLS-2$
		   this.myFrame.getActions().getToolBarButton("file","park"), //$NON-NLS-1$ //$NON-NLS-2$
		   true,
		   width,
		   height,
		   true, 
		   this.myFrame.getScroller());
   }
   
   class FullScreenKeyAdapter extends KeyAdapter
   {
      @Override
		public void keyPressed(KeyEvent event)
      {
   		if (event.isConsumed())
   		{
   			return;
   		}
			//System.out.println("FULL SCREEN keyPressed: " + event.getKeyCode() + "   KEY: " + KeyEvent.getKeyText(event.getKeyCode()));
			switch (event.getKeyCode())
			{
			case KeyEvent.VK_ENTER:
				// Alt did work here, but not on full screen...
				if ((event.getModifiersEx() & InputEvent.CTRL_DOWN_MASK) == InputEvent.CTRL_DOWN_MASK)
				{
					revert();
				}
				if ((event.getModifiersEx() & InputEvent.ALT_DOWN_MASK) == InputEvent.ALT_DOWN_MASK)
				{
					revert();
				}
				event.consume();
				return;
			default:
				// Prevent behaviors incompatible with full screen.
				event.consume();
			}
      }
      
   	@Override
		public void keyReleased(KeyEvent event)
		{
   		if (event.isConsumed())
   		{
   			return;
   		}
			//System.out.println("FULL SCREEN keyReleased: " + event.getKeyCode() + "   KEY: " + KeyEvent.getKeyText(event.getKeyCode()));
			switch (event.getKeyCode())
			{
			case KeyEvent.VK_F1:
				revert();
				FullScreen.this.myFrame.showHelp();
				event.consume();
				return;
			case KeyEvent.VK_ESCAPE:
			case KeyEvent.VK_F2:
				revert();
				event.consume();
				return;
			case KeyEvent.VK_F3:
				//toggleBottom();
				event.consume();
				return;
			case KeyEvent.VK_PAGE_DOWN:
				return; // Let main scroller handle
			case KeyEvent.VK_PAGE_UP:
				return; // Let main scroller handle
			case KeyEvent.VK_HOME:
				return; // Let main scroller handle
			case KeyEvent.VK_END:
				return; // Let main scroller handle
			case KeyEvent.VK_UP:
				return; // Let main scroller handle
			case KeyEvent.VK_LEFT:
				return; // Let main scroller handle
			case KeyEvent.VK_DOWN:
				return; // Let main scroller handle
			case KeyEvent.VK_RIGHT:
				return; // Let main scroller handle
			case KeyEvent.VK_SPACE:
				return; // Let main scroller handle
			case KeyEvent.VK_DELETE: 
				return; // Let main scroller handle
			case KeyEvent.VK_INSERT: 
				return; // Let main scroller handle
			case 109: // NumPad - File up
				return; // Let main scroller handle
			case 107: // NumPad + File down
				return; // Let main scroller handle
			case KeyEvent.VK_NUMPAD3: // File down
				return; // Let main scroller handle
			case KeyEvent.VK_NUMPAD9: // File up
				return; // Let main scroller handle
			case KeyEvent.VK_NUMPAD7: // home
				return; // Let main scroller handle
			case KeyEvent.VK_NUMPAD1: // end
				return; // Let main scroller handle
			case KeyEvent.VK_NUMPAD8: // up
				return; // Let main scroller handle
			case KeyEvent.VK_NUMPAD4: // left
				return; // Let main scroller handle
			case KeyEvent.VK_NUMPAD2: // down
				return; // Let main scroller handle
			case KeyEvent.VK_NUMPAD6: // right
				return; // Let main scroller handle
			case KeyEvent.VK_KP_UP: // up
				return; // Let main scroller handle
			case KeyEvent.VK_KP_LEFT: // left
				return; // Let main scroller handle
			case KeyEvent.VK_KP_DOWN: // down
				return; // Let main scroller handle
			case KeyEvent.VK_KP_RIGHT: // right
				return; // Let main scroller handle
			case KeyEvent.VK_NUMPAD0: // insert
				return; // Let main scroller handle
			case 110: // delete on the numeric keypad
				return; // Let main scroller handle
			default:
				// Prevent behaviors incompatible with full screen.
				event.consume();
			}
		}
   }
   
}
