/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
package org.gerhardb.jibs.viewer.shows.thumbs;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.dnd.DnDConstants;
import java.awt.dnd.DragGestureEvent;
import java.awt.dnd.DragGestureListener;
import java.awt.dnd.DragSource;
import java.awt.dnd.DragSourceListener;
import java.awt.dnd.DropTarget;
import java.awt.dnd.DropTargetDragEvent;
import java.awt.dnd.DropTargetDropEvent;
import java.awt.dnd.DropTargetEvent;
import java.awt.dnd.DropTargetListener;
import java.awt.event.InputEvent;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.image.BufferedImage;
import java.io.File;

import javax.swing.JComponent;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import org.gerhardb.jibs.viewer.PicInfoDialog;
import org.gerhardb.jibs.viewer.sorter.TileConstants;
import org.gerhardb.lib.dirtree.filelist.LinklessDragListener;
import org.gerhardb.lib.image.ImageChangeUtil;
import org.gerhardb.jibs.viewer.sorter.TilePanel;

class Picture extends JComponent implements MouseListener,
		ListSelectionListener, DropTargetListener
{
	private static int PADDING = 5;
	private static int PADDINGx2 = PADDING * 2;
	private int myCellIndex;
	private Thumbnails myParent;
	private BufferedImage myThumbnail;
	private String myName;
	private boolean iAmCurrentPicture = false;
	File myFile;
	int myScrollerIndex;
	private static int myPriorClickedIndex = -1;
	ListSelectionModel myListPicks;

	// Color.pink is Color(255, 175, 175)
	private static Color ROTATION = new Color(175, 255, 175); // green
	//private static Color ROTATION = new Color(175, 175, 255); // purlple

	boolean iShowRealImage = true;
	boolean iShowNothing = false;
	
	/**
	 * 
	 * @param frame
	 * @param index
	 *           count from 1
	 */
	public Picture(Thumbnails panel, int index)
	{
		this.myParent = panel;
		this.myCellIndex = index;
		addMouseListener(this);
		super.setOpaque(false);
		this.myParent.myViewerFrame.getFileList().addListSelectionListener(this);
		this.myListPicks = this.myParent.myViewerFrame.getFileList().getSelectionModel();

		Transferable trans = this.myParent.myFileList.getTransferable();
		DragSourceListener dsl = this.myParent.myFileList.getTransferable()
				.getDragSourceListener();
		DragGestureListener dgl = new PictureLinklessDragListener(dsl, trans,
				this.myParent.myDragSource);
		this.myParent.myDragSource.createDefaultDragGestureRecognizer(this,
				DnDConstants.ACTION_COPY_OR_MOVE, dgl);
		new DropTarget(this, this);
	}

	// ========================================================================
	// Non-Private Methods
	// ========================================================================
	void setImage(int scrollIndex)
	{
		this.myScrollerIndex = scrollIndex;
		this.myFile = this.myParent.myViewerFrame.getScroller().getFile(this.myScrollerIndex);
		if (this.myFile != null)
		{
			this.iShowNothing = false;
			this.myName = this.myFile.getName();
			this.myThumbnail = this.myParent.myCache.getPreviouslyCachedImage(this.myScrollerIndex);
			if (this.myThumbnail != null)
			{
				this.iShowRealImage = true;
			}
			else
			{
				this.iShowRealImage = false;
				this.myThumbnail = this.myParent.myJIBS;
			}
		}
		else
		{
			this.iShowNothing = true;			
			this.myName = null;
			this.myThumbnail = null;
		}
		repaint();
	}
	
	/**
	 * Designed to be called to get image where we know we have the image file
	 * but the image was not in the cache.  Designed to get the image from 
	 * cache in a non-UI thread.
	 */
	void grabImage()
	{
		if (this.iShowRealImage || this.iShowNothing)
		{
			//System.out.println("Grag image didn't bother");
			return;
		}
		this.myThumbnail = this.myParent.myCache.grabImage(this.myFile);
		if (this.myThumbnail != null)
		{
			this.iShowRealImage = true;
		}
		super.repaint();
	}
	
	void clearImage()
	{
		this.myName = null;
		this.myThumbnail = null;
		this.myFile = null;
		this.myScrollerIndex = 0;
		repaint();
	}	

	void setCurrentPicture(boolean current)
	{
		this.iAmCurrentPicture = current;
		repaint();
	}

	int getScrollerIndex()
	{
		return this.myScrollerIndex;
	}
	
	public boolean isRefreshNeeded()
	{
		// Not supposed to be showing anything.
		if (this.myScrollerIndex < 0) //-1 )
		{
			return false;
		}
		// Supposed to be showing, but not showing.
		if (this.myFile == null )
		{
			return true;			
		}
		// Has anything changed?
		if (this.myFile.equals(this.myParent.myViewerFrame.getScroller().getFile(this.myScrollerIndex)))
		{
			return false;
		}
		// If so, refresh is needed.
		return true;
	}
	
	// ==========================================================================
	// Overridden Methods
	// ==========================================================================

	@Override
	protected void paintComponent(Graphics graphics)
	{
		Graphics g = graphics.create();
		Graphics2D g2 = (Graphics2D) g;

		if (this.myListPicks.isSelectedIndex(this.myScrollerIndex))
		{
			setBackground(this.myParent.mySelectedColor);
		}
		else if (this.iAmCurrentPicture)
		{
			setBackground(Color.pink);
		}
		else
		{
			setBackground(Color.gray);
		}
		
		// Draw in our entire space, even if isOpaque is false.
		g.setColor(this.getBackground());
		g.fillRect(0, 0, getWidth(), getHeight());

		// Mark top and bottom if picture has been rotated.
		if (this.myScrollerIndex >= 0 && this.myParent.myRotations.getRotation(this.myScrollerIndex) != 0)
		{
			g.setColor(ROTATION);
			int bandHeight = getHeight()/5;
			int bandWidth = getWidth()/5;
			g.fillRect(0, 0, bandWidth, bandHeight);
			g.fillRect(getWidth() - bandWidth, getHeight() - bandHeight, getWidth(), getHeight());
		}

		// Double rubber tests.
		if (this.myThumbnail != null && !this.iShowNothing)
		{
			//System.out.println("Showing: " + this.myName);
			//maybeShowImage(myThumbnail,graphics,true);
			g2.setPaint(Color.BLACK);
			//String msg = Integer.toString(myCellIndex) + ": " + myName;
			String msg = Integer.toString(this.myParent.getScrollBarIndex(this.myCellIndex))
					+ ": " + this.myName; //$NON-NLS-1$

			Dimension textSize = setFont(g, msg, super.getWidth());

			// Adjust text height to leave room for text.
			int imgHeight = super.getHeight() - (int) textSize.getHeight()
					- PADDINGx2;
			int imgWidth = super.getWidth() - PADDINGx2;

			int topOfText = 0;
			// Rotate if real
			if (imgWidth > 0 && imgHeight > 0)
			{
				BufferedImage rotatedThumbnail = this.myParent.myRotations.getRotatedImage(
						this.myScrollerIndex, this.myThumbnail);
				
				if (rotatedThumbnail != null)
				{					
					BufferedImage sizedRotated = ImageChangeUtil.fitAspectDown(
							rotatedThumbnail, imgWidth, imgHeight);
					int centerX = ((super.getWidth() - sizedRotated.getWidth()) / 2);
					int centerY = ((super.getHeight() - sizedRotated.getHeight()) / 2);				
					g.drawImage(sizedRotated, centerX, centerY, this);				
					topOfText = centerY + sizedRotated.getHeight();
				}
			}

			if (textSize.getWidth() > 0)
			{
				// Widthwize, center the text.
				int textX = (int) ((super.getWidth() - textSize.getWidth()) / 2);

				// Draw the text just under the image now that we know the
				// final size of the image.
				// Text is draw up from the Ascent and down from the Descent.
				FontMetrics fm = g.getFontMetrics();
				int textY = (topOfText + fm.getMaxAscent());
				g.drawString(msg, textX, textY);
			}
		}
		else
		{
			//System.out.println("NOT SHOWING: " + this.myScrollerIndex);
		}
		
		TilePanel.clsTilePanel.clip(this.myFile, g2, super.getWidth(), super.getHeight());

		g.dispose();
	}

	/**
	 * http://www.particle.kth.se/~lindsey/JavaCourse/Book/Part1/Tech/Chapter06/plotDemo.html#PlotPanel
	 * Return height
	 */
	private static Dimension setFont(Graphics g, String msg, int box_width)
	{
		int type_size = 12;
		int type_size_min = 4;
		int msg_width;
		FontMetrics fm;

		do
		{
			Font f = new Font("Default", Font.PLAIN, type_size); //$NON-NLS-1$

			// Create the font and pass it to the Graphics context
			// g.setFont(new Font("Monospaced", Font.PLAIN, type_size));
			g.setFont(f);

			// Get measures needed to center the message
			fm = g.getFontMetrics();

			// How many pixels wide is the string
			msg_width = fm.stringWidth(msg);

			// See if the text will fit
			if (msg_width < box_width)
			{
				// { x = x_box + (box_width / 2) - (msg_width / 2);}
				break;
			}

			// Try smaller type
			type_size -= 2;
		}
		while (type_size >= type_size_min);

		// Don't display the numbers if they did not fit
		if (type_size < type_size_min) { return new Dimension(0, 0); }

		return new Dimension(msg_width, fm.getHeight());
	}

	private void flipSelection()
	{
		if (this.myListPicks.isSelectedIndex(this.myScrollerIndex))
		{
			this.myListPicks.removeSelectionInterval(this.myScrollerIndex, this.myScrollerIndex);
			myPriorClickedIndex = -1;
		}
		else
		{
			this.myListPicks.addSelectionInterval(this.myScrollerIndex, this.myScrollerIndex);
			myPriorClickedIndex = this.myScrollerIndex;
		}
	}

	// ==========================================================================
	// ListSelectionListener
	// ==========================================================================
	public void valueChanged(ListSelectionEvent e)
	{
		repaint();
	}

	// ==========================================================================
	// MouseListener
	// ==========================================================================
	public void mouseClicked(MouseEvent event)
	{
		if (event.getClickCount() == 2)
		{
			// Undo whatever single click did.
			flipSelection();
			this.myParent.myViewerFrame.getScroller().setValue(this.myScrollerIndex);
			this.myParent.myViewerFrame.gotoFullScreen(false);
		}
		else if (MouseEvent.BUTTON2 == event.getButton()
				|| MouseEvent.BUTTON3 == event.getButton())
		// two works in windows
		// if ( event.isPopupTrigger() )
		{
			if (this.myFile != null)
			{
				new PicInfoDialog(
						this.myFile, 
						this.myParent.myViewerFrame.getFrame(),
						this.myParent, 
						this.myParent,
						this.myParent.myViewerFrame.getActions().getToolBarButton("file", "trash"), 
						this.myParent.myViewerFrame.getActions().getToolBarButton("file", "park"), 
						false, -1, -1, false, 
						this.myParent.myViewerFrame.getScroller());
			}
		}
		else
		{
			if (this.myFile != null)
			{
				if ( (event.getModifiersEx() & InputEvent.SHIFT_DOWN_MASK) != 0 )
	         {
	            //Hightlight from appropriate other index.
	         	if (this.myListPicks.isSelectionEmpty())
               {
               	// At least turn this one on if no other are selected!!
    					flipSelection();	    	               	 
               }
	         	else
	         	{
	         		// We have a selection so these will both be > -1.
						int max = this.myListPicks.getMaxSelectionIndex();
						int min = this.myListPicks.getMinSelectionIndex();
						if (this.myScrollerIndex > max)
						{
							this.myListPicks.addSelectionInterval(max, this.myScrollerIndex);
						}
						else if (this.myScrollerIndex < min)
						{
							this.myListPicks.addSelectionInterval(min, this.myScrollerIndex);
						}
						else if (myPriorClickedIndex > -1)
						{
							// In between min and max.
							// Just select everything from last selection.
							//System.out.println("One Picked " + myPriorClickedIndex + " to " + myScrollerIndex);
							this.myListPicks.addSelectionInterval(myPriorClickedIndex,
									this.myScrollerIndex);
						}
						else
						{
							// At least turn this one on if no other are selected!!
							flipSelection();
						}
	         	}
	         }
	         else
	         {
					flipSelection();	         	
	         }
			}
			this.myParent.myViewerFrame.getScroller().requestFocus();
		}
	}

	public void mouseEntered(MouseEvent e)
	{
		// We don't care
	}

	public void mouseExited(MouseEvent e)
	{
		// We don't care
	}

	public void mousePressed(MouseEvent e)
	{
		// We don't care
	}

	public void mouseReleased(MouseEvent e)
	{
		// We don't care
	}

	// ==========================================================================
	// DragGestureListener Implementation
	// ==========================================================================
	class PictureLinklessDragListener extends LinklessDragListener
	{
		PictureLinklessDragListener(DragSourceListener dsl, Transferable t,
				DragSource ds)
		{
			super(dsl, t, ds);
		}

		@Override
		public void dragGestureRecognized(DragGestureEvent dge)
		{
			if (Picture.this.myFile == null) { return; }
			Picture.this.myListPicks.addSelectionInterval(Picture.this.myScrollerIndex, Picture.this.myScrollerIndex);
			super.dragGestureRecognized(dge);
		}
	}
	/*
	 * ========================================================================
	 * DropTargetListener Implementation
	 * Only used to turn off pictures dragged from sorter.
	 * ========================================================================
	 */

	private void evaluateDrag(DropTargetDragEvent event)
	{
		DataFlavor[] flavors  = event.getCurrentDataFlavors();
		if (flavors == TileConstants.CLS_DATA_FLAVORS)
		{
			event.acceptDrag(DnDConstants.ACTION_MOVE);
		}
		else
		{
			event.rejectDrag();
		}
	}

	public void dragEnter(DropTargetDragEvent event)
	{
		evaluateDrag(event);
	}

	public void dropActionChanged(DropTargetDragEvent event)
	{
		evaluateDrag(event);
	}

	public void dragExit(DropTargetEvent dte)
	{
		// System.out.println("DragDropList.dragExit");
	}

	public void dragOver(DropTargetDragEvent dtde)
	{
		// System.out.println( "DragDropList.dragOver" );
	}

	public void drop(DropTargetDropEvent event)
	{
		// Fine example of how to do the local drop if you need tighter
		// communications.
		if (event.isLocalTransfer())
		{
			DataFlavor[] flavors  = event.getCurrentDataFlavors();
			if (flavors == TileConstants.CLS_DATA_FLAVORS)
			{
				event.dropComplete(true);
			}
			else
			{
				event.dropComplete(false);
			}
		}
	}

}
