/*
 * Copyright (c) 2007 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.gerhardb.jibs.viewer.shows.thumbs;

import java.awt.datatransfer.Clipboard;
import java.awt.image.BufferedImage;
import java.io.File;
import java.util.ArrayList;

import javax.swing.BoundedRangeModel;
import javax.swing.DefaultBoundedRangeModel;
import javax.swing.JFrame;
import javax.swing.event.ListDataEvent;
import javax.swing.event.ListDataListener;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import org.gerhardb.jibs.Jibs;
import org.gerhardb.jibs.viewer.IFrame;
import org.gerhardb.jibs.viewer.ISave;
import org.gerhardb.jibs.viewer.ViewerPreferences;
import org.gerhardb.jibs.viewer.contact.ContactSheetDisplay;
import org.gerhardb.jibs.viewer.frame.SaverActions;
import org.gerhardb.lib.image.IOImage;
import org.gerhardb.lib.image.ImageFactory;
import org.gerhardb.lib.image.ImageUtilDrew;
import org.gerhardb.lib.util.ICancel;
import org.gerhardb.lib.util.ModalProgress;
import org.gerhardb.lib.dirtree.filelist.FileList;

import com.saic.isd.printscreen.ImageSelection;

/**
 * This gets called from SaverActions: mySortScreen.myShow.getSaver().rotateRight();
 * This is a utility class which does not know selections, rotations or images.
 * All must be fed into this class.
 */
public class ThumbnailSaver implements ISave, ListSelectionListener, ListDataListener, ICancel
{
	boolean iCancel = false;  // NO_UCD
	SaverActions mySaverActions;
	IFrame myViewerFrame;
	JFrame myJFrame;
	FileList  myFileList;	
	Thumbnails myThumbnails;
	ThumbnailCache myCache;

	// =========================================================================
	// Constructor
	// =========================================================================
	ThumbnailSaver(IFrame vf, Thumbnails t, ThumbnailCache c)
	{
		this.myViewerFrame = vf;
		this.myThumbnails = t; 
		this.myCache = c;
		
		this.mySaverActions = this.myViewerFrame.getActions().getSaverActions();
		this.myJFrame = this.myViewerFrame.getFrame();
		this.myFileList = this.myViewerFrame.getFileList();
		
		this.myFileList.addListSelectionListener(this);
		this.myFileList.getModel().addListDataListener(this);
		resetToolbar();
	}
	
	
   public void cancel()
   {
   	this.iCancel = true;
   }

	// ==========================================================================
	// ISave Interface
	// ==========================================================================
	
	/**
	 * We will ignore shrinkGrow which should never be turned on.
	 * We are ignoring it because the user could not see the results on the 
	 * thumbnail screen which resizes all pictures to fit the grid.
	 */
	public void shrink()
	{
		// We don't care		
	}
	public void grow()
	{
		// We don't care		
	}
	
	public void rotateRight()
	{
		adjustImage(90);
	}
	
	public void rotateLeft()
	{
		adjustImage(-90);
	}
	
	public void resetToolbar()
	{
		//System.out.println(">>>>>>>>>>>>>>>>>>>>>>>>>> RESETTING TOOLBAR <<<<<<<<<<<<<<<<<<<<<<<<<<<");
		this.mySaverActions.enableSave(false);	
		this.mySaverActions.enableSaveAs(false);
		this.mySaverActions.enableRotation(false);
		this.mySaverActions.enableShrinkGrow(false);
	}
	
	public void save(boolean saveAs)
	{
		/*
		 * We will ignore saveAs because we don't know how to rename multiple files.
		 * We do know how to save which will be used if the user has rotated.
		 */
		if (saveAs){return;}
		
		BoundedRangeModel range = new DefaultBoundedRangeModel();

		ModalProgress mp = new ModalProgress(this.myJFrame, Jibs.getString("ThumbnailSaver.0"), range); //$NON-NLS-1$
		mp.setMessage(Jibs.getString("ThumbnailSaver.1")); //$NON-NLS-1$
		mp.setCancel(this);
		
		DoIt doIt = new DoIt(range, mp);
		mp.start(doIt);
	}
	
	public void copy()
	{
		/*
		 * IShow obviously needs a getCurrentImage() item for copy, preview, print.
		 * Wait a minute, maybe ISave needs copy, preview, and print functions,
		 * then they can vary by show!!!! 
		 * Yeah, thats a better idea - thumbnails might work for pink item, 
		 * OR show for screen and make the pink item go away altogether as a 
		 * stupid idea.  Then delete would not do ANYTHING until user selects
		 * something.
		 */
		IOImage image = this.myViewerFrame.getScroller().getCurrentIOImage();
      ImageSelection clip = new ImageSelection(image, this.myFileList.getSelectedFiles());
      Clipboard clipboard = this.myJFrame.getToolkit().getSystemClipboard();
      clipboard.setContents( clip, clip );		
	}
	
	/*
	public void preview()
	{
		new ContactSheetDisplay(this.myJFrame, this.myFileList.getSelectedFiles(), ContactSheetDisplay.SHOW_PREVIEW, this.myViewerFrame.getScroller());
	}
	
	public void print(boolean highResolution)
	{
		new ContactSheetDisplay(this.myJFrame, this.myFileList.getSelectedFiles(), ContactSheetDisplay.SHOW_PRINT, this.myViewerFrame.getScroller());
	}	
	*/
	public void pageableAction(int action)
	{
		File[] files = null;
		// Can't use this because one file is always selected.
		// The one with the pink box.
		//File[] files = this.myFileList.getSelectedFiles();
		//int[] picked = this.myFileList.getSelectedIndices();
		
		if (this.myFileList.getSelectedIndices().length > 0)
		{
			files = this.myFileList.getSelectedFiles();
		}
		else
		{
			files = ThumbnailSaver.this.myFileList.getAllFiles();			
		}
		new ContactSheetDisplay(this.myJFrame, files, true, ContactSheetDisplay.SHOW_DIALOG, this.myViewerFrame.getScroller());
	}
	
	
	// ==========================================================================
	// ListSelectionListener
	// ==========================================================================
	public void valueChanged(ListSelectionEvent e)
	{
		// Enable rotation tool button to be on if the list is not empty.
		this.mySaverActions.enableRotation(!this.myFileList.isSelectionEmpty());		
	}
	
	// ==========================================================================
	// ListDataListener
	// ==========================================================================

	public void intervalAdded(ListDataEvent e)
	{
		resetToolbar();
		this.myThumbnails.showPageFromScroller();		
	}
	
	public void intervalRemoved(ListDataEvent e)
	{
		resetToolbar();	
		this.myThumbnails.showPageFromScroller();			
	}
	
	public void contentsChanged(ListDataEvent e)	
	{
		resetToolbar();	
	}
	
	// ==========================================================================
	// Non-Public Methods
	// ==========================================================================
	
	private void adjustImage(int rotation)
	{
		System.out.println("================ Thumbnail Saver adjustImage: " + rotation + " ================");
		this.mySaverActions.enableSave(true);	
		
		int[] picked = this.myFileList.getSelectedIndices();
		for (int i=0; i<picked.length; i++)
		{
			int index = picked[i];
			this.myThumbnails.myRotations.addRotation(index, rotation);
		}
		System.out.println("Thumbnail Saver adjustImage Finished.");
	}
	
	/*
	private BufferedImage getCurrentImage()
	{
		int scrollerIndex = this.myViewerFrame.getScroller().getValueZeroBased();
		try
		{
			BufferedImage image =  this.myViewerFrame.getScroller().getCurrentImage();
			return this.myThumbnails.myRotations.getRotatedImage(scrollerIndex, image);
		}
		catch (Exception ex)
		{
			System.out.println("Scroller.getCurrentImage FAILED"); //$NON-NLS-1$
			System.out.println(ex.getMessage());
			ex.printStackTrace();
		}
		return null;
	}
	*/
	
	boolean hasExifData(File file)
	{
		ArrayList<?> exifTags = ImageUtilDrew.getExifTags(file);
		if (exifTags.size() > 0)
		{
			return true;
		}
		return false;
	}
	
	private class DoIt implements Runnable
	{
		BoundedRangeModel myRange;
		ModalProgress myMP;

		DoIt(BoundedRangeModel range, ModalProgress mp)
		{
			this.myRange = range;
			this.myMP = mp;
		}

		public void run()
		{
			boolean exifWarning = ViewerPreferences.showExifWarning();
			boolean askExifWarning = true;
			boolean keepExifData = false;
			boolean saveThisFile = true;

			// Find out how many we need to do.
			int rotationCount = 0;
			for (int i=0; i< ThumbnailSaver.this.myThumbnails.myRotations.myRotations.length; i++)
			{
				if (ThumbnailSaver.this.myThumbnails.myRotations.myRotations[i] != 0)
				{
					rotationCount++;
				}
			}
			
			this.myRange.setMaximum(rotationCount - 1);
			this.myRange.setMinimum(0);
			this.myRange.setExtent(1);

			int count = 0;
			for (int i=0; i<ThumbnailSaver.this.myThumbnails.myRotations.myRotations.length; i++)
			{
				if (ThumbnailSaver.this.myThumbnails.myRotations.myRotations[i] != 0)
				{
					this.myRange.setValue(count++);
					
					File file = ThumbnailSaver.this.myViewerFrame.getScroller().getFile(i);
					if (file != null)
					{
						if (exifWarning)
						{
							// Only warn once and only if there is EXIF data.
							if (askExifWarning && hasExifData(file))
							{
								keepExifData = ThumbnailSaver.this.mySaverActions.cancelSaveForExifDialog(file);
								askExifWarning = false;
							}
							saveThisFile = true;
							// If we are supposed to keep exif data and this file 
							// has exif data, don't save the file!
							if (keepExifData && hasExifData(file))
							{
								saveThisFile = false;
							}
						}
						
						if (saveThisFile)
						{
							try
							{
								IOImage io = ImageFactory.getImageFactory().makeImage(file);
								BufferedImage image = ThumbnailSaver.this.myThumbnails.myRotations.getRotatedImage(i, io.getImage());
								io.save(image);
							}
							catch (Exception ex)
							{
								System.out.println(ex.getMessage());
								ex.printStackTrace();
							}
						}
					}
				}
			}				
			this.myRange.setValue(rotationCount);
			this.myMP.dispose();
			
			// Redisplay everything.
			ThumbnailSaver.this.myThumbnails.myCache.resetAndReshow(); // includes showPageFromScroller();		
		}
	}
}
