/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
package org.gerhardb.jibs.viewer.sorter;

import java.awt.Color;
import java.awt.Component;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.dnd.DnDConstants;
import java.awt.dnd.DragGestureEvent;
import java.awt.dnd.DragGestureListener;
import java.awt.dnd.DragSource;
import java.awt.dnd.DragSourceDragEvent;
import java.awt.dnd.DragSourceDropEvent;
import java.awt.dnd.DragSourceEvent;
import java.awt.dnd.DragSourceListener;
import java.awt.dnd.DropTarget;
import java.awt.dnd.DropTargetDragEvent;
import java.awt.dnd.DropTargetDropEvent;
import java.awt.dnd.DropTargetEvent;
import java.awt.dnd.DropTargetListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.image.BufferedImage;
import java.io.File;
import java.util.List;

import javax.swing.JComponent;
import javax.swing.SwingUtilities;

import org.gerhardb.jibs.viewer.PicInfoDialogBase;
import org.gerhardb.jibs.viewer.shows.classic.SingleDrag;
import org.gerhardb.lib.dirtree.filelist.FileDrag;
import org.gerhardb.lib.image.ImageChangeUtil;
import org.gerhardb.lib.image.ImageFactory;

class Tile extends JComponent implements DragGestureListener,
		DragSourceListener, Transferable, DropTargetListener, MouseListener
{
	private static final int PADDING = 5;

	private static final int PADDINGx2 = PADDING * 2;

	private boolean iBeDraggedOver = false;

	File myFile;

	private BufferedImage myImage;

	private DragSource myDragSource = new DragSource();

	public Tile()
	{
		this(-1, null);
	}

	/**
	 * 
	 * @param frame
	 * @param index
	 *           count from 1
	 */
	public Tile(int at, File file)
	{
		this.myFile = file;
		super.setOpaque(false);
		super.setPreferredSize(TilePanel.clsTileSizeDim);
		super.addMouseListener(this);
		new DropTarget(this, this);
		TilePanel.clsTilePanel.add(this, at);
		TilePanel.clsTilePanel.revalidate();

		if (this.myFile != null)
		{
			BufferedImage ezImage = ImageFactory.getImageFactory().getImageEZ(this.myFile);
			if (ezImage != null)
			{
				// Can't use getWidth() here because it has not been shown yet.
				int imgHeight = TilePanel.clsTileSize - PADDINGx2;
				int imgWidth = TilePanel.clsTileSize - PADDINGx2;
				this.myImage = ImageChangeUtil.fitAspectDown(ezImage, imgWidth, imgHeight);
			}
		}

		this.myDragSource.createDefaultDragGestureRecognizer(this,
				DnDConstants.ACTION_COPY_OR_MOVE, this);

	}

	int getIndex()
	{
		Component[] tiles = TilePanel.clsTilePanel.getComponents();
		for (int i = 0; i < tiles.length; i++)
		{
			if (tiles[i] == this)
			{
				return i;
			}
		}
		return -1;
	}

	// ==========================================================================
	// Overridden Methods
	// ==========================================================================

	@Override
	protected void paintComponent(Graphics g)
	{
		Graphics2D g2 = (Graphics2D) g;

		// Draw in our entire space, even if isOpaque is false.
		if (this.iBeDraggedOver)
		{
			g.setColor(Color.pink);
			g.fillRect(0, 0, getWidth(), getHeight());
		}
		else if (this.myFile == null)
		{
			g.setColor(Color.white);
			g.fillRect(0, 0, getWidth(), getHeight());
		}
		else if (this.myImage == null)
		{
			g.setColor(Color.green);
			g.fillRect(0, 0, getWidth(), getHeight());
		}

		if (this.myImage != null)
		{
			int centerX = ((super.getWidth() - this.myImage.getWidth()) / 2);
			int centerY = ((super.getHeight() - this.myImage.getHeight()) / 2);
			g.drawImage(this.myImage, centerX, centerY, this);
		}

		// Show Number
		String tileIndex = Integer.toString(getIndex() + 1);
		Dimension textSize = setFont(g, tileIndex, super.getWidth());
		if (textSize.getWidth() > 0)
		{
			int textY = (int) ((super.getHeight() - textSize.getHeight()) / 2);

			// Text is draw up from the Ascent and down from the Descent.
			// Font f = new Font("Default", Font.PLAIN, 48); //$NON-NLS-1$
			// g.setFont(f);
			FontMetrics fm = g.getFontMetrics();
			textY = textY + fm.getAscent();

			g2.setPaint(Color.red);
			/*
			 // Not drawing just for the moment... May want to add back.
			g.drawString(tileIndex,
					((super.getWidth() - fm.stringWidth(tileIndex)) / 2), (((super
							.getHeight() - fm.getHeight()) / 2) + fm.getAscent()));
			*/
		}

		g.dispose();
	}

	/**
	 * http://www.particle.kth.se/~lindsey/JavaCourse/Book/Part1/Tech/Chapter06/plotDemo.html#PlotPanel
	 * Return height
	 */
	private static Dimension setFont(Graphics g, String msg, int box_width)
	{
		int type_size = 40;
		int type_size_min = 4;
		int msg_width;
		FontMetrics fm;

		do
		{
			Font f = new Font("Default", Font.PLAIN, type_size); //$NON-NLS-1$

			// Create the font and pass it to the Graphics context
			// g.setFont(new Font("Monospaced", Font.PLAIN, type_size));
			g.setFont(f);

			// Get measures needed to center the message
			fm = g.getFontMetrics();

			// How many pixels wide is the string
			msg_width = fm.stringWidth(msg);

			// See if the text will fit
			if (msg_width < box_width)
			{
				// { x = x_box + (box_width / 2) - (msg_width / 2);}
				break;
			}

			// Try smaller type
			type_size -= 2;
		}
		while (type_size >= type_size_min);

		// Don't display the numbers if they did not fit
		if (type_size < type_size_min)
		{
			return new Dimension(0, 0);
		}

		return new Dimension(msg_width, fm.getHeight());
	}

	/*
	 * ========================================================================
	 * DragGestureListener Implementation
	 * ========================================================================
	 */
	public void dragGestureRecognized(DragGestureEvent dge)
	{
		Cursor cursor = DragSource.DefaultLinkNoDrop;
		switch (dge.getDragAction())
		{
			case DnDConstants.ACTION_MOVE:
				// System.out.println("Action: MOVE" );
				cursor = DragSource.DefaultMoveDrop;
				break;
			case DnDConstants.ACTION_COPY:
				// System.out.println("Action: COPY");
				cursor = DragSource.DefaultCopyDrop;
				break;
			default:
				return;
		}
		this.myDragSource.startDrag(dge, cursor, this, this);
	}

	// ==========================================================================
	// Transferable Implemenation
	// ==========================================================================

	public Object getTransferData(DataFlavor flavor)
	{
		if (TileConstants.TILE_DATA_FLAVOR.equals(flavor))
		{
			return this;
		}
		return null;
	}

	public DataFlavor[] getTransferDataFlavors()
	{
		return TileConstants.CLS_DATA_FLAVORS;
	}

	public boolean isDataFlavorSupported(DataFlavor flavor)
	{
		// System.out.println("FileList isDataFlavorSupported");
		for (int i = 0; i < TileConstants.CLS_DATA_FLAVORS.length; i++)
		{
			if (TileConstants.CLS_DATA_FLAVORS[i].equals(flavor))
			{
				return true;
			}
		}
		return false;
	}

	/*
	 * ========================================================================
	 * DropTargetListener Implementation
	 * ========================================================================
	 */

	private void evaluateDrag(DropTargetDragEvent event)
	{
		switch (event.getDropAction())
		{
			case DnDConstants.ACTION_MOVE:
			case DnDConstants.ACTION_COPY_OR_MOVE:
			case DnDConstants.ACTION_COPY:
				event.acceptDrag(DnDConstants.ACTION_COPY_OR_MOVE);
				this.iBeDraggedOver = true;
				super.repaint();
				break;
			default:
				this.iBeDraggedOver = false;
				super.repaint();
				event.rejectDrag();
		}
	}

	public void dragEnter(DropTargetDragEvent event)
	{
		// System.out.println("DragDropList.dragEnter");
		evaluateDrag(event);
	}

	public void dropActionChanged(DropTargetDragEvent event)
	{
		// System.out.println( "DragDropList.dropActionChanged" );
		evaluateDrag(event);
	}

	public void dragExit(DropTargetEvent dte)
	{
		// System.out.println("DragDropList.dragExit");
		this.iBeDraggedOver = false;
		super.repaint();
	}

	public void dragOver(DropTargetDragEvent dtde)
	{
		// System.out.println( "DragDropList.dragOver" );
	}

	public void drop(DropTargetDropEvent event)
	{
		// System.out.println("DragDropList.drop");
		switch (event.getDropAction())
		{
			case DnDConstants.ACTION_MOVE:
				// Accept below
				break;
			case DnDConstants.ACTION_COPY:
				// Accept below
				break;
			default:
				// System.out.println("Unsupported User Action");
				event.rejectDrop();
				return;
		} 

		// Fine example of how to do the local drop if you need tighter
		// communications.
		if (event.isLocalTransfer())
		{
			if (event.isDataFlavorSupported(TileConstants.TILE_DATA_FLAVOR))
			{
				// This is from the Sorter display.
				// Class Tile.
				System.out.println("Processing LOCAL Tile Drop"); //$NON-NLS-1$
				event.acceptDrop(event.getDropAction());
				processLocalTileDrag(event);
			}
			else if (event.isDataFlavorSupported(FileDrag.FILE_DRAG_DATA_FLAVOR))
			{
				// This is from the Thumbnail display.
				// Class Picture.
				//System.out.println("Processing LOCAL File Drop"); //$NON-NLS-1$
				event.acceptDrop(event.getDropAction());
				processLocalFileDrag(event);
			}
			else if (event.isDataFlavorSupported(SingleDrag.SINGLE_DRAG_DATA_FLAVOR))
			{
				// This is from the Single picture display.
				// Class Single and its decendents.
				System.out.println("Tile: Processing LOCAL Single Drop"); //$NON-NLS-1$
				event.acceptDrop(event.getDropAction());
				processLocalSingleDrag(event);
			}
		}
		else if (event.isDataFlavorSupported(DataFlavor.javaFileListFlavor))
		{
			event.acceptDrop(event.getDropAction());
			processFileList(event);
		}

		if (this.myFile == null)
		{
			//System.out.println("Scrolling"); //$NON-NLS-1$
			SwingUtilities.invokeLater(new Runnable()
			{
				public void run()
				{
					TilePanel.clsHorizontalScrollBar.setValue(TilePanel.clsHorizontalScrollBar.getMaximum());
				}
			});
		}
		
		// Reject here to keep files from being removed where they are.
		event.rejectDrop();
	}

	void processLocalTileDrag(DropTargetDropEvent event)
	{
		Transferable transfer = event.getTransferable();

		Object moveObj = null;
		try
		{
			// For drops from windows to work, you must accept the drop first.
			moveObj = transfer.getTransferData(TileConstants.TILE_DATA_FLAVOR);
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
			event.dropComplete(false);
			return;
		}
		if (moveObj == null || !(moveObj instanceof Tile))
		{
			event.dropComplete(false);
			return;
		}
		Tile tileDrag = (Tile) moveObj;

		new Tile(getIndex(), tileDrag.myFile);

		// This is OK here since I do NOT delete the files on the drop.
		event.dropComplete(true);
		this.iBeDraggedOver = false;
		super.repaint();
	}

	void processLocalFileDrag(DropTargetDropEvent event)
	{
		Transferable transfer = event.getTransferable();

		Object moveObj = null;
		try
		{
			// For drops from windows to work, you must accept the drop first.
			moveObj = transfer.getTransferData(FileDrag.FILE_DRAG_DATA_FLAVOR);
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
			event.dropComplete(false);
			return;
		}
		if (moveObj == null || !(moveObj instanceof FileDrag))
		{
			event.dropComplete(false);
			return;
		}
		FileDrag fileDrag = (FileDrag) moveObj;

		List<File> fileList = fileDrag.getFileList();
		Object[] loop = fileList.toArray();
		for (int i = 0; i < loop.length; i++)
		{
			if (loop[i] instanceof File)
			{
				new Tile(getIndex(), (File) loop[i]);
			}
		}

		// fileDrag.clearList();
		// This is OK here since I do NOT delete the files on the drop.
		event.dropComplete(true);
		this.iBeDraggedOver = false;
		super.repaint();
	}

	void processLocalSingleDrag(DropTargetDropEvent event)
	{
		Transferable transfer = event.getTransferable();

		Object moveObj = null;
		try
		{
			// For drops from windows to work, you must accept the drop first.
			moveObj = transfer.getTransferData(SingleDrag.SINGLE_DRAG_DATA_FLAVOR);
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
			event.dropComplete(false);
			return;
		}
		if (moveObj == null || !(moveObj instanceof SingleDrag))
		{
			event.dropComplete(false);
			return;
		}
		SingleDrag singleDrag = (SingleDrag) moveObj;
		
		File file = singleDrag.getFileToTransfer();
		if (file != null)
		{
			new Tile(getIndex(), singleDrag.getFileToTransfer());
		}
		
		// This is OK here since I do NOT delete the files on the drop.
		event.dropComplete(true);
		this.iBeDraggedOver = false;
		super.repaint();
		TilePanel.clsSortScreen.getScroller().refreshCurrentImage();
	}

	void processFileList(DropTargetDropEvent event)
	{
		Transferable transfer = event.getTransferable();

		Object moveObj = null;
		try
		{
			// For drops from windows to work, you must accept the drop first.
			moveObj = transfer.getTransferData(DataFlavor.javaFileListFlavor);
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
			event.dropComplete(false);
			return;
		}
		if (moveObj == null || !(moveObj instanceof List<?>))
		{
			event.dropComplete(false);
			return;
		}
		List<?> fileList = (List<?>) moveObj;
		Object[] loop = fileList.toArray();
		for (int i = 0; i < loop.length; i++)
		{
			if (loop[i] instanceof File)
			{
				//System.out.println(loop[i]);
				new Tile(getIndex(), (File) loop[i]);
			}
		}

		// Make sure this does not get deleted in the event of a move.
		event.dropComplete(false);
		this.iBeDraggedOver = false;
		super.repaint();
	}

	// ==========================================================================
	// DragSourceListener
	// ==========================================================================
	/**
	 * This is where we find out the drag worked!!!!
	 */
	public void dragDropEnd(DragSourceDropEvent event)
	{
		// Don't remove the blank at the end.
		if (this.myFile == null){return;}
		
		//System.out.println("Tile.dragDropEnd");
		if (event.getDropSuccess())
		{
			if (event.getDropAction() == DnDConstants.ACTION_MOVE)
			{
				TilePanel.clsTilePanel.remove(this);
				TilePanel.clsTilePanel.revalidate();
			}
		}
		event.getDragSourceContext().setCursor(
				Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));

		//Reshow the main viewer image to get rid on triangle markers.
		TilePanel.clsSortScreen.getScroller().refreshCurrentImage();
	}

	public void dropActionChanged(DragSourceDragEvent event)
	{
		// System.out.println( "FileDrag.dropActionChanged" );
		switch (event.getUserAction())
		{
			case DnDConstants.ACTION_MOVE:
				// System.out.println("FileDrag Accepting Drop: MOVE");
				event.getDragSourceContext().setCursor(DragSource.DefaultMoveDrop);
				break;
			case DnDConstants.ACTION_COPY:
				// System.out.println("FileDrag Accepting Drop: COPY");
				event.getDragSourceContext().setCursor(DragSource.DefaultCopyDrop);
				break;
			default:
		}
	}

	public void dragExit(DragSourceEvent dse)
	{
		// System.out.println("ScrolledSlider.dragExit");
	}

	public void dragEnter(DragSourceDragEvent dsde)
	{
		// System.out.println("ScrolledSlider.dragEnter");
	}

	public void dragOver(DragSourceDragEvent dsde)
	{
		// System.out.println( "ScrolledSlider.dragOver" );
	}

	// ==========================================================================
	// MouseListener
	// ==========================================================================
		public void mouseClicked(MouseEvent event)
		{
			if (MouseEvent.BUTTON1 == event.getButton() && event.getClickCount() == 2)
			{
				//ViewerGlobals.getScroller().setValue(myScrollerIndex);
				//TilePanel.clsSortScreen.revert(false);
			}
			else if (MouseEvent.BUTTON2 == event.getButton()
					|| MouseEvent.BUTTON3 == event.getButton())
			// two works in windows
			// if ( event.isPopupTrigger() )
			{

				if (this.myFile != null)
				{
					PicInfoDialogBase info = new PicInfoDialogBase(TilePanel.clsSortScreen.getFrame(), TilePanel.clsSortScreen.getScroller());
				   info.display(this.myFile, null, -1, -1, false); 
				}
			}
		}
		public void mouseEntered(MouseEvent e)
		{
			// We don't care
		}

		public void mouseExited(MouseEvent e)
		{
			// We don't care
		}

		public void mousePressed(MouseEvent e)
		{
			// We don't care
		}

		public void mouseReleased(MouseEvent e)
		{
			// We don't care
		}

}
