/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
package org.gerhardb.lib.dirtree;

import java.awt.Dimension;
import java.awt.EventQueue;
import java.awt.Insets;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Window;
import java.awt.dnd.Autoscroll;
import java.io.File;
import java.io.FileFilter;
import java.io.FileNotFoundException;
import java.util.ArrayList;
import java.util.Enumeration;

import javax.swing.JScrollBar;
import javax.swing.JScrollPane;
import javax.swing.JTree;
import javax.swing.SwingUtilities;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeCellRenderer;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;

import org.gerhardb.lib.util.Icons;

/**
 Derived from code originally presented in
 Core Swing Advanced Programming
 by Kim Topley, Printice Hall PTR, chapter 8.
 */
public class DirectoryTree
   extends JTree implements Autoscroll
{
   public static final Insets myDefaultScrollInsets = new Insets(8, 8, 8, 8);
   protected Insets myScrollInsets = myDefaultScrollInsets;
   protected Window myTopWindow;
   private FileFilter myFileFilter;
   private boolean iCountTargetFiles = true;
   private boolean iCountHiddenDirectories = false;
	File[] myExcludedDirectories;
	
   // --------------------------------------------------------------------------
   //                           Constructor
   // --------------------------------------------------------------------------

   /**
    * @param showDirCount Whether to show the directory count.
    * @throws FileNotFoundException
    * @throws java.lang.SecurityException
    */
   public DirectoryTree(Window window, FileFilter filter, boolean countHiddenDirectories)
   {
      super();
      this.myTopWindow = window;
      this.myFileFilter = filter;
      this.iCountHiddenDirectories = countHiddenDirectories;
      
      // Use horizontal and vertical lines
      putClientProperty("JTree.lineStyle", "Angled"); //$NON-NLS-1$ //$NON-NLS-2$

       // How to change the folder icons if desired.
       DefaultTreeCellRenderer renderer = new DefaultTreeCellRenderer();
       renderer.setLeafIcon(Icons.getIcon(Icons.DIR_NO_SUBS));
       renderer.setOpenIcon(Icons.getIcon(Icons.DIR_SUBS));
       renderer.setClosedIcon(Icons.getIcon(Icons.DIR_SUBS));
       renderer.setDisabledIcon(Icons.getIcon(Icons.DIR_SUBS));
       renderer.setIcon(Icons.getIcon(Icons.DIR_SUBS));
       this.setCellRenderer(renderer);
   } 

   /**
    * List that won't be scanned.
    * Used for when file systems are mounted on a drive 
    * and you are just trying to get info about the drive.
    * @param exclude
    */
   public void setExcludedDirectories(File[] exclude)
   {
   	this.myExcludedDirectories = exclude;
   }
   
   boolean isNotAnExcludedDirectory(File file)
   {
   	if (this.myExcludedDirectories == null || file == null)
   	{
   		return true;
   	}
   	for (int i=0; i<this.myExcludedDirectories.length; i++)
   	{
   		if (file.equals(this.myExcludedDirectories[i]))
   		{
   			//System.out.println("DirectoryTree Excluding: " + myExcludedDirectories[i]);
   			return false;
   		}
   	}
   	return true;
   }
   
   public Window getTopWindow()
   { return this.myTopWindow; }
   
   public FileFilter getFileFilter()
   {
   	return this.myFileFilter;
   }

   public boolean countHiddenDirectories(){return this.iCountHiddenDirectories;}
   public boolean getCountTargetFiles(){return this.iCountTargetFiles;}
   public void setCountTargetFiles(boolean setting){this.iCountTargetFiles = setting;}
      
   // --------------------------------------------------------------------------
   //                         Autoscroll Implementation
   // --------------------------------------------------------------------------
   public void setScrollInsets(Insets insets)
   {
      this.myScrollInsets = insets;
   }

   public Insets getScrollInsets()
   {
      return this.myScrollInsets;
   }

   // Implementation of Autoscroll interface
   public Insets getAutoscrollInsets()
   {
      Rectangle r = getVisibleRect();
      Dimension size = getSize();
      Insets i = new Insets(r.y + this.myScrollInsets.top, r.x + this.myScrollInsets.left,
                 size.height - r.y - r.height + this.myScrollInsets.bottom,
                 size.width - r.x - r.width + this.myScrollInsets.right);
      return i;
   }

   public void autoscroll(Point location)
   {
      JScrollPane scroller =
         (JScrollPane)SwingUtilities.getAncestorOfClass(JScrollPane.class, this);
      if (scroller != null)
      {
         JScrollBar hBar = scroller.getHorizontalScrollBar();
         JScrollBar vBar = scroller.getVerticalScrollBar();
         Rectangle r = getVisibleRect();
         if (location.x <= r.x + this.myScrollInsets.left)
         {
            // Need to scroll left
            hBar.setValue(hBar.getValue() - hBar.getUnitIncrement( -1));
         }
         if (location.y <= r.y + this.myScrollInsets.top)
         {
            // Need to scroll up
            vBar.setValue(vBar.getValue() - vBar.getUnitIncrement( -1));
         }
         if (location.x >= r.x + r.width - this.myScrollInsets.right)
         {
            // Need to scroll right
            hBar.setValue(hBar.getValue() + hBar.getUnitIncrement(1));
         }
         if (location.y >= r.y + r.height - this.myScrollInsets.bottom)
         {
            // Need to scroll down
            vBar.setValue(vBar.getValue() + vBar.getUnitIncrement(1));
         }
      }
   }

   // --------------------------------------------------------------------------
   //                     Public Methods
   // --------------------------------------------------------------------------
   
   public DirectoryTreeNode getRootNode()
   {
	  return (DirectoryTreeNode)super.getModel().getRoot();
   }
   
   
   public void updateTreeLabels()
   {
      EventQueue.invokeLater(new Runnable()
      {
         public void run()
         {
            treeDidChange();
         }
      });
   }

    /**
    * Finds the node representing this path.
    * @param fileOfNode path to find
    * @return null if nothing found.
    */
   public DirectoryTreeNode findANode(File fileOfNode)
   {
	  //System.out.println("DirectoryTree.findANode: " + fileOfNode);
      String absolutePath = fileOfNode.getAbsolutePath();
      DefaultTreeModel model = (DefaultTreeModel)this.getModel();
      Object anItem = null;
      DefaultMutableTreeNode rootNode = (DefaultMutableTreeNode)model.getRoot();
      Enumeration<?> theEnum = rootNode.breadthFirstEnumeration();
      while (theEnum.hasMoreElements())
      {
			anItem = theEnum.nextElement();
			if (anItem instanceof DirectoryTreeNode)
			{
				DirectoryTreeNode node = (DirectoryTreeNode)anItem;
				if (node.getAbsolutePath().equals(absolutePath))
				{
				    return node;
				}
			}
      }
   return null;
   }

   // --------------------------------------------------------------------------
   //                     Package Methods
   // --------------------------------------------------------------------------
   TreePath[] getExpandedDescendants(TreeNode parent, TreePath butNotMe)
   {
   	int listSize = 33;
      ArrayList<TreePath> expandedList = new ArrayList<TreePath>(listSize);
      Enumeration<TreePath> theEnum = getExpandedDescendants(new TreePath(parent));
      if (theEnum != null)
      {
	      while (theEnum.hasMoreElements())
	      {
	         TreePath path = theEnum.nextElement();
	         if (!path.equals(butNotMe))
	         {
	            expandedList.add(path);
	         }
	      }
      }
      TreePath[] rtnMe = new TreePath[expandedList.size()];
      expandedList.toArray(rtnMe);
      return rtnMe;
   }

   void expandPath(TreePath[] expand)
   {
      for (int i = 0; i < expand.length; i++)
      {
         setExpandedState(expand[i], true);
      }
   }

}
