/*
 * Copyright (c) 2007 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.gerhardb.lib.dirtree;

import java.io.File;
import java.io.FileNotFoundException;
import javax.swing.*;
import javax.swing.tree.DefaultTreeModel;

import org.gerhardb.lib.util.ICancel;
import org.gerhardb.lib.util.startup.AppStarter;
import org.gerhardb.lib.util.startup.ILoadingMessage;
import org.gerhardb.lib.util.startup.Loading;

/**
 * Utility performing various actions.
 */
public class TreeManager
{
 	private DirectoryTreeNode myRootNode;	
	DirectoryTree myTree;
	boolean iRememberDirectories;
	protected DTNReaderWriter myReaderWriter;
	protected JFrame myTopFrame;
	
	// ==========================================================================
	// Constructor
	// ==========================================================================
	public TreeManager(JFrame top, DirectoryTree tree, DTNReaderWriter readWrite, boolean rememberDirectories)
	{
		this.myTopFrame = top;
		this.myTree = tree;
		this.myReaderWriter = readWrite;
		this.iRememberDirectories = rememberDirectories;
	}
	
	// ==========================================================================
	// Public
	// ==========================================================================
	public DTNReaderWriter getDTNReaderWriter()
	{
		return this.myReaderWriter;
	}

	public DirectoryTreeNode getRootNode()
	{
		return this.myRootNode;
	}
	
	public void reloadAllNodes()
	{
		this.myRootNode.reloadNodes();
	}
	
	public void store()
	throws Exception
	{
		this.myReaderWriter.getWriter().store(this.myRootNode);
	}
	
	public void loadSavedTreeDirectory(final String savedTreeDirectoryName)
	{
		final Loading newLoading = new Loading(this.myTopFrame, "Tree Manager");
		Runnable runMe = new Runnable()
		{
			public void run()
			{
				File rootDirFile = new File(savedTreeDirectoryName);
				showViewTree(
						rootDirFile, 
						newLoading);
				newLoading.dispose();
			}
		};
		newLoading.start(runMe);				
	}
	
	public void showViewTree(
			File rootDirectoryOrFile, 
			Loading loading)
	{
		try
		{
			this.myRootNode = makeRootNode(rootDirectoryOrFile, loading);			
			DefaultTreeModel daModel = new DefaultTreeModel(this.myRootNode);
			this.myTree.setModel(daModel);
		}
		catch (FileNotFoundException ex)
		{
			ex.printStackTrace();
		}
	}
	
	private DirectoryTreeNode makeRootNode(
			File rootDirectory, 
			Loading loading)
			throws FileNotFoundException, SecurityException
	{
		if (rootDirectory.exists() == false)
		{
			throw new FileNotFoundException(AppStarter.getString("TreeManager.1") + " " + rootDirectory //$NON-NLS-1$ //$NON-NLS-2$
					+ " " + AppStarter.getString("TreeManager.2")); //$NON-NLS-1$ //$NON-NLS-2$
		}

		// Don't use isDir() since a drive (which is a valid root)
		// is NOT a directory in Windows!
		if (rootDirectory.isFile())
		{
			throw new FileNotFoundException(rootDirectory
					+ " " + AppStarter.getString("TreeManager.3")); //$NON-NLS-1$ //$NON-NLS-2$
		}

		DirectoryTreeNode.resetRecursiveLoading();

		loading.setCancel(new ICancel()
		{
			public void cancel()
			{
				DirectoryTreeNode.cancelRecursiveLoading();
			}
		});

		// Populates the Tree.
		DirectoryTreeNode rootNode = null;
		if (this.iRememberDirectories) //this.myCoordinator.myBasicOptionsManager.getRememberDirectories())
		{
			// msg is set by subroutine.
			rootNode = populateFromStored(
					rootDirectory, this.myTree, loading);
		}
		else
		{
			loading.setText(AppStarter.getString("TreeManager.5")); //$NON-NLS-1$
		}

		//java.util.Date startTime = new Date();
		if (rootNode == null)
		{
			//System.out.println("JIBS STARTUP: MANUAL POPULATION"); //$NON-NLS-1$
			rootNode = new DirectoryTreeNode(this.myTree, rootDirectory, loading);
			
			// The NodeAdded() below will update the loading screen.
			// On startup this is already running in a non-swing thread.
			rootNode.populate(loading);

		} 
		//java.util.Date stopTime = new Date();
		//long elapsed = (stopTime.getTime() - startTime.getTime())/1000;
		//System.out.println("JIBS STARTUP: Seconds to Load ROOT NODE: " + elapsed); //$NON-NLS-1$
		
		// Turn cancel back off
		loading.setCancel(null);
		
		return rootNode;		
	}
	
	// ==========================================================================
	// Private
	// ==========================================================================
	// --------------------------------------------------------------------------
   //                     Store and Retrieve
   // --------------------------------------------------------------------------

   
   /**
    * Used to restore from a file store.
    * @param rootDirectory
    * @return
    */
   private DirectoryTreeNode populateFromStored(
		   File rootDirectory, DirectoryTree tree, ILoadingMessage loadingMsg)
   {
	   DTNReader reader = this.myReaderWriter.getReader(rootDirectory, tree, loadingMsg);
		return reader.getRootNode(); 
   }  
   
}
