/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package org.gerhardb.lib.dirtree.filelist;

import java.awt.Cursor;
import java.util.List;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.dnd.DnDConstants;
import java.awt.dnd.DragGestureListener;
import java.awt.dnd.DragSource;
import java.awt.dnd.DragSourceAdapter;
import java.awt.dnd.DragSourceDragEvent;
import java.awt.dnd.DragSourceDropEvent;
import java.awt.dnd.DragSourceListener;
import java.io.File;
import java.util.ArrayList;


/**
 * Search for: DROPdoneHERE
 * 
 * The list of files for a particular directory, generally shown on the left
 * side of the main view screen from which you can drag onto the file tree.
 * 
 * This does the transfer for drag & drop.  
 * This is the origination and list of files to be dropped.
 * Gets dropped onto ExtendedFileTree and processed by drop().
 * 
 * Very important that what is dropped is a List of Files with 
 * DataFlavor.javaFileListFlavor.  That way drag and drop with the 
 * native file system is enabled.
 */
public class FileDrag implements Transferable
{
	public static final DataFlavor FILE_DRAG_DATA_FLAVOR = new
    DataFlavor(DataFlavor.javaJVMLocalObjectMimeType +
            "; class=org.gerhardb.lib.dirtree.filelist.FileDrag", //$NON-NLS-1$  
            "Local FileDrag"); //$NON-NLS-1$
	protected DataFlavor[] myDataFlavors = 
	{ DataFlavor.javaFileListFlavor, FILE_DRAG_DATA_FLAVOR };
	private DragSourceListener myDSListener = new DSListener();
	private FileList myFileList;
	
	// ==========================================================================
	// Constructor
	// ==========================================================================
	public FileDrag(FileList list)
	{
		this.myFileList = list;
		DragSource ds = new DragSource();
		DragGestureListener dgl =  
			new LinklessDragListener(this.myDSListener, this, ds);
		ds.createDefaultDragGestureRecognizer(
				list,
				DnDConstants.ACTION_COPY_OR_MOVE, 
				dgl);
	}
	
	public DragSourceListener getDragSourceListener()
	{
		return this.myDSListener;
	}
	
	public List<File> getFileList()
	{
		Object[] selected = this.myFileList.myDragSelections; // COULD USE WORK
		if (selected == null)
		{
			//System.out.println("FileDrag getting Selected Values since no drag found...");
			selected = this.myFileList.getSelectedValues();   // COULD USE WORK
		}
		else
		{
			//System.out.println("FileDrag FOUND SOME DRAG SELECTIONS!!!");			
		}
		//System.out.println("FileDrag.getFileList().length: " + selected.length);

		// OK, we don't need this anymore so reset.
		this.myFileList.myDragSelections = null;  // COULD USE WORK

		//System.out.println(".... selected length: " + selected.length);
		ArrayList<File> droppedList = new ArrayList<File>(selected.length);
		for (int i = 0; i < selected.length; i++)
		{
			// Not sure why this happens, but it does.
			if (selected[i] != null)
			{
				// Java "convenently" catches an exception from this method
				// and then never reports it, so you don't even know.
				// Delivish to diagnose.
				try
				{
					if (selected[i] instanceof File)
					{
						droppedList.add((File)selected[i]);
					}
				}
				catch (Exception ex)
				{
					System.out.println("EXCEPTION ************  FileList getTransferData"); //$NON-NLS-1$
					System.out.println(ex.getMessage());
				}
			}
		}
		//System.out.println(">>>>>>>>>>> FileList size: " + droppedList.size());
		return droppedList;
	}
	
	// ==========================================================================
	// Transferable Implementation
	// ==========================================================================

	/**
	 * This is called from a drop.
	 */
	public Object getTransferData(DataFlavor flavor)
	{
		//System.out.println("FileDrag.getTransferData!!!!!!!");
		if (FILE_DRAG_DATA_FLAVOR.equals(flavor))
		{
			//System.out.println("Returning LOCAL Flavor"); //$NON-NLS-1$
			return this;
		}
		return getFileList();
	}


	public DataFlavor[] getTransferDataFlavors()
	{
		//System.out.println("FileDrag returing DataFlavors");
		return this.myDataFlavors;
	}

	public boolean isDataFlavorSupported(DataFlavor flavor)
	{
		//System.out.println("FileDrag isDataFlavorSupported");
		for (int i = 0; i < this.myDataFlavors.length; i++)
		{
			if (this.myDataFlavors[i].equals(flavor)) { return true; }
		}
		return false;
	}

	// ==========================================================================
	// DragSourceListener
	// ==========================================================================
	class DSListener extends DragSourceAdapter
	{
		/**
		 * This is where we find out the drag worked!!!!
		 */
		@Override
		public void dragDropEnd(DragSourceDropEvent event)
		{			
			event.getDragSourceContext().setCursor(Cursor.getPredefinedCursor( Cursor.DEFAULT_CURSOR ));
		}
		
		@Override
		public void dropActionChanged(DragSourceDragEvent event)
		{
			//System.out.println( "FileDrag.dropActionChanged" );
			switch ( event.getUserAction())
			{
			case DnDConstants.ACTION_MOVE:
				//System.out.println("FileDrag Accepting Drop: MOVE");
				event.getDragSourceContext().setCursor(DragSource.DefaultMoveDrop);
				break;
			case DnDConstants.ACTION_COPY:
				//System.out.println("FileDrag Accepting Drop: COPY");
				event.getDragSourceContext().setCursor(DragSource.DefaultCopyDrop);
				break;
			default:		
			}
		}


		 /*
		 public void dragExit(DragSourceEvent dse)
		 {
			 System.out.println("ScrolledSlider.dragExit");
		 }
		 
		 public void dragEnter(DragSourceDragEvent dsde)
		 {
		 System.out.println("ScrolledSlider.dragEnter");
		 }

		 public void dragOver(DragSourceDragEvent dsde)
		 {
		 //System.out.println( "ScrolledSlider.dragOver" );
		 }

		 */
	}

}
