/*
 * Copyright (c) 2007 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package  org.gerhardb.lib.dirtree.filelist;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.InputEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.File;
import javax.swing.DefaultListCellRenderer;
import javax.swing.DropMode;
import javax.swing.JList;
import javax.swing.ListCellRenderer;
import javax.swing.ListSelectionModel;
import javax.swing.event.AncestorEvent;
import javax.swing.event.AncestorListener;

import org.gerhardb.lib.dirtree.DTNFile;
import org.gerhardb.lib.dirtree.ExtendedDirectoryTree;
import org.gerhardb.lib.dirtree.filelist.popup.FileNameChanger;
import org.gerhardb.lib.scroller.IScroll;
import org.gerhardb.lib.scroller.ScrollerChangeEvent;
import org.gerhardb.lib.scroller.ScrollerListener;
import org.gerhardb.lib.util.Range;

/*
 * Now Java6 uses DropMode.ON but it does not do much. Just use right mouse
 * button to get originally desired effect of not having selections go away.
 * 
 * 4468566 A fix for this bug has been made! JTextComponent, JList, JTable and
 * JTree now all have a new setDropMode(DropMode) method. This method controls
 * how the components track and indicate their drop location. The new drop modes
 * and the components that support them are:
 * 
 * USE_SELECTION - JTextComponent, JList, JTable, JTree ON - JList, JTable,
 * JTree INSERT - JTextComponent, JList, JTable, JTree INSERT_ROWS - JTable
 * INSERT_COLS - JTable ON_OR_INSERT - JList, JTable, JTree ON_OR_INSERT_ROWS -
 * JTable ON_OR_INSERT_COLS - JTable
 * 
 * The default drop mode for all components will be USE_SELECTION (which uses
 * the selection to track the drop location), for backward compatibility.
 * However, the new drop modes offer better options for tracking and indicating
 * the drop location, without affecting the component's selection.
 * 
 * Each of these components will also have a new getDropLocation() method which
 * will return a subclass of TransferHandler.DropLocation, customized to
 * indicate the drop location as it makes sense for that component type. This
 * getDropLocation() method is intended for use by UI code that renders the drop
 * location. Additionally, the DropLocation will be passed to canImport,
 * shouldIndicateAnyway, and importData of TransferHandler so that the behavior
 * can be customized based on drop location.
 */

/**
 * The list of files for a particular directory, generally shown on the left
 * side of the main view screen from which you can drag onto the file tree.
 */
public class FileList extends javax.swing.JList implements AncestorListener,
		ScrollerListener
{
	protected static final Color SHOWING_BACKGROUND = new Color(235, 235, 235);
	protected static Color myDefaultBackground;

	protected boolean isPopulated = false;

	protected Object[] myDragSelections; // USED BY FileDrag
	
	protected FileDrag myFileDrag;
	protected FileListPlugins myPlugins;
	private ExtendedDirectoryTree myExtendedDirectoryTree;	
	
	// ==========================================================================
	// Constructor
	// ==========================================================================
	public FileList(FileListPlugins plugins, ExtendedDirectoryTree edt)
	{
		super();
		this.myPlugins = plugins;	
		this.myExtendedDirectoryTree = edt;
		this.myFileDrag = new FileDrag(this);
		this.myPlugins.getScroller().addScrollerListener(this);

		ListCellRenderer lcr = new FileRenderer();
		myDefaultBackground = lcr.getListCellRendererComponent(this,
				"", 0, false, false) //$NON-NLS-1$
				.getBackground();

		super.setCellRenderer(lcr);
		super.addAncestorListener(this);
		super.setMinimumSize(new Dimension(100, 100));

		super.addKeyListener(this.myPlugins.getScroller().getScrollerKeyListener()); // Added back aug 3
		//super.addKeyListener(new AdaptedKey()); // removed aug 3
		
		FileListSelectionModel model = new FileListSelectionModel();
		super.setSelectionModel(model);
		this.myPlugins.getScroller().setFileList(this);
		
		super.setDropMode(DropMode.ON);

		super.addMouseListener(new AdaptedMouse());
	}

	public FileDrag getTransferable()
	{
		return this.myFileDrag;
	}

	/**
	 * Override so as to not make too wide.
	 */
	@Override
	public Dimension getPreferredSize()
	{
		Dimension oldSize = super.getPreferredSize();
		return new Dimension(100, (int) oldSize.getHeight());
	}

	public FileListSelectionModel getFileListSelectionModel()
	{
		return (FileListSelectionModel)getSelectionModel();
	}
	
	public File[] getSelectedFiles()
	{
		Object[] selected = super.getSelectedValues();
		if (selected.length == 0)
		{
			return new File[]{this.myPlugins.getScroller().getCurrentFile()};
		}
		File[] files = new File[selected.length];
		for (int i=0;i<files.length;i++)
		{
			//System.out.println(selected[i].getClass());
			files[i] = ((DTNFile)selected[i]).getAbsoluteFile();
		}
		return files;
	}
	
	public File[] getAllFiles()
	{
		return this.myPlugins.getScroller().getSortedFiles();
	}

	/**
	 * Used to get a drag to drag the selection BEFORE pick to avoid the
	 * SelectAll not working.
	 * @param e MouseEvent
	 */
	@Override
	protected void processMouseEvent(MouseEvent e)
	{
		boolean letSuperClassProcessEvent = true;
		//System.out.println("Mouse event ");
		if (e.getID() == MouseEvent.MOUSE_PRESSED)
		{
			//System.out.println("Processing Mouse Pressed");
			
			// Have to do this on a pressed because its processed by release.
			//itemClickedOnWasSelected = getSelectionModel().isSelectedIndex(locationToIndex(e.getPoint()));
			
			if ((e.getModifiersEx() & InputEvent.CTRL_DOWN_MASK) != 0)
			{
				//System.out.println("control down");
			}
			else if ((e.getModifiersEx() & InputEvent.SHIFT_DOWN_MASK) != 0)
			{
				//System.out.println("shift down");            
			}
			else 
			{
				// If shift or ctrl is down, we will assume the user is NOT
				// doing a SELECT ALL deal.
				// In fact, we only want this effect for a SELECT ALL deal.
				int[] picked = this.getSelectedIndices();
				//System.out.println("FileList picked: " + picked.length);
				//System.out.println("FileList size: " + getModel().getSize());
				if (picked.length > 0)
				{
					//System.out.println("Storing Selected Values in Drag Selections.");
					this.myDragSelections = getSelectedValues();
					// can't return here because user may not be doing a drop!
				}
				else
				{
					// We get here if only one file is being moved.
					Point p = e.getPoint();
					
					int index = locationToIndex(p);					
					Object obj = this.getModel().getElementAt(index);
					Rectangle bounds = getCellBounds(index, index);
					if (null != bounds && bounds.contains(p.x, p.y))
					{
						//System.out.println(obj);
						Object[] objArray = {obj};
						this.myDragSelections = objArray;						
					}
				}				
				repaint(getBounds());// Dec 2010.  I'm not sure why this is here...
			}
		}
		
		if (e.getID() == MouseEvent.MOUSE_RELEASED)
		{
			//System.out.println("Processing Mouse Released");

			// Must not be a drag, so reset.
			this.myDragSelections = null;
			repaint(getBounds());
		}
		
		if (letSuperClassProcessEvent)
		{
			super.processMouseEvent(e);
		}
	}

	// ==========================================================================
	// Overriden Methods
	// ==========================================================================

	public void setModel() // NO_UCD
	{
		//System.out.println("FileList setModel");
		if (isVisible())
		{
			populate();
		}
		else
		{
			this.isPopulated = false;
		}
	}

	// ==========================================================================
	// AncestorListener
	// ==========================================================================
	public void ancestorAdded(AncestorEvent event)
	{
		if (!this.isPopulated)
		{
			populate();
		}
	}

	public void ancestorRemoved(AncestorEvent event)
	{
		// We don't care
	}

	public void ancestorMoved(AncestorEvent event)
	{
		// We don't care
	}

	// ==========================================================================
	// ScrollerListener
	// ==========================================================================

	public void scrollerChanged(ScrollerChangeEvent ce)
	{
		ListSelectionModel model = super.getSelectionModel();
		switch (ce.getEventType())
		{
			case ScrollerChangeEvent.LIST_RELOADED:
			case ScrollerChangeEvent.ONE_FILE_REMOVED:  // Unclear why this is here...
				//System.out.println("Clear: LIST_RELOADED");
				super.clearSelection();  
				return; // THIS IS WHAT WORKED!!!!  
				// If return is not here, the bouncing blue stuff will screw up clearing on a directory change.
		}
		// Fixes the bouncing blue selection item in the File List
		// This also means that a single click will NOT select anything on the file list.
		// This is good since a single click is likely to navigate to the start of the 
		// next set of file vs making a selection.  To make a selection you can use boxes,
		// CTRL or SHIFT.
		if (model.isSelectionEmpty())
		{
			int index = this.myPlugins.getScroller().getValueZeroBased();
			model.setLeadSelectionIndex(index);
			model.setAnchorSelectionIndex(index);
		}
	}
	
	// ==========================================================================
	// Public Functions
	// ==========================================================================

	public void sliderAdjusted()
	{
		//System.out.println("FileList sliderAdjusted");
		super.repaint();
		int index = this.myPlugins.getScroller().getValueZeroBased(); //.getValue();
		super.ensureIndexIsVisible(index);
		int first = super.getFirstVisibleIndex();
		int last = super.getLastVisibleIndex();
		//System.out.println( "FileList: " + first + " -- " + last );
		if (first > -1 && last > 0)
		{
			int range = last - first;
			int middle = (range / 2);
			int show = index + middle - 1;
			super.ensureIndexIsVisible(show);
			show = index - middle + 1;
			super.ensureIndexIsVisible(show);
		}
		//System.out.println("FileList sliderAdjusted FINISHED");
	}
	
	// ==========================================================================
	//   protected
	// ==========================================================================	
	protected void populate()
	{
		//System.out.println("FileList populate");
		if (this.myPlugins.getScroller() == null) { return; }
		super.setModel(this.myPlugins.getScroller());
		this.isPopulated = true;
	}

	protected void updateSelection(MouseEvent e)
	{
		//System.out.println("FileList Selected");
		int index = super.locationToIndex(e.getPoint());
		if (e.getClickCount() == 2)
		{
			super.clearSelection();	
		}
		else // Single mouse click
		{
			// If the current item is selected, deselect it.
			ListSelectionModel listSelection = super.getSelectionModel();
			//if (itemClickedOnWasSelected) // August 23 2009
			if (!e.isControlDown() && !e.isShiftDown() && !e.isAltDown())
			{
				listSelection.removeSelectionInterval(index, index);
			}
		}
		this.myPlugins.getScroller().setValue(index); // added  july 4 2009    
	}

	public void selectFiles(File[] filesToSelect)
	{
		if (filesToSelect == null) { return; }
		File[] sortedFiles = this.myPlugins.getScroller().getSortedFiles();
		for (int i = 0; i < filesToSelect.length; i++)
		{
			for (int sortedIndex = 0; sortedIndex < sortedFiles.length; sortedIndex++)
			{
				if (filesToSelect[i] != null && sortedFiles[sortedIndex] != null)
				{
					if (filesToSelect[i].equals(sortedFiles[sortedIndex]))
					{
						super.addSelectionInterval(sortedIndex, sortedIndex);
					}
				}
			}
		}
	}

	// ==========================================================================
	// Rename and Mouse Stuff
	// ==========================================================================	

	public void renameSomeFiles()
	{
		Object[] selected = super.getSelectedValues();
		if (selected.length == 0)
		{
			selected = new File[]{this.myPlugins.getScroller().getCurrentFile()};
		}
		popUpRename(selected, true);
	}
	
	class AdaptedMouse extends MouseAdapter
	{
		@Override
		public void mousePressed(MouseEvent e)
		{
			// This did NOT work because the selection changes before this
			// is ever called. BUT processMouseEvent is called BEFORE
			// the selection changes.
			//myDragSelections = getSelectedValues();
			//e.consume();
			//System.out.println("Mouse Pressed");
		}

		@Override
		public void mouseReleased(MouseEvent e)
		{
			//System.out.println("Mouse Released");
			// Must not be a drag, so reset.
			//myDragSelections = null;
		}

		@Override
		public void mouseClicked(MouseEvent e)
		{
			if ((e.getModifiers() & InputEvent.BUTTON2_MASK) != 0)
			{
				popUp(e);
				return;
			}
			else if ((e.getModifiers() & InputEvent.BUTTON3_MASK) != 0)
			{
				popUp(e);
				return;
			}
			else
			{
				updateSelection(e);
				return;
			}
		}
	}

	// ==========================================================================
	//   protected
	// ==========================================================================	
	protected void popUp(MouseEvent e)
	{
		//System.out.println("FileList popUp");
		Object[] selected = super.getSelectedValues();
		if (selected.length > 1)
		{
			popUpRename(selected, false);
		}
		else
		{
			e.getPoint();
			int index = this.locationToIndex(e.getPoint());
			Object obj = this.getModel().getElementAt(index);

			if (!(obj instanceof File)) { return; }

			File[] fileList = new File[]{(File)obj};
			popUpRename(fileList, false);
		}
	}

	protected void popUpRename(Object[] selected, boolean showRenameTabAtTop)
	{
		File[] fileList = new File[selected.length];
		for (int i = 0; i < selected.length; i++)
		{
			if (selected[i] != null && selected[i] instanceof File)
			{
				fileList[i] = (File) selected[i];
			}
		}
		FileNameChanger fnc = new FileNameChanger(fileList);
		fnc.showDialog(this.myPlugins, this.myExtendedDirectoryTree, true, showRenameTabAtTop);
		if (fileList.length > 1 && fnc.renamed())
		{
			File[] changedFiles = fnc.getChangedNames();
			if (changedFiles == null || changedFiles.length == 1)
			{
				this.myPlugins.getScroller().reloadScroller(0, IScroll.KEEP_CACHE);
			}
			else
			{
				this.myPlugins.getScroller().reloadScroller(0, IScroll.KEEP_CACHE);
			}
			super.clearSelection();
			selectFiles(changedFiles);
			if (changedFiles != null)
			{
				this.myPlugins.getScroller().selectFile(changedFiles[0]);
			}
		}
		this.myPlugins.setWaitCursor(false);
	}

	// ==========================================================================
	// Inner Class: Actions
	// ==========================================================================
	protected class FileRenderer extends DefaultListCellRenderer
	{
		@Override
		public Component getListCellRendererComponent(JList list, Object value,
				int index, boolean isSelected, boolean cellHasFocus)
		{
			Component rtnMe = super.getListCellRendererComponent(list, value,
					index, isSelected, cellHasFocus);

			// Set defaults
			//rtnMe.setForeground(Color.BLACK);
			//rtnMe.setBackground(Color.WHITE);

			if (value != null && value instanceof File)
			{
				super.setText(((File) value).getName());
			}

			if (FileList.this.myPlugins.getScroller() == null) { return rtnMe; }

			// Set Foreground
			if (FileList.this.myPlugins.getScroller().getValueZeroBased() == index)
			{
				rtnMe.setForeground(Color.RED);
			}

			// Set Background
			Range range = FileList.this.myPlugins.showingIndexes();
			if (range.getFirst() > -1)
			{
				if (index >= range.getFirst() && index <= range.getLast())
				{
					if (rtnMe.getBackground() == myDefaultBackground)
					{
						rtnMe.setBackground(SHOWING_BACKGROUND);
					}
				}
			}
			return rtnMe;
		}
	}

}
