/*
 * Copyright (c) 2007 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.gerhardb.lib.dirtree.filelist;

import java.awt.Window;
import java.io.File;
import java.util.ArrayList;

import javax.swing.JButton;
import javax.swing.JFileChooser;
import javax.swing.JToolBar;

import org.gerhardb.lib.dirtree.*;
import org.gerhardb.lib.dirtree.rdp.PathManager;
import org.gerhardb.lib.dirtree.rdp.TreeManagerSet;
import org.gerhardb.lib.io.FilenameFileFilter;
import org.gerhardb.lib.scroller.FileListFactory;
import org.gerhardb.lib.scroller.ListMaker;
import org.gerhardb.lib.util.startup.ActiveActions;
import org.gerhardb.lib.util.startup.AppStarter;

/**
 * Utility performing various actions.
 */
public class FileListManager
{
	TreeManagerSet myTreeManagerSet;

	ArrayList<ListMaker> myBackList = new ArrayList<ListMaker>(40);
	ArrayList<ListMaker> myForwardList = new ArrayList<ListMaker>(10);
   FilenameFileFilter myFilter;

   // Can't be null, but set after starting...
	JButton myBackBtn = new JButton("dummy");
	JButton myForwardBtn = new JButton("dummy");
	// ==========================================================================
	// Constructor
	// ==========================================================================
	public FileListManager(TreeManagerSet tm, FilenameFileFilter filter)
	{
		this.myTreeManagerSet = tm;
      this.myFilter = filter;
	}

	// ==========================================================================
	// Public
	// ==========================================================================	
	public boolean setTargetDir(String dir)
	{
		if (dir == null) { return false; }
		return setTargetDir(new File(dir));
	}

	public boolean setTargetDir(File dir)
	{
		if (dir == null) { return false; }
		try
		{
			this.myTreeManagerSet.getPathManager().setDirectory(PathManager.DIR_LAST_VIEWED, dir.getAbsolutePath());
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
		}
		
		try
		{
			ListMaker maker = null;
			DirectoryTreeNode node = this.myTreeManagerSet.getRDPplugins().getExtendedDirectoryTree().findANode(dir);
			if (node == null)
			{
				maker = new FileListFactory(dir);
				displayScroll(maker, true);
				return true;
			}
			return setTargetDir(node, false);
		}
		catch (Exception ex)
		{
			System.out.println("ScrollFactory: " + ex.getMessage()); //$NON-NLS-1$
		}
		return false;
	}

	public boolean setTargetDir(DirectoryTreeNode node, boolean nonImage)
	{
		if (node == null) { return false; }
		DirectoryTreeNode[] nodeArray = new DirectoryTreeNode[] { node };
		return setTargetDir(nodeArray, nonImage);
	}

	public boolean setTargetDir(DirectoryTreeNode[] nodeList, boolean nonImage)
	{
		if (nodeList == null || nodeList.length == 0) { return false; }
		
		try
		{
			this.myTreeManagerSet.getPathManager().setDirectory(PathManager.DIR_LAST_VIEWED, nodeList[0].getAbsolutePath());
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
		}
		
		try
		{
			// update defaults for the usual single directory selection,
			// and make sure count match.
			ListMaker maker = new DTNListFactory(nodeList, nonImage, this.myFilter);
			displayScroll(maker, true);
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
			return false;
		}
		return true;
	}
	
	public void askUserForViewDirectory(Window window) 
	{
		File newDirectory = null;

		// -------------------------------------------------------------------------
		// Look up the New Directory
		// -------------------------------------------------------------------------
		JFileChooser chooser = new JFileChooser();
		chooser.setDialogTitle(AppStarter.getString("Popups.10")); //$NON-NLS-1$
		chooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);

		String oldViewDirName = this.myTreeManagerSet.getRDPplugins().getPathManager().getDirectoryAbsolute(PathManager.DIR_LAST_VIEWED);
		if (oldViewDirName != null)
		{
			File oldViewDir = new File(oldViewDirName);
			if (oldViewDir.exists() && oldViewDir.isDirectory())
			{
				chooser.setCurrentDirectory(oldViewDir);
			}
		}

		int returnVal = chooser.showOpenDialog(window);
		if (returnVal == JFileChooser.APPROVE_OPTION)
		{
			try
			{
				newDirectory = chooser.getSelectedFile();
			}
			catch (Exception ex)
			{
				// So remote as to ignore
				ex.printStackTrace();
			}
		}

		// -------------------------------------------------------------------------
		// Take action based on New Directory
		// -------------------------------------------------------------------------
		if (newDirectory == null || !newDirectory.exists()
				|| !newDirectory.isDirectory()) { return; }

		setTargetDir(newDirectory);
	}

	public void addButtonsToToolBar(JToolBar bar, ActiveActions aa)
	{
		this.myBackBtn = aa.getToolBarButton("view", "back"); //$NON-NLS-1$ //$NON-NLS-2$
		this.myForwardBtn = aa.getToolBarButton("view", "forward");		 //$NON-NLS-1$ //$NON-NLS-2$
		
		statusForwardAndBackButtons();
		
		bar.add(this.myBackBtn);
		bar.add(this.myForwardBtn);
	}
	
	public void forward()
	{
		if (this.myForwardList.isEmpty()) { return; }
		int lastItem = this.myForwardList.size() - 1;
		ListMaker maker = this.myForwardList.remove(lastItem);
		this.myBackList.add(maker);
		statusForwardAndBackButtons();
		displayScroll(maker, false);
	}

	public void back()
	{
		// if ( myBackList.isEmpty() ){return;}
		int size = this.myBackList.size();
		if (size > 0)
		{
			// Move current directory.
			this.myForwardList.add(this.myBackList.remove(size - 1));
		}
		ListMaker maker = null;
		if (size > 1)
		{
			// Get directory before current directory.
			// Don't remove it because it becomes the current diretory.
			maker = this.myBackList.get(size - 2);
		}
		statusForwardAndBackButtons();
		displayScroll(maker, false);
	}
	
	// ==========================================================================
	// Private
	// ==========================================================================

	private void statusForwardAndBackButtons()
	{
		//System.out.println("statusForwardAndBackButtons()");
		this.myForwardBtn.setEnabled(!this.myForwardList.isEmpty());
		if (this.myBackList.size() > 1)
		{
			this.myBackBtn.setEnabled(true);
		}
		else
		{
			this.myBackBtn.setEnabled(false);			
		}
	}
	
	private void displayScroll(ListMaker maker, boolean newDirectory)
	{
		this.myTreeManagerSet.getRDPplugins().setWaitCursor(true);
	
		this.myTreeManagerSet.getScroller().setListMaker(maker);

		// Update this display
		this.myTreeManagerSet.getRDPmanager().getIScrollDirTree().fileListUpdated();
		this.myTreeManagerSet.getMoveManager().setUndoOff();
		if (newDirectory)
		{
			this.myForwardList.clear();
			this.myBackList.add(maker);
			while (this.myBackList.size() > 50)
			{
				this.myBackList.remove(0);
			}
		}
		statusForwardAndBackButtons();
		this.myTreeManagerSet.getScroller().requestFocus();

		this.myTreeManagerSet.getRDPplugins().setWaitCursor(false);
	}	
}
