/*
 * Copyright (c) 2007 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.gerhardb.lib.dirtree.filelist.popup;

import java.text.DecimalFormat;

import org.gerhardb.lib.io.FileRename;

/**
 *
 * @author  Gerhard Beck
 */
public class FileName
{
	DecimalFormat myDecimalFormat;
	String myBeforeNumber;
	String myAfterNumber;

	public FileName(int listLength)
	{
		this.myDecimalFormat = FileRename.getDecimalFormat(listLength);
	}

	/**
	 * 
	 * @param oldName Don't include the path.
	 */
	public void setTemplate(String template)
	{
		this.myBeforeNumber = null;
		this.myAfterNumber = null;
		if (template == null || template.length() == 0)
		{
			// do nothing
		}
		else if (template.indexOf("#") > -1) // contains() is 1.5 or greater //$NON-NLS-1$
		{
			//System.out.println("CONTAINS #");
			StringBuffer beforeBuff = new StringBuffer(100);
			StringBuffer afterBuff = new StringBuffer(100);
			boolean before = true;

			// Copy all characters before first # to beforeBuff.
			// Strip all other # and put rest into afterBuff.
			for (int i = 0; i < template.length(); i++)
			{
				char aChar = template.charAt(i);
				if (aChar == '#')
				{
					before = false;
				}
				else
				{
					if (before)
					{
						beforeBuff.append(aChar);
					}
					else
					{
						afterBuff.append(aChar);
					}
				}
			}
			this.myBeforeNumber = beforeBuff.toString();
			this.myAfterNumber = afterBuff.toString();
		}
		else
		{
			this.myBeforeNumber = template + "-"; //$NON-NLS-1$
		}
	}

	public String getDecimal(int i, boolean html)
	{
		if (html)
		{
			return "<big><b color=\"red\">" + this.myDecimalFormat.format(i) //$NON-NLS-1$
					+ "</b></big>"; //$NON-NLS-1$
		}
		return this.myDecimalFormat.format(i);
	}

	public String getNewName(int i, String oldName)
	{
		return getNewName(i, oldName, false, null);
	}
	
	public String getNewNameHTML(int i, String oldName)
	{
		return getNewName(i, oldName, true, null);
	}
	
	public String getNewName(int i, String oldName, boolean html, String ending)
	{
		if (html)
		{
			return "<html>" + getUnwrappedNewName(i, oldName, true, ending) + "</html>"; //$NON-NLS-1$ //$NON-NLS-2$
		}
		return getUnwrappedNewName(i, oldName, false, ending);
	}

	public String getUnwrappedNewName(int i, String oldName, boolean html, String newEnding)
	{
		String baseName = null;
		if (this.myBeforeNumber == null && this.myAfterNumber == null)
		{
			baseName = getDecimal(i, html);
		}
		else if (this.myBeforeNumber == null && this.myAfterNumber != null)
		{
			baseName = getDecimal(i, html) + this.myAfterNumber;
		}
		else if (this.myBeforeNumber != null && this.myAfterNumber == null)
		{
			baseName = this.myBeforeNumber + getDecimal(i, html);
		}
		else
		{
			baseName = this.myBeforeNumber + getDecimal(i, html) + this.myAfterNumber;
		}

		// FAST RETURN if we have a new ending.
		if (newEnding != null)
		{
			// If it has a period already, don't add one.
			if ( newEnding.lastIndexOf('.') > -1)
			{
				return baseName + newEnding;	
			}
			return baseName + "." + newEnding;					 //$NON-NLS-1$
		}
		
		// OK, we don't have an ending to use, so let's compute one.
		if (oldName == null) { return baseName; }

		int lastPeriod = oldName.lastIndexOf('.');
		
		// We don't care if the file has no ending.
		// If the file has more than one ending, take the first one.
		// And we need a front and a back.
		// Also, we don't care in the impossible case of a name of '.'.
		if (lastPeriod < 0 || oldName.length() == 1) { return baseName; }

		String oldEnding = oldName.substring(lastPeriod + 1).toLowerCase();
		//System.out.println("ending: " + ending);

		return baseName + "." + oldEnding; //$NON-NLS-1$
	}

	/**
	 * Tests this class.
	 */
	public static void main(String[] args)
	{
		FileName fn = new FileName(Integer.MAX_VALUE);
		testSet(fn, null);
		System.out.println("######################################################"); //$NON-NLS-1$
		testSet(fn, ".mpg"); //$NON-NLS-1$
		System.out.println("######################################################"); //$NON-NLS-1$
		testSet(fn, "jpeg"); //$NON-NLS-1$
		System.out.println("######################################################"); //$NON-NLS-1$
		for (int i = 1; i < 1002; i++)
		{
			System.out.println(fn.getNewName(i, "a.jpg")); //$NON-NLS-1$   		
		}
	}
	
	static void testSet(FileName fn, String ending)
	{
		test(fn, null, ending);
		test(fn, "", ending); //$NON-NLS-1$
		test(fn, "#", ending); //$NON-NLS-1$
		test(fn, "###", ending); //$NON-NLS-1$
		test(fn, "onlyFRONT", ending); //$NON-NLS-1$
		test(fn, "front#back", ending); //$NON-NLS-1$
		test(fn, "front##back", ending); //$NON-NLS-1$
		test(fn, "front########back", ending); //$NON-NLS-1$
		test(fn, "front#b#ack", ending); //$NON-NLS-1$
		test(fn, "front#b#a#c#k#", ending);		 //$NON-NLS-1$
	}

	static void test(FileName fn, String template, String ending)
	{
		System.out.println("=================================================="); //$NON-NLS-1$
		System.out.println("Template: " + template); //$NON-NLS-1$
		fn.setTemplate(template);
		System.out.println(fn.getNewName(1, null, false, ending)); //$NON-NLS-1$
		System.out.println(fn.getNewName(1, "", false, ending)); //$NON-NLS-1$
		System.out.println(fn.getNewName(1, ".", false, ending)); //$NON-NLS-1$
		System.out.println(fn.getNewName(1, "aaa", false, ending)); //$NON-NLS-1$
		System.out.println(fn.getNewName(1, "a.jpg", false, ending)); //$NON-NLS-1$
		System.out.println(fn.getNewName(1, "a.JPG", false, ending)); //$NON-NLS-1$
		System.out.println(fn.getNewName(1, "a.b.c.jpg", false, ending)); //$NON-NLS-1$
		System.out.println(fn.getNewName(1, "a.cow", false, ending)); //$NON-NLS-1$  	
	}
}
