/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
package org.gerhardb.lib.dirtree.rdp;

import java.awt.event.ActionEvent;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.ButtonGroup;
import javax.swing.ButtonModel;
import javax.swing.Icon;
import javax.swing.JMenu;
import javax.swing.JRadioButton;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.event.MenuEvent;
import javax.swing.event.MenuListener;

import org.gerhardb.lib.scroller.Scroller;
import org.gerhardb.lib.util.ActionHelpers;
import org.gerhardb.lib.util.startup.ActiveActions;
import org.gerhardb.lib.util.startup.AppStarter;

/**
 * 
 */
public class SortManager 
{
	private static final String SORT_ORDER = "sortOrder"; //$NON-NLS-1$
	
	ListShowTreeCoordinator myListShowTreeCoordinator;
	private ButtonModel[] mySortButtonModels;
	
	Action[] myActions = new Action[5]; 
	
	// ==========================================================================
	// Constructor
	// ==========================================================================	
	public SortManager(ListShowTreeCoordinator lst)
	{
		this.myListShowTreeCoordinator = lst;		
		this.myActions[Scroller.SORT_OFF] = makeStartupAction("SortScreen.menu","sort", "none", new NoSort(), null); //$NON-NLS-1$ //$NON-NLS-2$
		this.myActions[Scroller.SORT_NAME_CASE_INSENSATIVE] = makeStartupAction("SortScreen.menu","sort", "insensative", new NameSortInsensative(), null); //$NON-NLS-1$ //$NON-NLS-2$
		this.myActions[Scroller.SORT_NAME_CASE_SENSATIVE] = makeStartupAction("SortScreen.menu","sort", "sensative", new NameSortSensative(), null); //$NON-NLS-1$ //$NON-NLS-2$
		this.myActions[Scroller.SORT_DATE] = makeStartupAction("SortScreen.menu","sort", "date", new DateSort(), null); //$NON-NLS-1$ //$NON-NLS-2$
		this.myActions[Scroller.SORT_LENGTH] = makeStartupAction("SortScreen.menu","sort", "length", new LengthSort(), null); //$NON-NLS-1$ //$NON-NLS-2$
	}
	
	// ==========================================================================
	//                        Basic Items
	// ==========================================================================	 
	public int getSortOrder()
	{
		return this.myListShowTreeCoordinator.getPreferences().getInt(SORT_ORDER, Scroller.SORT_NAME_CASE_INSENSATIVE);
	}

	public void setSortOrder(int order)
	{
		this.myListShowTreeCoordinator.getPreferences().putInt(SORT_ORDER, order);
	}
	
	private Action makeStartupAction(String application, String menu, String item, Action action, Icon icon)
	{
		String key = ActiveActions.makeKey(application, menu, item);
		return ActiveActions.loadAction(key, action, icon);
	}
	
	// ==========================================================================
	// SORT Menu
	// ==========================================================================

	public JMenu getSortMenu()
	{
		JMenu menu = ActionHelpers.makeMenu("sort"); //$NON-NLS-1$
		final JRadioButtonMenuItem[] menuItems = new JRadioButtonMenuItem[this.myActions.length];
		menu.addMenuListener(new MenuListener()
		{
			public void menuCanceled(MenuEvent e)
			{
				// We don't care
			}

			public void menuDeselected(MenuEvent e)
			{
				// We don't care
			}

			public void menuSelected(MenuEvent e)
			{
				switch (getSortOrder())
				{
				case Scroller.SORT_OFF:
					menuItems[Scroller.SORT_OFF].setSelected(true);
					break;
				case Scroller.SORT_NAME_CASE_INSENSATIVE:
					menuItems[Scroller.SORT_NAME_CASE_INSENSATIVE].setSelected(true);
					break;
				case Scroller.SORT_NAME_CASE_SENSATIVE:
					menuItems[Scroller.SORT_NAME_CASE_SENSATIVE].setSelected(true);
					break;
				case Scroller.SORT_DATE:
					menuItems[Scroller.SORT_DATE].setSelected(true);
					break;
				case Scroller.SORT_LENGTH:
					menuItems[Scroller.SORT_LENGTH].setSelected(true);
					break;
				}
			}
		});

		this.mySortButtonModels = new ButtonModel[this.myActions.length];
		ButtonGroup sortGroup = new ButtonGroup();
		for (int i = 0; i < this.myActions.length; i++)
		{
			menuItems[i] = new JRadioButtonMenuItem(this.myActions[i]);
			menu.add(menuItems[i]);
			sortGroup.add(menuItems[i]);
			this.mySortButtonModels[i] = menuItems[i].getModel();
		}
		
		switch (getSortOrder())
		{
		case Scroller.SORT_OFF:
			menuItems[Scroller.SORT_OFF].setSelected(true);
			break;
		case Scroller.SORT_NAME_CASE_INSENSATIVE:
			menuItems[Scroller.SORT_NAME_CASE_INSENSATIVE].setSelected(true);
			break;
		case Scroller.SORT_NAME_CASE_SENSATIVE:
			menuItems[Scroller.SORT_NAME_CASE_SENSATIVE].setSelected(true);
			break;
		case Scroller.SORT_DATE:
			menuItems[Scroller.SORT_DATE].setSelected(true);
			break;
		case Scroller.SORT_LENGTH:
			menuItems[Scroller.SORT_LENGTH].setSelected(true);
			break;
		}
		return menu;
	}
	
	public JRadioButton[] makeSortRadioButtons()
	{
		JRadioButton[] rtnMe = new JRadioButton[this.myActions.length];
		for (int i = 0; i < this.myActions.length; i++)
		{
			rtnMe[i] = new JRadioButton(this.myActions[i]);
			rtnMe[i].setModel(this.mySortButtonModels[i]);
		}
		return rtnMe;
	}
	
	class NoSort extends AbstractAction
	{
		NoSort()
		{
			super(AppStarter.getString("Actions.202")); //$NON-NLS-1$
		}

		public void actionPerformed(ActionEvent event)
		{
			setSortOrder(Scroller.SORT_OFF);
			SortManager.this.myListShowTreeCoordinator.getScroller().sort(Scroller.SORT_OFF);
		}
	}

	class NameSortInsensative extends AbstractAction
	{
		NameSortInsensative()
		{
			super(AppStarter.getString("Actions.203")); //$NON-NLS-1$
		}

		public void actionPerformed(ActionEvent event)
		{
			setSortOrder(Scroller.SORT_NAME_CASE_INSENSATIVE);
			SortManager.this.myListShowTreeCoordinator.getScroller().sort(Scroller.SORT_NAME_CASE_INSENSATIVE);
		}
	}

	class NameSortSensative extends AbstractAction
	{
		NameSortSensative()
		{
			super(AppStarter.getString("Actions.204")); //$NON-NLS-1$
		}

		public void actionPerformed(ActionEvent event)
		{
			setSortOrder(Scroller.SORT_NAME_CASE_SENSATIVE);
			SortManager.this.myListShowTreeCoordinator.getScroller().sort(Scroller.SORT_NAME_CASE_SENSATIVE);
		}
	}

	class DateSort extends AbstractAction
	{
		DateSort()
		{
			super(AppStarter.getString("Actions.205")); //$NON-NLS-1$
		}

		public void actionPerformed(ActionEvent event)
		{
			setSortOrder(Scroller.SORT_DATE);
			SortManager.this.myListShowTreeCoordinator.getScroller().sort(Scroller.SORT_DATE);
		}
	}

	class LengthSort extends AbstractAction
	{
		LengthSort()
		{
			super(AppStarter.getString("Actions.206")); //$NON-NLS-1$
		}

		public void actionPerformed(ActionEvent event)
		{
			setSortOrder(Scroller.SORT_LENGTH);
			SortManager.this.myListShowTreeCoordinator.getScroller().sort(Scroller.SORT_LENGTH);
		}
	}
	

}