/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package org.gerhardb.lib.image;

import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;

import org.gerhardb.lib.util.startup.AppStarter;
import org.gerhardb.lib.io.FileUtil;

/**
 * This is used to read images and write them back out to a file.
 * This wraps the stream thereby keeping the Metadata.
 */
public abstract class IOImage
{
   File myFile;
   BufferedImage myImage;

   // ==========================================================================
   //                            Constructor
   // ==========================================================================
   public IOImage(File file) throws Exception
   {
      this.myFile = file;
      if (this.myFile == null || !this.myFile.exists())
      {
         throw new IOException(this.myFile + " " + AppStarter.getString("IOImage.0")); //$NON-NLS-1$ //$NON-NLS-2$
      }
    }

   // ==========================================================================
   //                            Abstract Methods
   // ==========================================================================

   public abstract void save(BufferedImage image, File saveAsFile) throws Exception;
   
   public abstract String getImageFactoryName();
      
   // ==========================================================================
   //                            Public Methods
   // ==========================================================================
   public File getFile()
   {
      return this.myFile;
   }
 
   /**
    * Gets a buffered image.
    * Note that the file lock will not go away until the BufferedImage goes
    * away.
    * @throws IOException
    * @return null if no image was found.
    */
	public BufferedImage getImage()
   {
      return this.myImage;
   }
	
   public void saveJPEG(BufferedImage image, float quality)
      throws IOException
   {
   	saveJPEG(image, quality, this.myFile.getAbsolutePath());
   }

   public void saveJPEG(BufferedImage image, float quality, String saveAsName)
      throws IOException
   {
      // Make sure name is correct
      String testName = saveAsName.toLowerCase();
      if (!(testName.endsWith(".jpeg") || //$NON-NLS-1$
         testName.endsWith(".jpg"))) //$NON-NLS-1$
      {
         // Add the ending of the correct type.
         saveAsName = FileUtil.fileNameNoExtension(saveAsName) + ".jpeg"; //$NON-NLS-1$
      }
      File theFile = new File(saveAsName);
      ImageUtil.saveJPEG(image, quality, theFile);
      this.myImage = image;
      this.myFile = theFile;
   }

   public void save(BufferedImage image) throws Exception
	{
		save(image, this.myFile.getAbsolutePath());
	}

	public void save(BufferedImage image, String saveAsName)
			throws Exception
	{
		File theFile = new File(saveAsName);
		save(image, theFile);
		this.myImage = image;
		this.myFile = theFile;
	}
   
   public static void main(String[] args)
   {
      File in = new File("e:/testpics/proof/a.jpg"); //$NON-NLS-1$
      File out = new File("e:/testpics/a.jpeg"); //$NON-NLS-1$
      
      try
      {
         BufferedImage img = ImageFactory.getImageFactory().makeImage(in).getImage();
         ImageUtil.saveJPEG(img, 1.0f, out);
      }
      catch(Exception ex)
      {
      	ex.printStackTrace();
      }
   }
}