/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package org.gerhardb.lib.image;

import ij.ImagePlus;
import ij.io.FileSaver;

import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;

import org.apache.sanselan.Sanselan;

/**
 * This is used to read images and write them back out to a file.
 * This wraps the stream thereby keeping the Metadata.
 */
class IOImageSanselan extends IOImage
{
   // ==========================================================================
   //                            Constructor
   // ==========================================================================
   public IOImageSanselan(File file) throws Exception
   {
      super(file);
      try
      {
         this.myImage = Sanselan.getBufferedImage(this.myFile);
      }
      catch (Exception ex)
      {
      	//ex.printStackTrace();
      	/*
       	System.out.println("Sanselan Format Compliance: " + Sanselan.getFormatCompliance(myFile));
        	System.out.println("Sanselan ICC Profile: " + Sanselan.getICCProfile(myFile));
        	System.out.println("Sanselan Image Info: " + Sanselan.getImageInfo(myFile));
        	System.out.println("Sanselan Image Size: " + Sanselan.getImageSize(myFile));
        	System.out.println("Sanselan Metadata: " + Sanselan.getMetadata(myFile));
        	System.out.println("Sanselan Dump: " + Sanselan.dumpImageFile(myFile));
         */
      	throw ex;
      }
   }

   // ==========================================================================
   //                        Super Class Implementation
   // ==========================================================================
   
   @Override
	public String getImageFactoryName()
   {
   	return "Sanselan"; //$NON-NLS-1$
   }

   @Override
	public void save(BufferedImage image, File saveAsFile)
   throws IOException
	{
   	// Not worth figuring out Sanselan save process...
   	// Now that we have BufferedImage, ImageJ should work...
    	FileSaver.setJpegQuality(100); // 0 is poorest, 100 is highest
    	ImagePlus saveMe = new ImagePlus("Title", image);  // Title doesn't get saved...
      FileSaver saver = new FileSaver(saveMe);
    	saver.saveAsJpeg(saveAsFile.getAbsolutePath());
 	}
   
   // ==========================================================================
   //                            Test Bed
   // ==========================================================================
   public static void main(String[] args)
   {
     	ImageFactorySanselan factory = new ImageFactorySanselan();
      try
      {
      	//File aFile = new File("/news/bad/java6-invisable/java4-invisable/am-dmw081715745-jibs022357.jpg");
      	//File aFile = new File("/news/unworked/in/morris/Mirror03.jpg");
      	File aFile = new File("/news/unworked/in/morris/Mirror03.jpg"); //$NON-NLS-1$
      	factory.canHandle(aFile);
        	System.out.println("Looked Good"); //$NON-NLS-1$
      	IOImageSanselan image = new IOImageSanselan(aFile); 
      	System.out.println("Created OK"); //$NON-NLS-1$
         image.save(image.getImage());
         System.out.println("Image Saved!"); //$NON-NLS-1$
      }
      catch (Exception ex)
      {
         ex.printStackTrace();
      }
   }
}