/**
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package org.gerhardb.lib.image;

import java.awt.Graphics2D;
import java.awt.geom.AffineTransform;
import java.awt.image.AffineTransformOp;
import java.awt.image.BufferedImage;

/**
 */
public class ImageChangeUtil
{
   private ImageChangeUtil()
   {
		// Prevent public creation.
   }

   public static BufferedImage rotateRight(BufferedImage image)
   {
      return rotateLeftOrRight(image, 90);
   }

   public static BufferedImage rotateLeft(BufferedImage image)
   {
      return rotateLeftOrRight(image, -90);
   }

   public static BufferedImage flip(BufferedImage image)
   {
      double theta = Math.toRadians(180);
      //System.out.println("Rotate: " + theta);

      int width = image.getWidth();
      int height = image.getHeight();
      //System.out.println("w: " + width + "  h: " + height);

      double x = (double)width / 2;
      double y = (double)height / 2;

      // Original
      AffineTransform at = new AffineTransform();
      at.setToTranslation(x, y); // S3: final translation
      at.rotate(theta); // S2: rotate around anchor

      // S1: translate anchor to origin
      at.translate(-x, -y);

      BufferedImage rtnMe =
         new BufferedImage(width, height, BufferedImage.TYPE_INT_RGB);
      Graphics2D g2x = rtnMe.createGraphics();
      g2x.drawImage(image, at, null);

      return rtnMe;
   }

   public static BufferedImage scale(
      BufferedImage image,
      float scale)
   //throws Exception
   {
      //System.out.println( "Scale: " + scale);
      try
      {
         AffineTransform at = AffineTransform.getScaleInstance(scale, scale);
         AffineTransformOp op = new AffineTransformOp(at, null);
         return op.filter(image, null);
       }
		//catch (OutOfMemoryError ex){throw new IOException("Out of Memory.  Try lower DPI or adjust Java memory settings used to start JIBS.");	}
      catch (Exception ex)
      {
         System.err.println(ex.getMessage());
         ex.printStackTrace();
         // Fail safe because this randomly blew up once in Linux.
         // Might not be a real problem, but what the heck!
         return image;
      }
   }

   /**
    * Adjust image to fit in the given size maintaining the aspect ratio.
    * Only scales DOWN.
    * @param image BufferedImage
    * @param width int
    * @param height int
    * @return BufferedImage
    */
   public static BufferedImage fitAspectDown(
      BufferedImage image,
      int scaleWidth,
      int scaleHeight)
   {
   	if ( image == null )
   	{
   		return null;
   	}
      int imgWidth = image.getWidth();
      int imgHeight = image.getHeight();
      
      if (scaleWidth > 0 && scaleHeight > 0)
      {
         float theta = ImageChangeUtil.calcThetaDown(
                       imgWidth, imgHeight, scaleWidth, scaleHeight);
         //System.out.println("fitAspect " + theta + "    " + scaleWidth + ":" + scaleHeight + "   " + imgWidth + ":" + imgHeight);
         if (theta > 0)
         {
            image = ImageChangeUtil.scale(image, theta);
         }
      }
      return image;
   }
   
  /**
    * Adjust image to fit in the given size maintaining the aspect ratio.
    * Will scale up or down.
    * @param image BufferedImage
    * @param width int
    * @param height int
    * @param scaleUp boolean If true, scale the image up to fit as well as down to fit.
    * @return BufferedImage
    */
   public static BufferedImage fitAspectFill(
      BufferedImage image,
      int scaleWidth,
      int scaleHeight)
   {
   	if ( image == null )
   	{
   		return null;
   	}
      int imgWidth = image.getWidth();
      int imgHeight = image.getHeight();
      
      if (scaleWidth > 0 && scaleHeight > 0)
      {
         float theta = ImageChangeUtil.calcTheta(
                       imgWidth, imgHeight, scaleWidth, scaleHeight);
         //System.out.println("fitAspect " + theta + "    " + scaleWidth + ":" + scaleHeight + "   " + imgWidth + ":" + imgHeight);
         image = ImageChangeUtil.scale(image, theta);
         //System.out.println("Scaled to: " + image.getWidth() + ":" + image.getHeight());
      }
      return image;
   }

   /**
      Untested
      Image must be loaded before being called.
    */
   /*
   public static BufferedImage bufferImage(Image image)
   {
      int width = image.getWidth(null);
      int height = image.getHeight(null);
      BufferedImage rtnMe =
         new BufferedImage(width, height, BufferedImage.TYPE_INT_RGB);
      Graphics2D g2d = rtnMe.createGraphics();
      g2d.drawImage(image, null, null);
      return rtnMe;
   }
   */

   /**
       * @return ResizedImage Returns null if the image did not need to be resized.
    */
   public static BufferedImage resizeImageDown(
      BufferedImage image,
      double toWidth,
      double toHeight)
      throws Exception
   {
      if (toWidth < 10)
      {
         return null;
      }
      if (toHeight < 10)
      {
         return null;
      }

      float theta = ImageChangeUtil.calcThetaDown(
                    image.getWidth(), image.getHeight(), toWidth, toHeight);

      if (theta > 0)
      {
         //System.out.println( "Resizing Needed!" );
         return scale(image, theta);
      }
      return null;
   }

   /**
    * How to size the picture up or down to fit in the area given.
    * All of the picture will be shown with a best fit.
    */
   public static float calcTheta(
      int fromWidth, int fromHeight,
      double toWidth, double toHeight)
   {
      if ( (int)toWidth == fromWidth && (int)toHeight == fromHeight )
      {
         return 1.0f;
      }

      // If either dimension is smaller, shrink.
      if ( toWidth < fromWidth || toHeight < fromHeight )
      {
      	//System.out.println("calcThetaDown");
         return calcThetaDown(fromWidth, fromHeight, toWidth, toHeight);
      }
      // Otherwise grow.
      else if ( toWidth > fromWidth || toHeight > fromHeight )
      {
      	//System.out.println("calcThetaUP");
         return calcThetaUp(fromWidth, fromHeight, toWidth, toHeight);
      }
      else
      {
         return 1.0f;
      }
   }

   /**
    * How to size the picture to fit a given width.
    */
   public static float calcThetaWidth(
      int fromWidth, double toWidth)
   {
      //System.out.println( "Converting From: w: " + fromWidth + " To: w: " + toWidth );
      float theta = 0;
      if (fromWidth != toWidth)
      {
         theta = (float)toWidth / fromWidth;
      }
      //System.out.println("Theta: " + theta);
      return theta;
   }


   /**
    * How to size the picture down to fit in the area given.
    */
   public static float calcThetaDown(
      int fromWidth, int fromHeight,
      double toWidth, double toHeight)
   {
      //System.out.println( "Converting From: w: " + fromWidth + " h: " + fromHeight );
      //System.out.println( "Converting To: w: " + toWidth + " h: " + toHeight );

      float wideBy = 0;
      if (fromWidth > toWidth)
      {
         wideBy = (float)toWidth / fromWidth;
      }

      float highBy = 0;
      if (fromHeight > toHeight)
      {
         highBy = (float)toHeight / fromHeight;
      }

      // Take the smaller fraction to result in the highest size reduction.
      float theta = 0;

      // If both the same, take one of them.
      if (wideBy == highBy)
      {
         theta = highBy;
      }
      // Otherwise take the smallest number greater than zero.
      else
      {
         if (wideBy > 0 && highBy > 0)
         {
            if (wideBy < highBy)
            {
               theta = wideBy;
            }
            else
            {
               theta = highBy;
            }
         }
         else if (wideBy > 0)
         {
            theta = wideBy;
         }
         else if (highBy > 0)
         {
            theta = highBy;
         }
         //System.out.println("DOWN Wide by: " + wideBy + "  igh By: " + highBy);
         //System.out.println("DOWN Theta: " + theta);
      }
      return theta;
   }

   /**
    * How to size the picture up to fit in the area given.
    */
   public static float calcThetaUp(
      int fromWidth, int fromHeight,
      double toWidth, double toHeight)
   {
      //System.out.println("Converting From: w: " + fromWidth + " h: "
      //   + fromHeight);
      //System.out.println("Converting To: w: " + toWidth + " h: " + toHeight);

      float wideBy = 0;
      if (fromWidth < toWidth)
      {
         wideBy = (float)toWidth / fromWidth;
      }

      float highBy = 0;
      if (fromHeight < toHeight)
      {
         highBy = (float)toHeight / fromHeight;
      }

      // Take the smaller fraction to result in the smallest size increase.
      float theta = 0;

      // If both the same, take one of them.
      if (wideBy == highBy)
      {
         theta = highBy;
      }
      // Otherwise take the smallest number greater than zero
      else
      {
         if (wideBy < highBy)
         {
            theta = wideBy;
         }
         else
         {
            theta = highBy;
         }
      }
      return theta;
   }

   //===========================================================================
   //                           Private Classes
   //===========================================================================

   /**
    *
    * @param image
    * @param degrees 90, -90 only!!
    * @return
    */
   private static BufferedImage rotateLeftOrRight(
      BufferedImage image,
      int degrees)
   {
      switch (degrees)
      {
         case 90:
         case -90:
            break;
         default:
            return image;
      }

      //System.out.println("Degrees: " + degrees);
      double theta = Math.toRadians(degrees);
      //System.out.println("Rotate: " + theta);

      // Make double here for ensuing math.
      int width = image.getWidth();
      int height = image.getHeight();
      //System.out.println("w: " + width + "  h: " + height);

      double x = (double)width / 2;
      double y = (double)height / 2;
      double diff = ((double)(width - height)) / 2;
      //System.out.println("w-h=" + diff);

      // Original
      AffineTransform at = new AffineTransform();
      at.setToTranslation(x, y); // S3: final translation
      at.rotate(theta); // S2: rotate around anchor
      double xDiff = 0, yDiff = 0;
      // S1: translate anchor to origin
      if (degrees == 90)
      {
         xDiff = -x + diff;
         yDiff = -y + diff;
         at.translate(xDiff, yDiff);
      }
      else if (degrees == -90)
      {
         xDiff = -x - diff;
         yDiff = -y - diff;
         at.translate(xDiff, yDiff);
      }

      BufferedImage rtnMe =
         new BufferedImage(height, width, BufferedImage.TYPE_INT_RGB);
      Graphics2D g2x = rtnMe.createGraphics();
      g2x.drawImage(image, at, null);

      return rtnMe;
   }

}
