/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package org.gerhardb.lib.image;

import java.awt.image.BufferedImage;
import java.io.File;
import java.util.*;
import java.util.prefs.Preferences;

import org.gerhardb.lib.util.startup.AppStarter;
import org.gerhardb.lib.io.FilenameFileFilter;

public class ImageFactory
{	
	public static String[] TEXT_ENDINGS = {".txt", ".htm", ".html", ".nat"};
	
	private static Preferences clsPrefs;
	private static ImageFactory clsImageFactory;
	
	public static ImageFactory getImageFactory()
	{
		if (clsImageFactory == null)
		{
			loadPlugins();
		}
		return clsImageFactory;	
	}
	
   public static void loadPlugins()
   {   	
		clsImageFactory = new ImageFactory();
   }

	private FilenameFileFilter myFilter = new FilenameFileFilter();	
	private ImageFactoryPlugin[] myPlugins;
	private String[] myPluginsNot;
   
   private ImageFactory()
   {
   	//System.out.println("ImageFactory Classpath: " + System.getProperty("java.class.path")); //$NON-NLS-1$ //$NON-NLS-2$
    	
   	ArrayList<ImageFactoryPlugin> plugins = new ArrayList<ImageFactoryPlugin>(20);
   	ArrayList<String> pluginsNot = new ArrayList<String>(20);
   	// Add plugins which are compiled into JIBS
   	// Will be processed in this order!!
   	plugins.add(new ImageFactoryImageJ()); // Best on test fixed files with bad ICC conversions.
   	plugins.add(new ImageFactoryJava());
   	plugins.add(new ImageFactorySanselan());  // Way old at this point.
   	//addRuntimePlugins(plugins, pluginsNot);
   	this.myPlugins = plugins.toArray(new ImageFactoryPlugin[plugins.size()]);
   	this.myPluginsNot = pluginsNot.toArray(new String[pluginsNot.size()]);
 		
		clsPrefs = Preferences.userRoot().node("org/gerhardb/lib/image/ImageFactory");
     	updateEndings();   
   }
   
   private void updateEndings()
   {
		HashSet<String> endingsSet = new HashSet<String>();
		for(int i=0; i<this.myPlugins.length; i++)
		{
			String[] someEndings = this.myPlugins[i].getEndingsLowercase();
			for(int j=0; j<someEndings.length; j++)
			{
				endingsSet.add(someEndings[j]);
			}			
		}

		if (isTextFilesShowingPreferenceSet())
		{
			for(int j=0; j<TEXT_ENDINGS.length; j++)
			{
				endingsSet.add(TEXT_ENDINGS[j]);
			}						
		}				
		String[] allEndings = endingsSet.toArray(new String[endingsSet.size()]);		
		this.myFilter.setEndings(allEndings);	   	
   }
   
	/*
   private void addRuntimePlugins(ArrayList plugins, ArrayList pluginsNot)
   {  
   	// Added to distro so this technique is not used at the moment.
   	// Left in if I add more plugins later.
     	if (ImageFactoryImageJ.okToUse(this.getClass().getClassLoader()))
     	{
     		System.out.println("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!");
     		System.out.println("ADDED IMAGEJ PLUGIN");
     		System.out.println("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!");
     		plugins.add(new ImageFactoryImageJ()); 
     	} 	
     	else
     	{
     		pluginsNot.add("ImageJ");
   }
     	*/
	
   public FilenameFileFilter getFilter()
   {
   	return this.myFilter;
   }
   
	public String[] getImagePluginsListing()
	{
		String[] rtnMe = new String[this.myPlugins.length];
		for(int i=0; i<this.myPlugins.length; i++)
		{
			StringBuffer buff = new StringBuffer(200);
			buff.append(this.myPlugins[i].toString());
			buff.append(": "); //$NON-NLS-1$
			String[] endings = this.myPlugins[i].getEndingsLowercase();
			for(int j=0; j<endings.length; j++)
			{
				if (j>0)
				{
					buff.append(", ");					 //$NON-NLS-1$
				}
				buff.append(endings[j]);
			}
			rtnMe[i] = buff.toString();
		}
		return rtnMe;
	}
	
	public String[] getImagePluginsNotLoadedListing()
	{
		return this.myPluginsNot;
	}
	
	 /**
    * @return Returns the image if good, else returns null.
    */
   public IOImage makeImage(File aFile)
   throws Exception
   {
   	if (aFile == null)
   	{
   		throw new Exception("Null files not allowed"); //$NON-NLS-1$
   	}
   	
   	if (!aFile.exists())
   	{
   		throw new Exception(AppStarter.getString("ImageFactory.0") + AppStarter.getString("colon") + " " + aFile); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
   	}
		//System.out.println("File: " + aFile.getName());					
		for(int i=0; i<this.myPlugins.length; i++)
		{
			//System.out.println("ImageFactory Plugin: " + i + " " + myPlugins[i].toString());					
			if (this.myPlugins[i].canHandle(aFile))
			{	
				try
				{
					IOImage rtnMe = this.myPlugins[i].makeImage(aFile);
					//System.out.println(rtnMe.getImageFactoryName() + " handled");				
					if ( rtnMe != null )
					{
						return rtnMe;
					}
				}
				catch(Exception ex)
				{
					System.out.println(this.myPlugins[i] + " could NOT handle: " + ex.getMessage()); //$NON-NLS-1$
				}
			}
		}
		throw new Exception("No plugin handled image"); //$NON-NLS-1$
   }

	 /**
    * @return Returns the image if good, else returns null.
    */
   public IOImage makeImageEZ(File aFile)
   {
      try
      {
         return makeImage(aFile);
      }
      catch (Exception x)
      {
      	System.out.println(x.getMessage());
         return null;
      }
   }

   public BufferedImage getImageEZ(File file)
   {
      try
      {
         IOImage io = makeImage(file);
         return io.getImage();
      }
      catch (Exception ex)
      {
         //System.out.println("getImageEZ  (" + file.getName() + "): " + ex.getMessage());
         //ex.printStackTrace();
         return null;
      }
   }	
	// ===========================================================================
	// Show Text Files - treat sorta like images...
	// ===========================================================================
	private static final String SHOW_TEXT_FILES = "showTextFiles"; //$NON-NLS-1$

	public boolean isTextFilesShowingPreferenceSet()
	{
		return clsPrefs.getBoolean(SHOW_TEXT_FILES, false);
	}

	public void setTextFilesShowingPreference(boolean enabled)
	{
		clsPrefs.putBoolean(SHOW_TEXT_FILES, enabled);
     	updateEndings();   
	}

   public static void main (String[] args)
   {
   	org.gerhardb.lib.util.startup.AppStarter.startUpApp(args, "org.gerhardb.jibs.Jibs", true);
  	   System.out.println(ImageFactory.getImageFactory().getFilter().toString());

		String[] formats = ImageFactory.getImageFactory().getImagePluginsListing();
		for (int i = 0; i < formats.length; i++)
		{
			System.out.println(formats[i]);
		}
		System.out.println("------------------------------------------------------------------------");					 //$NON-NLS-1$
   	
      try
      {
      	ImageFactory.getImageFactory().makeImage(new File("D:/testpics/bad/a.jpg")); //$NON-NLS-1$
      }
      catch (Exception ex)
      {
         ex.printStackTrace();
      }
   }
}