/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package org.gerhardb.lib.image;

import java.awt.image.BufferedImage;
import java.awt.image.ColorModel;
import java.awt.image.PixelGrabber;
import java.awt.*;
import java.io.File;
//import java.io.FileOutputStream;
import java.io.IOException;
import javax.swing.ImageIcon;
//import com.sun.image.codec.jpeg.JPEGCodec;
//import com.sun.image.codec.jpeg.JPEGEncodeParam;
//import com.sun.image.codec.jpeg.JPEGImageEncoder;

import javax.imageio.*;
import javax.imageio.stream.*;
import java.util.Iterator;

/**
 * This is used to read images and write them back out to a file.
 * This wraps the stream thereby keeping the Metadata.
 */
public class ImageUtil
{
   /**
	* This method returns a buffered image with the contents of an image.
	* From The Java Developers Almanac 1.4
	* e667. Creating a Buffered Image from an Image
	* An Image object cannot be converted to a BufferedImage object. 
	* The closest equivalent is to create a buffered image and then draw the image on the buffered image. This example defines a method that does this.
	*/
	public static BufferedImage toBufferedImage(Image image)
	{
		if (image instanceof BufferedImage) { return (BufferedImage) image; }

		// This code ensures that all the pixels in the image are loaded
		image = new ImageIcon(image).getImage();

		// Determine if the image has transparent pixels; for this method's
		// implementation, see e661 Determining If an Image Has Transparent Pixels
		boolean hasAlpha = hasAlpha(image);

		// Create a buffered image with a format that's compatible with the screen
		BufferedImage bimage = null;
		GraphicsEnvironment ge = GraphicsEnvironment
				.getLocalGraphicsEnvironment();
		try
		{
			// Determine the type of transparency of the new buffered image
			int transparency = Transparency.OPAQUE;
			if (hasAlpha)
			{
				transparency = Transparency.BITMASK;
			}

			// Create the buffered image
			GraphicsDevice gs = ge.getDefaultScreenDevice();
			GraphicsConfiguration gc = gs.getDefaultConfiguration();
			bimage = gc.createCompatibleImage(image.getWidth(null), image
					.getHeight(null), transparency);
		}
		catch (HeadlessException e)
		{
			// The system does not have a screen
		}

		if (bimage == null)
		{
			// Create a buffered image using the default color model
			int type = BufferedImage.TYPE_INT_RGB;
			if (hasAlpha)
			{
				type = BufferedImage.TYPE_INT_ARGB;
			}
			bimage = new BufferedImage(image.getWidth(null),
					image.getHeight(null), type);
		}

		// Copy image to buffered image
		Graphics g = bimage.createGraphics();

		// Paint the image onto the buffered image
		g.drawImage(image, 0, 0, null);
		g.dispose();

		return bimage;
	}


	/**
	 * This method returns true if the specified image has transparent pixels
	 * e661. Determining If an Image Has Transparent Pixels
	 */
	public static boolean hasAlpha(Image image)
	{
		// If buffered image, the color model is readily available
		if (image instanceof BufferedImage)
		{
			BufferedImage bimage = (BufferedImage) image;
			return bimage.getColorModel().hasAlpha();
		}

		// Use a pixel grabber to retrieve the image's color model;
		// grabbing a single pixel is usually sufficient
		PixelGrabber pg = new PixelGrabber(image, 0, 0, 1, 1, false);
		try
		{
			pg.grabPixels();
		}
		catch (InterruptedException ex)
      {
      	ex.printStackTrace();
      }

		// Get the image's color model
		ColorModel cm = pg.getColorModel();
		return cm.hasAlpha();
	}	

	/*
	 * // REMOVE = NO LONGER SUPPORTED BY SUN
   public static void saveJPEGold(BufferedImage image, float quality, File saveAsFile)
   throws IOException
	{
	   if (image == null)
	   {
	      throw new IllegalArgumentException("Must provide image and file"); //$NON-NLS-1$
	   }
	
	   // Do the actual save.
	   FileOutputStream out = new FileOutputStream(saveAsFile);
	   JPEGImageEncoder encoder = JPEGCodec.createJPEGEncoder(out);
	   JPEGEncodeParam param = encoder.getDefaultJPEGEncodeParam(image);
	   param.setQuality(quality, false); //1.0f
	   encoder.setJPEGEncodeParam(param);
	   encoder.encode(image);
	   out.close();
	}		
   */

	  /**
    * 
    * @param image
    * @param quality An integer between 0 and 1. 1 specifies minimum compression and maximum quality.
    * @param saveAsFile
    * @throws IOException
    */
   public static void saveJPEG(BufferedImage inputImage, float quality, File saveAsFile)
   throws IOException
	{
   	save(inputImage, quality, saveAsFile, "jpeg");
	}
	
   /**
    * Deafults to jpeg if an unknown type is passed in.
    * @param image
    * @param quality An integer between 0 and 1. 1 specifies minimum compression and maximum quality.
    * @param saveAsFile
    * @param type For example "jpeg".
    * @throws IOException
    */
   public static void save(BufferedImage inputImage, float quality, File saveAsFile, String type)
   throws IOException
	{
	   if (inputImage == null || saveAsFile == null)
	   {
	      throw new IllegalArgumentException("Must provide image and file"); //$NON-NLS-1$
	   }
	
	   // Default to jpeg if we don't find anything appropriate.
	   Iterator<?> iter = ImageIO.getImageWritersByFormatName(type);
	   if (!iter.hasNext())
	   {
	   	iter = ImageIO.getImageWritersByFormatName("jpeg");
	   }
	   
	   // Choose the first image writer available
	   ImageWriter writer = (ImageWriter)iter.next();
	   // instantiate an ImageWriteParam object with default compression options
	   ImageWriteParam iwp = writer.getDefaultWriteParam();

	   // Now, we can set the compression quality:
	   iwp.setCompressionMode(ImageWriteParam.MODE_EXPLICIT);
	   iwp.setCompressionQuality(quality);   // an integer between 0 and 1
	   // 1 specifies minimum compression and maximum quality

	   //  Output the file:
	   FileImageOutputStream output = new FileImageOutputStream(saveAsFile);
	   writer.setOutput(output);
	   IIOImage newImage = new IIOImage(inputImage, null, null);
	   writer.write(null, newImage, iwp);
	   writer.dispose();
	   output.close();
	}   
}