/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package org.gerhardb.lib.image;

import java.io.File;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;

import org.gerhardb.lib.util.startup.AppStarter;

import com.drew.imaging.jpeg.JpegMetadataReader;
import com.drew.imaging.jpeg.JpegProcessingException;
import com.drew.imaging.jpeg.JpegSegmentReader;
import com.drew.metadata.Directory;
import com.drew.metadata.Metadata;
import com.drew.metadata.Tag;
import com.drew.metadata.exif.ExifDirectory;
import com.drew.metadata.exif.ExifReader;
import com.drew.metadata.iptc.IptcReader;

/**
 * This is used to read images and write them back out to a file.
 * This wraps the stream thereby keeping the Metadata.
 */
public class ImageUtilDrew
{
   public static int TAG_DATETIME = ExifDirectory.TAG_DATETIME; 
   public static int TAG_DATETIME_ORIGINAL = ExifDirectory.TAG_DATETIME_ORIGINAL; 
   public static int TAG_DATETIME_DIGITIZED = ExifDirectory.TAG_DATETIME_DIGITIZED; 
   public static int TAG_MAKE = ExifDirectory.TAG_MAKE; 
   public static int TAG_MODEL = ExifDirectory.TAG_MODEL; 

	//2004:01:02 12:43:23
   private static final SimpleDateFormat EXIF_DATE_FORMAT =
      new SimpleDateFormat("yyyy:MM:dd HH:mm:ss"); //$NON-NLS-1$
   
   private File myFile;
   Metadata myMetadata;
   
	public static ArrayList<String> getExifTags(File file)
	{
		ImageUtilDrew drew = new ImageUtilDrew(file);
		return drew.getExifTags();
	}
   
   public ImageUtilDrew(File file)
   {
   	this.myFile = file;
   	if (this.myFile == null)
   	{
   		throw new IllegalArgumentException("File must not be null");
   	}
   }
   
	public Date getPicDate()
	{	 
		if (!parsableFileType())
		{
			return null;
		}

      try
      {
         Metadata metadataEZ = JpegMetadataReader.readMetadata(this.myFile);
         Directory directory = metadataEZ.getDirectory(ExifDirectory.class);
         String timestamp = directory.getString(
                            ExifDirectory.TAG_DATETIME_ORIGINAL);
         if (timestamp == null)
         {
            System.out.println(
               AppStarter.getString("WorkerCamera.34") + " " //$NON-NLS-1$ //$NON-NLS-2$
               + this.myFile);
            return null;
         }
			Date date = EXIF_DATE_FORMAT.parse(timestamp);
			return date;
      }
      catch (JpegProcessingException ex)
      {
         //System.err.println(AppStarter.getString("WorkerCamera.43") + AppStarter.getString("colon") + " " + ex.getMessage()); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
      }
      catch (Exception ex)
      {
         ex.printStackTrace();
      }		
      return null;
	}
	
	public boolean parsableFileType()
	{
		String fileName = this.myFile.getName().toLowerCase();
		if (fileName.endsWith("jpeg") || fileName.endsWith("jpg"))
		{
			return true;
		}
		return false;
	}
	
	private void parse()
	throws JpegProcessingException
	{
		if (this.myMetadata == null)
		{
			JpegSegmentReader segmentReader = new JpegSegmentReader(this.myFile);
			byte[] exifSegment = segmentReader
					.readSegment(JpegSegmentReader.SEGMENT_APP1);
			byte[] iptcSegment = segmentReader
					.readSegment(JpegSegmentReader.SEGMENT_APPD);
			this.myMetadata = new Metadata();
			new ExifReader(exifSegment).extract(this.myMetadata);
			new IptcReader(iptcSegment).extract(this.myMetadata);			
		}
	}
	
	public String getExifString(int tag)
	{
		if (!parsableFileType())
		{
			return null;
		}
		
		try
		{
			parse();
			Directory exifDirectory =
				this.myMetadata.getDirectory(ExifDirectory.class);
			return exifDirectory.getString(tag);
		}
		catch(Exception ex)
		{
			return null;
		}

	}
	
	public ArrayList<String> getExifTags()
	{
		ArrayList<String> rtnMe = new ArrayList<String>();
		if (!parsableFileType())
		{
			rtnMe.add(AppStarter.getString("ImageUtil.3")); //$NON-NLS-1$
			return rtnMe;
		}

		try
		{
			parse();

			// iterate through metadata directories
			Iterator<?> directories = this.myMetadata.getDirectoryIterator();
			while (directories.hasNext())
			{
				Directory directory = (Directory) directories.next();
				// iterate through tags and print to System.out
				Iterator<?> tags = directory.getTagIterator();
				while (tags.hasNext())
				{
					Tag tag = (Tag) tags.next();
					rtnMe.add(tag.toString());
				}
			}
		}
		catch (Exception ex)
		{
			rtnMe.add(AppStarter.getString("ImageUtil.2")); //$NON-NLS-1$
		}
		return rtnMe;
	}	
}