/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package org.gerhardb.lib.io;

import java.awt.Color;
import java.awt.Component;
import java.awt.Point;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.awt.dnd.DnDConstants;
import java.awt.dnd.DragGestureEvent;
import java.awt.dnd.DragGestureListener;
import java.awt.dnd.DragSource;
import java.awt.dnd.DragSourceDragEvent;
import java.awt.dnd.DragSourceDropEvent;
import java.awt.dnd.DragSourceEvent;
import java.awt.dnd.DragSourceListener;
import java.awt.dnd.DropTarget;
import java.awt.dnd.DropTargetDragEvent;
import java.awt.dnd.DropTargetDropEvent;
import java.awt.dnd.DropTargetEvent;
import java.awt.dnd.DropTargetListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import javax.swing.DefaultListModel;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.ListCellRenderer;

/**
 * A drag/drop enabled so that you can reorder the directories.
 */
public class DragDropList
   extends javax.swing.JList implements
   Transferable,
   DropTargetListener,
   DragGestureListener,
   DragSourceListener
{
   static final DataFlavor[] myDataFlavors =
      {
      DataFlavor.javaFileListFlavor};
   DefaultListModel myListModel;

   java.awt.dnd.DragSource myDragSource = new java.awt.dnd.DragSource();
   ArrayList<Object> myTransferData = new ArrayList<Object>();

   int myDragOverIndex = -1;
   int[] myDragSelections;
   boolean okToDropFiles;

   /*
    =============================================================================
                              Constructor
    =============================================================================
    */

   /**
    * Drag drop enable list for FILES or DIRECTORIES.
    * @param model DefaultListModel
    * @param filesOK boolean If true files are accepted.
    * Otherwise only directories are accepted.
    */
   public DragDropList(DefaultListModel model, boolean filesOK)
   {
      super(model);
      super.setCellRenderer(new MyCellRenderer());
      this.myListModel = model;
      this.myDragSource.createDefaultDragGestureRecognizer(
         this, DnDConstants.ACTION_MOVE, this);
      this.okToDropFiles = filesOK;
		new DropTarget(this, this);
   }

   /*
    =============================================================================
                             Public Functions
    =============================================================================
    */

   public void removeSelected()
   {
      Object[] picked = this.getSelectedValues();
      for (int i = 0; i < picked.length; i++)
      {
         this.myListModel.removeElement(picked[i]);
      }
   }

   /*
    ==============================================================================
                           DropTargetListener Implementation
    ==============================================================================
    */
   public void dragEnter(DropTargetDragEvent event)
   {
      //System.out.println("DragDropList.dragEnter");
      // page 748 core swing advanced programming
      if (0 == (event.getSourceActions() & DnDConstants.ACTION_COPY_OR_MOVE))
      {
         //System.out.println("DragDropList.dragEnter: Rejecting Drag");
         event.rejectDrag();
      }
      else if (0 == (event.getDropAction() & DnDConstants.ACTION_COPY_OR_MOVE))
      {
         //System.out.println("DragDropList.dragEnter: Accepting Drag");
         event.acceptDrag(DnDConstants.ACTION_COPY_OR_MOVE);
      }
      this.myDragSelections = this.getSelectedIndices();
   }

   public void dragExit(DropTargetEvent dte)
   {
      //System.out.println("DragDropList.dragEnter");
      this.myDragOverIndex = -1;
      this.myDragSelections = null;
   }

   public void dragOver(DropTargetDragEvent dtde)
   {
      //System.out.println( "DragDropList.dragOver" );
      dtde.getLocation();
      Point dropPoint = dtde.getLocation();
      int index = this.locationToIndex(dropPoint);
      this.myDragOverIndex = index;

      int[] indicesPlus = new int[this.myDragSelections.length + 1];
      for (int i = 0; i < this.myDragSelections.length; i++)
      {
         indicesPlus[i] = this.myDragSelections[i];
      }
      indicesPlus[this.myDragSelections.length] = index;
      this.setSelectedIndices(indicesPlus);
   }

   public void drop(DropTargetDropEvent event)
   {
      //System.out.println("DragDropList.drop");

      // Decide on the drop first.
      if (!event.isDataFlavorSupported(myDataFlavors[0]))
      {
         //System.out.println("DragDropList Rejecting Drop, Bad Flavor.");
         event.rejectDrop();
         return;
      }

      switch (event.getDropAction())
      {
         case DnDConstants.ACTION_MOVE:

            //System.out.println("DragDropList Accepting Drop AS MOVE.");
            event.acceptDrop(DnDConstants.ACTION_MOVE);
            break;
         case DnDConstants.ACTION_COPY:

            //System.out.println("DragDropList Accepting Drop AS COPY.");
            event.acceptDrop(DnDConstants.ACTION_COPY);
            break;
         default:

            //System.out.println("Unsupported User Action");
            break;
      } // switch

      // Reset all the drag marker stuff
      this.myDragOverIndex = -1;
      // Do NOT null out: myDragSelections  -- its still needed by drop!

      // Now we can get transferable.
      event.dropComplete(doDrop(event));
   }

   public void dropActionChanged(DropTargetDragEvent dtde)
   {
      //System.out.println( "DragDropList.dropActionChanged" );
   }

   /*
    =============================================================================
                               DragGestureListener
    =============================================================================
    */

   public void dragGestureRecognized(DragGestureEvent dge)
   {
      //System.out.println("dragGestureRecognized");

      this.myDragSource.startDrag(
         dge,
         DragSource.DefaultMoveDrop,
         this,
         this);
   } // dragGestureRecognized

   /*
    =============================================================================
                               DragSourceListener
    =============================================================================
    */

   public void dragDropEnd(DragSourceDropEvent event)
   {
      //System.out.println("DragDropList.dragDropEnd");
      if (event.getDropSuccess())
      {
         //System.out.println("Drop Succeeded");
      }
      else
      {
         //System.out.println("Drop Failed");
      }
   } 

   public void dragEnter(DragSourceDragEvent dsde)
   {
      //System.out.println( "ScrolledSlider.dragEnter" );
   }

   public void dragExit(DragSourceEvent dse)
   {
      //System.out.println( "ScrolledSlider.dragExit" );
   }

   public void dragOver(DragSourceDragEvent dsde)
   {
      //System.out.println( "ScrolledSlider.dragOver" );
   }

   public void dropActionChanged(DragSourceDragEvent dsde)
   {
      //System.out.println( "ScrolledSlider.dropActionChanged" );
   }

   /*
    ===============================================================================
                             Transferable
    ===============================================================================
    */
   public Object getTransferData(DataFlavor flavor)
   {
      //System.out.println("getTransferData");
      this.myTransferData = new ArrayList<Object>();
      /*
     Object[] picked = this.getSelectedValues();
              for (int i = 0; i < picked.length; i++)
             {
         myTransferData.add(picked[i]);
             }
       */

      // less portable, but works within...
      // Problem being that the drop target is selected at this point.
      for (int i = 0; i < this.myDragSelections.length; i++)
      {
         this.myTransferData.add(this.myListModel.elementAt(this.myDragSelections[i]));
      }
      return this.myTransferData;
   }

   public DataFlavor[] getTransferDataFlavors()
   {
      //System.out.println("getTransferDataFlavors");
      return myDataFlavors;
   }

   public boolean isDataFlavorSupported(DataFlavor flavor)
   {
      //System.out.println("isDataFlavorSupported");
      for (int i = 0; i < myDataFlavors.length; i++)
      {
         if (myDataFlavors[i].equals(flavor))
         {
            return true;
         }
      }
      return false;
   }

   /*
    ===============================================================================
                             Private Methods
    ===============================================================================
    */

   private boolean doDrop(DropTargetDropEvent event)
   {
      //System.out.println("************** DragDropList.doDrop ********************");
      Transferable transfer = event.getTransferable();
      try
      {
         // First, find out directory to move file to.
         Point dropPoint = event.getLocation();
         int index = locationToIndex(dropPoint);

         // What's being moved?
         Object moveObj = transfer.getTransferData(myDataFlavors[0]);

         //this.setSelectedIndices(myDragSelections);

         // If we are moving inside this list, just use the selections.
         if (DnDConstants.ACTION_MOVE == event.getDropAction()
            && this.myTransferData.equals(moveObj))
         {
            for (int i = this.myDragSelections.length - 1; i >= 0; i--)
            {
               if (this.myDragSelections[i] < index)
               {
                  index--;
               }
               //System.out.println("removing " + i + ": " + myDragSelections[i]
               //   + "  -- " + myListModel.getElementAt(myDragSelections[i]));
               this.myListModel.remove(this.myDragSelections[i]);
            }
         }

         if (moveObj instanceof List<?>)
         {
            List<?> fileList = (List<?>)moveObj;
            int[] newIndexes = new int[fileList.size()];
            for (int i = 0; i < newIndexes.length; i++)
            {
               Object obj = fileList.get(i);
               if (obj instanceof File)
               {
                  File saveMe = (File)obj;
                  boolean doDrop = true;
                  if (!this.okToDropFiles && !saveMe.isDirectory())
                  {
                     doDrop = false;
                  }
                  if (doDrop)
                  {
                     this.myListModel.add(index, saveMe);
                     newIndexes[i] = index;
                     index++;
                  }
               }
            }
            this.setSelectedIndices(newIndexes);
            return true;
         }
      }
      catch (UnsupportedFlavorException ex)
      {
         ex.printStackTrace();
         System.err.println("DragDropList.drop: " + ex.getMessage()); //$NON-NLS-1$
         event.rejectDrop();
      }
      catch (IOException ex)
      {
         ex.printStackTrace();
         System.err.println("DragDropList.drop: " + ex.getMessage()); //$NON-NLS-1$
         event.rejectDrop();
      }
      return false;
   }

   /*
    ===============================================================================
                              Inner Classes
    ===============================================================================
    */


   class MyCellRenderer
      extends JLabel implements ListCellRenderer
   {
      public MyCellRenderer()
      {
         setOpaque(true);
      }

      // This is the only method defined by ListCellRenderer.
      // We just reconfigure the JLabel each time we're called.

      public Component getListCellRendererComponent(
         JList list,
         Object value, // value to display
         int index, // cell index
         boolean isSelected, // is the cell selected
         boolean cellHasFocus) // the list and the cell have the focus
      {
         String s = value.toString();
         setText(s);
         setEnabled(list.isEnabled());
         setFont(list.getFont());

         if (DragDropList.this.myDragOverIndex == index)
         {
            setBackground(Color.YELLOW);
            setForeground(Color.BLACK);
         }
         else if (isSelected)
         {
            setBackground(list.getSelectionBackground());
            setForeground(list.getSelectionForeground());
         }
         else
         {
            setBackground(list.getBackground());
            setForeground(list.getForeground());
         }

         //if (index == myBreak){setForeground(Color.RED);}

         return this;
      }
   }

   /*
    =============================================================================
                             Main
    =============================================================================
    */
   public static void main(String[] args)
   {
      try
      {
         DirListModel model = new DirListModel(null);
         DragDropList dirList = new DragDropList(model, true);
         dirList.setMinimumSize(new java.awt.Dimension(200, 100));
         model.recurseAddDirectory(new File("D:/testpics/Free Play")); //$NON-NLS-1$

         JFrame f = new JFrame("File Tree Test"); //$NON-NLS-1$
         f.getContentPane().add(dirList);
         f.setSize(300, 300);
         f.addWindowListener(
            new WindowAdapter()
         {
            @Override
				public void windowClosing(WindowEvent evt)
            {
               System.exit(0);
            }
         });
         f.setVisible(true);
      }
      catch (Exception ex)
      {
         ex.printStackTrace();
      }
      //System.out.println("Main Exiting Normally");
   } 

}
