/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
package org.gerhardb.lib.io;

import java.awt.BorderLayout;
import java.awt.Cursor;
import java.awt.FlowLayout;
import java.awt.event.*;
import java.io.File;

import javax.swing.*;

import org.gerhardb.lib.util.startup.AppStarter;
import org.gerhardb.lib.dirtree.filelist.popup.FileName;
import org.gerhardb.lib.scroller.Scroller;
import org.gerhardb.lib.swing.JPanelRows;
import org.gerhardb.lib.swing.SwingUtils;
import org.gerhardb.lib.util.Icons;

import com.saic.isd.swing.filechooser.SimpleFileFilter;

/**
 *
 * @author  Gerhard Beck
 */
public class ZipDialog
   extends javax.swing.JDialog
{
	File[] myFileList;
	FileName myFileName;
	private Scroller myScroller;
  
    //===========================================================================
   //                              Constructor
   //===========================================================================
   public ZipDialog(JFrame frame, File[] fileList, Scroller scroller)
   {
      super(frame, AppStarter.getString("ZipDialog.0"), true); //$NON-NLS-1$
		this.setIconImage(Icons.getIcon(Icons.JIBS_16).getImage());
      this.myFileList = fileList;
   	this.myFileName = new FileName(this.myFileList.length);
		this.myScroller = scroller;
   	layoutScreen();
   }
   
   void ok()
   {
		JFileChooser chooser = new JFileChooser();
		chooser.setCurrentDirectory(this.myFileList[0]);
		chooser.setDialogTitle(AppStarter.getString("ZipDialog.1")); //$NON-NLS-1$
		chooser.setFileSelectionMode(JFileChooser.FILES_AND_DIRECTORIES);
		SimpleFileFilter filter = new SimpleFileFilter("zip", //$NON-NLS-1$
	          AppStarter.getString("ZipDialog.3") ); //$NON-NLS-1$
	   chooser.setFileFilter(filter);
		int returnVal = chooser.showSaveDialog(this.myScroller.getFocusComponent());
		if (returnVal == JFileChooser.APPROVE_OPTION)
		{
			File saveZipFile = chooser.getSelectedFile();
			if (!saveZipFile.getName().toLowerCase().endsWith(".zip")) //$NON-NLS-1$
			{
				saveZipFile = new File(saveZipFile.getAbsolutePath() + ".zip"); //$NON-NLS-1$
			}
			if (saveZipFile.exists())
			{
	         int cancel = JOptionPane.showConfirmDialog(
                  this,
                  AppStarter.getString("ZipDialog.6"),  //$NON-NLS-1$
                  AppStarter.getString("ZipDialog.7"),  //$NON-NLS-1$
                  JOptionPane.YES_NO_OPTION);
	         if (cancel == JOptionPane.NO_OPTION)
	         {
	         	return;
	         }				
			}
	   	setWaitCursor(true);
			try{ Zip.makeAZip(this.myFileList, saveZipFile, this.myFileName);}
			catch(Exception ex)
			{
		   	setWaitCursor(false);
				ex.printStackTrace();
	         JOptionPane.showMessageDialog(
	         		ZipDialog.this,
	               ex.getMessage(),
	               AppStarter.getString("problem"), //$NON-NLS-1$
	               JOptionPane.ERROR_MESSAGE );				
			}
	   	setWaitCursor(false);
	      setVisible(false);
	      dispose();
		}
   }

   private void layoutScreen()
   {
  	 	addWindowListener(
          new WindowAdapter()
       {
          @Override
			public void windowClosing(WindowEvent e)
          {
             cancel();
          }
       }
       );
       
       
       JButton okBtn = new JButton(AppStarter.getString("ok")); //$NON-NLS-1$
       okBtn.addActionListener
       (
          new ActionListener()
       {
          public void actionPerformed(ActionEvent ae)
          {
             ok();
          }
       }
       );

       JButton cancelBtn = new JButton(AppStarter.getString("cancel")); //$NON-NLS-1$
       cancelBtn.addActionListener
       (
          new ActionListener()
       {
          public void actionPerformed(ActionEvent ae)
          {
             cancel();
          }
       }
       );
       
       JPanel btnPanel = new JPanel(new FlowLayout(FlowLayout.CENTER));
       btnPanel.add(okBtn);
       btnPanel.add(cancelBtn);

       
       JPanel thePanel = new JPanel(new BorderLayout());
       thePanel.add(getNamePanel(), BorderLayout.CENTER);
       thePanel.add(btnPanel, BorderLayout.SOUTH);
       
       addAccelerators(okBtn, thePanel);
       this.getContentPane().add(thePanel);
       //this.setSize(new Dimension(600, 430));
       pack();
       SwingUtils.centerOnScreen(this);

       this.setVisible(true);
     }
   
   private JPanel getNamePanel()
   {
   	final JTextField textFld = new JTextField(33);
      final JLabel sampleLbl = new JLabel("<html><big>&nbsp;</big></html>"); //$NON-NLS-1$
      
      textFld.setText("jibs-#"); //$NON-NLS-1$
   	this.myFileName.setTemplate(textFld.getText());
   	sampleLbl.setText(this.myFileName.getNewNameHTML(1, textFld.getText()));
      
  	   JPanelRows rtnMe = new JPanelRows(FlowLayout.CENTER);
      textFld.addKeyListener(new KeyAdapter()
      {
         @Override
			public  void 	keyReleased(KeyEvent e)
         {
         	ZipDialog.this.myFileName.setTemplate(textFld.getText());
         	sampleLbl.setText(ZipDialog.this.myFileName.getNewNameHTML(1, textFld.getText()));
         }
      });

      JPanel row = rtnMe.topRow();
      row.add(new JLabel("Change name to" + AppStarter.getString("colon") + " ")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
      row.add(textFld);

      row = rtnMe.nextRow();
      row.add(new JLabel("<html>&nbsp;</html>")); //$NON-NLS-1$
      row.add(sampleLbl);
      
      row = rtnMe.nextRow();
      row.add(new JLabel(AppStarter.getString("ZipDialog.11"))); //$NON-NLS-1$
      
      row = rtnMe.nextRow();
      row.add(new JLabel(AppStarter.getString("ZipDialog.12")));     //$NON-NLS-1$
      
      return rtnMe;
   }
   
   void addAccelerators(JButton okBtn, JPanel thePanel)
   {
   	InputMap map = null;
   	
      // Accelerator - ok
		// Turned off because this does not consume the Enter key which then
		// goes on to bring up the move dialog.  From line 429 in Scroller.
		this.getRootPane().setDefaultButton(okBtn);
		
		// 429 solution
		// This works - ALT K is how to activate.
		okBtn.setMnemonic(KeyEvent.VK_K);
   	
	   // Accelerator - cancel
      class Dismiss
         extends AbstractAction
      {
         Dismiss()
         {
            super("dismiss"); //$NON-NLS-1$
         }

         public void actionPerformed(ActionEvent e)
         {
            cancel();
         }
      }
      Action dismiss = new Dismiss();
      String cancelID = "dismiss"; //$NON-NLS-1$
      thePanel.getActionMap().put(cancelID, dismiss);
      map = thePanel.getInputMap(JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT);
      map.put(KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0), cancelID);
   }
  
   void cancel()
   {
      setVisible(false);
      dispose();
   }
   
	public void setWaitCursor(boolean wait)
	{
		if (wait)
		{
			setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
		}
		else
		{
			setCursor(Cursor.getDefaultCursor());
		}
	}
	

}
