/*
 * Copyright (c) 2011 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package org.gerhardb.lib.print;

import java.awt.*;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import javax.swing.*;

import org.gerhardb.lib.util.startup.AppStarter;
import org.gerhardb.lib.swing.JPanelRows;
import org.gerhardb.lib.swing.SwingUtils;
import org.gerhardb.lib.util.Icons;

/**
 * Preview Pageables.
 */
public class PageablePreviewer
{   
   Pageable myPageable;
   PreviewPanel myPreviewPanel;
   
   // First page will be 1 not zero.
   // This matches convention in Pageable.
   int myCurrentPage = 0;
   int myLastPage = 0;
   JButton myFirst = new JButton(AppStarter.getString("PageablePreviewer.0")); //$NON-NLS-1$
   JButton myPrior = new JButton(AppStarter.getString("PageablePreviewer.1")); //$NON-NLS-1$
   JButton myNext = new JButton(AppStarter.getString("PageablePreviewer.2")); //$NON-NLS-1$
   JButton myLast = new JButton(AppStarter.getString("PageablePreviewer.3")); //$NON-NLS-1$
   JButton[] myActionBtns;
   
   // Progress printing pages in a directory
   JProgressBar myPageBar = new JProgressBar();
   
   // Progress pinting pictures on a page
   JProgressBar myPicBar = new JProgressBar();

   JFrame myFrame = new JFrame(AppStarter.getString("print.preview")); //$NON-NLS-1$;
	
   public PageablePreviewer(
   		Pageable pageable, 
   		boolean exitOnClose,
   		JButton pdfBtn)
   {
   	this.myPageable = pageable;
   	this.myLastPage = this.myPageable.getNumberOfPages();
   	this.myPreviewPanel = new PreviewPanel(this.myPageable); //, this);
   	
   	this.myPageable.setTopFrame(PageablePreviewer.this.myFrame);
   	
   	if (exitOnClose)
   	{
	   	this.myFrame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
	   	this.myFrame.addWindowListener(
				new WindowAdapter()
				{
					@Override
					public void windowClosing(WindowEvent e)
					{
						System.exit(0);
					}
				});		
   	}		
   	
   	this.myFrame.addWindowListener(new WindowAdapter() {
   		@Override
   		public void windowClosing(WindowEvent e) {close();}
   		});
   	
   	this.myActionBtns = getDefaultButtons(pdfBtn);
   	
    	setupDialog();
             
      this.myFrame.setIconImage(Icons.getIcon(Icons.JIBS_16).getImage());
      this.myFrame.setVisible(true);   
      
      // Will set next, prior, last buttons.
      // Assumes there is always at least one page.
      // Requires canvas so must be after displaying.
      showPreview(1); 
   }
   
   void close()
   {
      //this.myPageable.removeProgressBar(this.myPicBar);
      PageablePreviewer.this.myFrame.dispose();
   }
   
    // Assume you can always show Page 0.
   void showPreview(int page)
   {
      // Check for valid value and adjust to fail-safe values.
      if (page < 1)
      {
         // Show first page
         this.myCurrentPage = 1;
      }
      else if (this.myLastPage > 0 && page > this.myLastPage)
      {
         // Show last page
         this.myCurrentPage = this.myLastPage;
      }
      else
      {    
         // Show requested page
      	this.myCurrentPage = page;     	
      }
      
     // this.myPreviewablePage.setPageIndex(this.myCurrentPage);  
      this.myPreviewPanel.showPage(this.myCurrentPage);  
      
      updateButtons();      
   }

   void updateButtons()
   {
   	//System.out.println("pp Current: " + myCurrentPage + "   Last:  " + myLastPage);
      if (this.myCurrentPage > 1)
      {
         this.myPrior.setEnabled(true);
      }
      else
      {
         this.myPrior.setEnabled(false);
      }

      if (this.myLastPage > 1)
      {
         this.myLast.setEnabled(true);
      }
      else
      {
         this.myLast.setEnabled(false);
      }

      if (this.myCurrentPage < this.myLastPage)
      {
         this.myNext.setEnabled(true);
      }
      else
      {
         this.myNext.setEnabled(false);
      }
   }
   
   void setupDialog()
   {
      this.myFirst.addActionListener(new java.awt.event.ActionListener()
      {
         public void actionPerformed(java.awt.event.ActionEvent ae)
         {
             showPreview(1);
          }
      });

      this.myPrior.addActionListener(new java.awt.event.ActionListener()
      {
         public void actionPerformed(java.awt.event.ActionEvent ae)
         {
            showPreview(PageablePreviewer.this.myCurrentPage - 1);
         }
      });

      this.myNext.addActionListener(new java.awt.event.ActionListener()
      {
         public void actionPerformed(java.awt.event.ActionEvent ae)
         {
            showPreview(PageablePreviewer.this.myCurrentPage + 1);
         }
      });

      this.myLast.addActionListener(new java.awt.event.ActionListener()
      {
         public void actionPerformed(java.awt.event.ActionEvent ae)
         {
             showPreview(PageablePreviewer.this.myLastPage);
         }
      });

      JButton closeBtn = new JButton(AppStarter.getString("close")); //$NON-NLS-1$
      closeBtn.addActionListener(new java.awt.event.ActionListener()
      {
         public void actionPerformed(java.awt.event.ActionEvent ae)
         {
            PageablePreviewer.this.close();
         }
      });

      JPanelRows bottomPanel = new JPanelRows(FlowLayout.CENTER);
      JPanel aRow = bottomPanel.topRow(FlowLayout.CENTER);
      if (this.myLastPage > 0)
      {
         aRow.add(this.myFirst);
         aRow.add(this.myPrior);
         aRow.add(this.myNext);
         aRow.add(this.myLast);
         aRow = bottomPanel.nextRow(FlowLayout.CENTER);
      }
		for(int i=0; i<this.myActionBtns.length; i++)
		{
			aRow.add(this.myActionBtns[i]);
		}
      aRow.add(closeBtn);
      
      this.myPageable.setProgressBar(this.myPicBar);
      JPanel progressBars = new JPanel(new BorderLayout());
      progressBars.add(this.myPicBar, BorderLayout.NORTH);
      progressBars.add(this.myPageBar, BorderLayout.SOUTH);
      aRow = bottomPanel.nextRow(new BorderLayout());
      aRow.add(progressBars, BorderLayout.CENTER);

      //JComponent preview = new PreviewPanel(this.myPreviewablePage);
      //JComponent preview = new JScrollPane(new PreviewPanel(this.myPreviewablePage));
      
		JScrollPane scrollPane = new JScrollPane(this.myPreviewPanel,
				ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS,
				ScrollPaneConstants.HORIZONTAL_SCROLLBAR_ALWAYS);
           
      JPanel content = new JPanel(new BorderLayout());
      content.add(scrollPane, BorderLayout.CENTER);
      content.add(bottomPanel, BorderLayout.SOUTH);
      this.myFrame.setContentPane(content);
      
      //SwingUtils.sizeScreen(this.myFrame, .60f);
      this.myFrame.pack();
      SwingUtils.centerOnScreen(this.myFrame);
   }
   
   private JButton[] getDefaultButtons(JButton pdfBtn)
   {	 	
		// Print Low Resolution
	 	JButton lowResButton = new JButton(AppStarter.getString("ContactSheetPrintOptions.4")); //$NON-NLS-1$  Print Low Resolution
	 	lowResButton.addActionListener(new java.awt.event.ActionListener()
		{
			public void actionPerformed(java.awt.event.ActionEvent ae)
			{
				Pageable printMe = PageablePreviewer.this.myPageable.getPageableForPrinting();
				PageableUtils.printToPrinter(printMe, PageablePreviewer.this.myFrame, PageablePreviewer.this.myPageBar);
			}
		});
	
		// Save As Jpg
	 	JButton jpegButton = new JButton("Save As JPG"); 
	 	jpegButton.addActionListener(new java.awt.event.ActionListener()
		{
			public void actionPerformed(java.awt.event.ActionEvent ae)
			{
				Pageable printMe = PageablePreviewer.this.myPageable.getPageableForPrinting();
				PageableUtils.saveAsJpg(printMe, PageablePreviewer.this.myFrame, PageablePreviewer.this.myPageBar);
			}
		});	 	
	 	jpegButton.setEnabled(this.myPageable.getSaveInfo() != null);
	 	
	 	if (pdfBtn == null)
	 	{
	 		return new JButton[]{lowResButton, jpegButton};
	 	}
	 	return new JButton[]{lowResButton,  pdfBtn, jpegButton};
	}
   
	// ==========================================================================
	//                       Main
	// ==========================================================================
	public static void main(String[] args)
	{
		org.gerhardb.lib.util.startup.AppStarter.startUpApp(args, "org.gerhardb.jibs.Jibs", true);
		PageableTest pageableTest = new PageableTest("D:/dev/gnuJpdf/a.jpg");
		new PageablePreviewer(
				pageableTest, 
	   		true,
	   		null);
	} 
}

