/*
 * Copyright (c) 2007 Gerhard Beck. All rights reserved.
 * SAIC (C) Copyright 2002 - Science Applications International Corporation
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.gerhardb.lib.print; 

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Container;
import java.awt.Cursor;
import java.awt.Dialog;
import java.awt.FlowLayout;
import java.awt.Frame;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.StringSelection;
import java.awt.print.PageFormat;
import java.awt.print.Printable;
import java.awt.print.PrinterException;
import java.awt.print.PrinterJob;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;

import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JEditorPane;
import javax.swing.JFileChooser;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.RepaintManager;
import javax.swing.text.JTextComponent;
import javax.swing.text.View;
import javax.swing.text.html.HTMLEditorKit;

import org.gerhardb.lib.util.startup.AppStarter;

import com.saic.isd.swing.filechooser.SimpleFileFilter;
import com.saic.isd.util.report.Report;

/**
 * Prints text in a JTextComponent.
 * <p>
 * Idea for and initial code for this class thanks to:<br>
 * http://www.jalice.net/exkit.htm
*/
public class PrintableText implements Printable
{
  JTextComponent myText;

  public PrintableText( JTextComponent text )
  {
    this.myText = text;
  }

  /*
  =============================================================================
                        Printable Interface
  =============================================================================
  */
  public int print( Graphics g, PageFormat pform, int pageNumber )
  {
    double pageOffset = pageNumber * pform.getImageableHeight();

    View view = this.myText.getUI().getRootView( this.myText );

    if ( pageOffset > view.getPreferredSpan( View.Y_AXIS ) )
    {
      return Printable.NO_SUCH_PAGE;
    }

    ((Graphics2D)g).translate( 0d, -pageOffset );

    Rectangle rect = new Rectangle();

    rect.setRect(
      pform.getImageableX(),
      pform.getImageableY(),
      pform.getImageableWidth(),
      pform.getImageableHeight() + pageOffset );

    view.paint( g, rect );

    return Printable.PAGE_EXISTS;
  } 
  
  /*
  =============================================================================
                          Public Classes
  =============================================================================
  */
  public void printContent()
  {
	  System.out.println("00000000000000000000000000000000000"); //$NON-NLS-1$
    PrinterJob pjob  = PrinterJob.getPrinterJob();

    // This is commented out in the lab as of 11/18/2002
    //PageFormat pform = pjob.pageDialog( pjob.defaultPage() );

    /* to speed up performance at least a bit */
    RepaintManager.currentManager( this.myText ).setDoubleBufferingEnabled( false );
    pjob.setPrintable( this );

    Cursor oldCursor = this.myText.getCursor();
    this.myText.setCursor( Cursor.getPredefinedCursor( Cursor.WAIT_CURSOR ) );

    try
    {
      if ( pjob.printDialog() )
      {
        pjob.print();
      }
    }
    catch (PrinterException pe)
    {
      System.err.println("PrinterException has occurred:" + pe.getMessage()); //$NON-NLS-1$
    }
    finally
    {
      this.myText.setCursor( oldCursor );
    }
  } 

  public void copy()
  {
    String contents = this.myText.getText();
    StringSelection clip = new StringSelection( contents );

    Clipboard clipboard = this.myText.getToolkit().getSystemClipboard();
    clipboard.setContents( clip, clip );
  } 

  public void file()
  {
    String contents = this.myText.getText();

    JFileChooser chooser = new JFileChooser( getUserHomeDirectory() );
    SimpleFileFilter filter = new SimpleFileFilter( "html", AppStarter.getString("report") ); //$NON-NLS-1$ //$NON-NLS-2$
    chooser.addChoosableFileFilter( filter );
    chooser.setFileFilter( filter );

    int returnVal  = chooser.showSaveDialog( this.myText );
    if ( returnVal == JFileChooser.APPROVE_OPTION )
    {
      File file = filter.attachExtension( chooser.getSelectedFile() );
      try
      {
        FileWriter stream = new FileWriter( file );
        stream.write( contents );
        stream.close();
      }
      catch( IOException ex )
      {
        JOptionPane.showMessageDialog(
          this.myText,
          ex.getMessage(),
          AppStarter.getString("save.failed"), //$NON-NLS-1$
          JOptionPane.ERROR_MESSAGE);
      }
    }
  }

  /*
  public void preview()
  {
    String contents = makeContents();
    Util.previewText( myComponentToPrint, null, contents );
  } // print()
  */

 public static File getUserHomeDirectory()
 {
   StringBuffer buf = new StringBuffer( 80 );
   buf.append( System.getProperty("user.home") ); //$NON-NLS-1$
   buf.append( File.separator );
   buf.append( "data" ); //$NON-NLS-1$
   File rtnMe = new File( buf.toString() );
   if ( ! rtnMe.exists() )
   {
      rtnMe.mkdirs();
   }
   else if ( rtnMe.isFile() )
   {
      rtnMe = rtnMe.getParentFile();
   }
   return rtnMe;
 }

 public void previewHTML( Component owner )
 {
   String title = AppStarter.getString("print.preview"); //$NON-NLS-1$

   if ( owner instanceof Frame )
   {
     preview(
       new JDialog( (Frame)owner, title, false ),
       this.myText.getText(),
       new HTMLEditorKit() );
   }
   else if ( owner instanceof Dialog )
   {
     preview(
       new JDialog( (Dialog)owner, title, false ),
       this.myText.getText(),
       new HTMLEditorKit() );
   }
   else
   {
     preview(
       new JDialog( (Dialog)null, title, false ),
       this.myText.getText(),
       new HTMLEditorKit() );
   }
 }

 public static void preview(
   JDialog dialog,
   String printMe,
   javax.swing.text.EditorKit kit )
 {
   final JDialog previewDlg = dialog;

   JEditorPane pane = new JEditorPane();
   pane.setEditorKit( kit );
   pane.setText( printMe );
   pane.validate();
   pane.setEditable( false );
   //pane.addHyperlinkListener( new Hyperactive() );

   final String prtThis = printMe;
   final javax.swing.text.EditorKit theKit = kit;

   JButton okBtn = new JButton( AppStarter.getString("print") ); //$NON-NLS-1$
   okBtn.addActionListener
   (
     new java.awt.event.ActionListener()
     {
       public void actionPerformed( java.awt.event.ActionEvent ae )
       {
         print( prtThis, theKit );
       }
     }
   );

   JButton cancelBtn = new JButton( AppStarter.getString("cancel") ); //$NON-NLS-1$
   cancelBtn.addActionListener
   (
     new java.awt.event.ActionListener()
     {
       public void actionPerformed( java.awt.event.ActionEvent ae )
       {
         previewDlg.dispose();
       } // actionPerformed(ActionEvent ae)
     }
   );

   JPanel btnPanel = new JPanel( new FlowLayout( FlowLayout.CENTER ) );
   btnPanel.add( okBtn );
   btnPanel.add( cancelBtn );

   Container contentPane = previewDlg.getContentPane();
   contentPane.setLayout( new BorderLayout() );
   contentPane.add( new JScrollPane( pane ), BorderLayout.CENTER );
   contentPane.add( btnPanel, BorderLayout.SOUTH );

   previewDlg.setSize( 800, 600 );
   previewDlg.setVisible(true);
 } // previewHTML

 public static void print(
   String printMe,
   javax.swing.text.EditorKit kit )
 {
   JEditorPane pane = new JEditorPane();
   pane.setEditorKit( kit );
   pane.setText( printMe );
   pane.validate();

   PrintableText pt = new PrintableText( pane );
   pt.printContent();
 } 

	// ==========================================================================
	// Test Harness
	// ==========================================================================
	public static void main(String[] args)
	{
		Report.showReport("Test PrintableText", "Sample Report", "Hello World!"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
	}
}

