/*
 * Copyright (c) 2011 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.gerhardb.lib.print.contactSheet;

import java.awt.*;
import java.io.File;
import javax.swing.JFrame;
import javax.swing.JProgressBar;
import javax.swing.SwingUtilities;
import org.gerhardb.lib.print.*;
import org.gerhardb.lib.swing.GlassPane;


/**
 * Easier to user replacement for java.Pageable.
 * Not really, but it servers the same purpose in this framework.
 */
public class ContactSheetPageable implements Pageable, Cloneable
{
	// Passed In
	private PageLayout pageLayout;

	private ContactSheetInfo csInfo;
	private HeaderFooterInfo hfInfo;
	private SaveInfo saveInfo;
	private boolean showFilesNotDirectories;
	
	// Computed
	private String[] fonts = new String[3];
	private HeaderFooter headerFooter;
	InteriorLayout interior;
	PicBoxSizes picBoxSizes;

	JFrame topFrame;
	
	private JProgressBar picBar;	
	ListToDraw drawIt;
	GlassPane eatMe = new GlassPane();
	
	int pageNumber;
	
	DrawAPic drawAPic;
	ResizedImageFactory resizedImageFactory;
	int picsDrawn;
	ResizedImageCache imageCache;
	boolean useBackgroundThead;
	
	public ContactSheetPageable(File[] f, boolean showFilesNotDirectoriesIn, 
			PageLayout pl, HeaderFooterInfo hfi, ContactSheetInfo csi, SaveInfo si,
			boolean useBackgroundTheadIn)
	throws Exception
	{
		//System.out.println("new ContactSheetPageable");
		this.showFilesNotDirectories = showFilesNotDirectoriesIn;
		this.pageLayout = pl;
		this.hfInfo = hfi;
		this.csInfo = csi;	
		this.saveInfo = si;
		this.useBackgroundThead = useBackgroundTheadIn;
		
		this.fonts[0] = this.hfInfo.headerFontFamily;
		this.fonts[1] = this.hfInfo.footerFontFamily;
		this.fonts[2] = this.csInfo.pictureFontFamily;
				
		// This needs picsPerPage
		// This will correctly set number of pages needed for HeaderFooter.
		this.drawIt = ListToDraw.makeListToDraw(f, 
				this.csInfo.rows, this.csInfo.cols, this.showFilesNotDirectories);
		
		// Now that we know the page count, we can set HeaderFooter
		this.interior = HeaderFooter.calcInterior(this.hfInfo, this.pageLayout.getInteriorLayout());			
		this.picBoxSizes = new PicBoxSizes(this.interior, this.csInfo, this.drawIt, this.pageLayout, false);
		
		// Moving this to a more general location and using setGraphics
		// didn't work out.  Like picBoxWidth, ect wasn't set yet.
		this.drawAPic = new DrawAPic(
				this.csInfo.pictureFontSize, 
				this.picBoxSizes.picBoxWidth, 
				this.picBoxSizes.picBoxHeight, 
				this.csInfo);

		this.resizedImageFactory = new ResizedImageFactory(
				this.csInfo, 
				this.drawAPic.netPicWidth, 
				this.drawAPic.netPicHeight);
		
		this.imageCache = new ResizedImageCache(this.resizedImageFactory);
		
		this.headerFooter = new HeaderFooter(getPageLayout().getInteriorLayout(), getNumberOfPages());		
	}
	   
	// ==========================================================================
	// Thread Implementation
	// ==========================================================================
	
	private void doInBackground()
	{
		//System.out.println("ContactSheetPageable.doInBackground() =========================================================");
		Thread backgroundThread = new Thread(new Runnable()
		{
			public void run()
			{
				cacheImages(ContactSheetPageable.this.drawIt.getFilesForPage(ContactSheetPageable.this.pageNumber));			
				SwingUtilities.invokeLater(new Runnable()
				{
					public void run()
					{
						ContactSheetPageable.this.topFrame.repaint();
					}
				});
			}
		});
		backgroundThread.start();		
	}
	
 	// ==========================================================================
	// Pageable Implementation
	// ==========================================================================
	
	public void setTopFrame(JFrame topFrameIn)
	{
		this.topFrame = topFrameIn;
	}

	/**
	 * Whole document will have only one page layout.
	 * @return
	 */
	public PageLayout getPageLayout()
	{
		return this.pageLayout;
	}
	
	public int getNumberOfPages()
	{
		return this.drawIt.pageCount;
	}
	
	// Because gnuJpdf only works with internal fonts.
	// Used to turn off gnuJpdf button.
	public String[] getFonts()
	{
		return this.fonts;
	}
	
	public SaveInfo getSaveInfo()
	{
		return this.saveInfo;
	}
	
	public String convert(String text, int pageIndex)
	{
		if (text == null){return null;}
		text = text.replaceAll(HeaderFooterInfo.PAGE_INDEX, Integer.toString(pageIndex));
		text = text.replaceAll(HeaderFooterInfo.PAGE_COUNT, Integer.toString(ContactSheetPageable.this.drawIt.pageCount));		
		return this.drawIt.convert(text, pageIndex);
	} 

	public boolean backgroundTreadEnabled()
	{
		return this.useBackgroundThead;
	}
	
	public void prepareToPrint(int pageIndex)
	{
		this.pageNumber = pageIndex;			
		if (this.useBackgroundThead)
		{
			this.doInBackground();
		}
		else
		{
			cacheImages(this.drawIt.getFilesForPage(this.pageNumber));
		}
	}
	
	public void print(Graphics g)
	{
		//new Exception("CSP Page Index: " + pageIndex).printStackTrace();
		g.setColor(this.csInfo.backgroundColor);
		g.fillRect(
				0, 
				0, 
				this.pageLayout.getPageWidth(),
				this.pageLayout.getPageHeight());
		
		g.setColor(this.csInfo.foregroundColor);
		
		// Now we can start drawing
		this.headerFooter.draw(g, this.pageNumber, this.hfInfo, this);	
		
		if (drawTestPattern(false, g))
		{
			return;
		}

		Font picFont = new Font(
				this.csInfo.pictureFontFamily, 
				Font.PLAIN,
				this.csInfo.pictureFontSize);
		g.setFont(picFont);
		FontMetrics fm = g.getFontMetrics();
		
		ListToDraw.FilesForPage ffp = this.drawIt.getFilesForPage(this.pageNumber);
		if (ffp.filesToPrint())
		{
			drawImages(g, ffp);
		}
		else
		{				
			// Handle edge case of printing directories and the directory is empty.
			// So if nothing drawn on the first page, we will assume empty directory.
			// Only really applies if the DrawIt in question happens to have been
			// a directory.
			if (this.pageNumber == 1)
			{
				String printMe = "Directory had no pictures";	
				int textY = this.pageLayout.getPageHeight()/2;		
				int textX = (this.pageLayout.getCenterX() - (fm.stringWidth(printMe)/ 2));
				g.drawString(printMe, textX, textY);
			}
		}
	}
		
	public void setProgressBar(JProgressBar bar)
	{
		// If we are running background thread, let the pictures spaak for 
		// themselves.  Also, if you scroll ahead, different pages will be 
		// competing.
		if (this.useBackgroundThead)
		{
			this.picBar = new JProgressBar();
		}
		else
		{
			this.picBar = bar;
		}
		this.picBar.setMaximum(this.drawIt.picsPerPage);
	}
	
	public Pageable getPageableForPrinting()
	{
		ContactSheetPageable rtnMe = this;
		try
		{
			rtnMe = (ContactSheetPageable)this.clone();
			rtnMe.useBackgroundThead = false;
		}
		catch(Exception ex)
		{
			ex.printStackTrace();
		}
		return rtnMe;
	}

 	// ==========================================================================
	// Other
	// ==========================================================================
	
	@Override
	public Object clone()
   throws CloneNotSupportedException
   {
		return super.clone();
	}
		

	public ContactSheetInfo getContactSheetInfo()
	{
		return this.csInfo;
	}

	public HeaderFooterInfo getHeaderFooterInfo()
	{
		return this.hfInfo;
	}

	public boolean showFilesNotDirectories()
	{
		return this.showFilesNotDirectories;
	}
	
	protected void cacheImages(ListToDraw.FilesForPage ffp) 
	{
		int picsPrinted = 0;
		this.picBar.setValue(picsPrinted);
		while (picsPrinted < this.drawIt.picsPerPage)
		{
			int drawIndex = ffp.startPicIndex + picsPrinted;
			//System.out.println("drawImages: " + drawIndex);
			if (drawIndex < ffp.files.length)
			{			
				File imageFile = ffp.files[drawIndex];
				
				//ResizedImage resizedImage = this.resizedImageFactory.makeResizedImage(imageFile, drawIndex);
				ResizedImage resizedImage = this.imageCache.putImage(imageFile, drawIndex); 
				
				if (resizedImage != null)
				{
					resizedImage.xOffset = this.picBoxSizes.xPicBoxOffsets[picsPrinted]; 
					resizedImage.yOffset = this.picBoxSizes.yPicBoxOffsets[picsPrinted];
				}
				picsPrinted++;
				this.picBar.setValue(picsPrinted);
				
				if (this.useBackgroundThead)
				{
					SwingUtilities.invokeLater(new Runnable()
					{
						public void run()
						{
							ContactSheetPageable.this.topFrame.repaint();
						}
					});					
				}
			}
			else
			{
				// You get here if the number of pics LEFT to show are less than the
				// the number of pics you CAN show on a page.  This happens on the last page.
				this.picBar.setValue(this.drawIt.picsPerPage);
				break;
			}
		}
	}
	
	protected void drawImages(Graphics g, ListToDraw.FilesForPage ffp) 
	{
		for (int i=0; i<this.drawIt.picsPerPage; i++)
		//while (picsPrinted < this.drawIt.picsPerPage)
		{
			int drawIndex = ffp.startPicIndex + i;
			//System.out.println("drawImages: " + drawIndex);
			if (drawIndex < ffp.files.length)
			{			
				File imageFile = ffp.files[drawIndex];
				if (imageFile != null)
				{
					ResizedImage cachedImage = this.imageCache.getImage(imageFile); 
					if (cachedImage != null)
					{
						this.drawAPic.print(g, cachedImage);
					}
				}
			}
		}
	}
	
				// THIS IS A NOTE ON WHY THE GET IMAGE IS WRITTEN AS IT IS.
				// Don't show errors getting the picture.
				//buff = ImageUtil.getImage(myImageFiles[currentIndex]);
				// This code will show error on getting the picture.
				 /*
				try
				{
					IOImage io = new IOImage(myImageFiles[currentIndex]);
					buff = io.getImage();
				} 
				catch (Exception ex)
				{
					throw new PrinterException(ex.getMessage());
				}
				*/
	
	private boolean drawTestPattern(boolean rtnMe, Graphics g)
	{
		if (rtnMe)
		{			
			g.setColor(Color.RED);
			
			// Paint cross hairs for whole page
			g.drawLine(this.interior.getCenterX(), this.interior.getTopMargin(), this.interior.getCenterX(), this.interior.getBottomMargin());
			g.drawLine(this.interior.getLeftMargin(), this.interior.getCenterY(), this.interior.getRightMargin(), this.interior.getCenterY());
	
			// Paint margin for whole page
			g.drawRect(this.interior.getLeftMargin(), this.interior.getTopMargin(), 
					this.interior.getWidth(),
					this.interior.getHeight());
			
			for (int i = 0; i < this.drawIt.picsPerPage; i++)
			{
				g.setColor(Color.BLUE);
				int x = this.picBoxSizes.xPicBoxOffsets[i];
				int y =  this.picBoxSizes.yPicBoxOffsets[i];
				int w =  this.picBoxSizes.picBoxWidth;
				int h =  this.picBoxSizes.picBoxHeight;
				g.drawRect(x, y, w, h);
				
				// Paint X
				g.setColor(Color.BLUE);
				g.drawLine(x, y, x + w, y + h);
				g.drawLine(x + w, y, x, y + h);
			}
			
			g.setColor(Color.BLACK);
		}
		return rtnMe;
	}
	
	// ==========================================================================
	//                       Main
	// ==========================================================================
	/*
	public static void main(String[] args)
	{
		org.gerhardb.lib.util.startup.AppStarter.startUpApp(args, "org.gerhardb.jibs.Jibs", true);
	
		boolean shortTest = false;
		if (shortTest)
		{
			File[] theList;
			boolean filesNotDir = true;
			if (filesNotDir)
			{
				//File dir = new File("D:/testpics/Fixed Files/contactSheet/a");
				//theList = dir.listFiles();
				
				theList = new File[]{new File("D:/dev/gnuJpdf/a.jpg")};
				theList = new File[]{new File("D:/testpics/Fixed Files/contactSheet/c/aaa-a.jpg")};
				theList = new File("D:/testpics/Fixed Files/proof/drewnoakes").listFiles();
			}
			else
			{
				File dirA = new File("D:/testpics/Fixed Files/contactSheet/a");
				File dirB = new File("D:/testpics/Fixed Files/contactSheet/b");
				File dirC = new File("D:/testpics/Fixed Files/contactSheet/c");
				theList = new File[]{dirA, dirB, dirC};
			}
			
			PageLayout pl = PageLayout.makeDefault();
			
			HeaderFooterInfo hfi = new HeaderFooterInfo();
			hfi.headerLeft = "Left";
			hfi.headerCenter = "<%DIR%>";
			hfi.headerRight = "Right";	
			
			ContactSheetInfo csi = new ContactSheetInfo();
			csi.rows = 2;
			csi.cols = 2;
			csi.showPictureName = true;
			csi.showPictureNumber = true;
			csi.showPictureSize = true;
			csi.frameSize = 1;
			csi.shadowSize = 4;
			csi.pictureFontSize = 10;
			csi.pictureFontFamily = "SansSerif";
			csi.pictureTextColor = Color.BLACK;
			csi.backgroundColor = Color.WHITE;
			csi.foregroundColor = Color.BLACK;
			
			SaveInfo si = null;
			try
			{
				si = new SaveInfo(new File("D:/trash"));
			}
			catch(Exception ex)
			{
				ex.printStackTrace();
			}
			
			try
			{
				ContactSheetPageable test = new ContactSheetPageable(theList, filesNotDir, pl, hfi, csi, si true);
				new PageablePreviewer(test, true, null);
			}
			catch(Exception ex)
			{
				ex.printStackTrace();
			}
		}
	} 		
	*/
}


/*
// April 9 2011
// Easiest to just forget Landscape because it's too hard to do the rotation
// with just a Graphics and there is no easy way to go from Graphics2D back
// to a Graphics without completely screwing up image quality.
public void print(Graphics g, int pageIndex)
{
	if (this.pageLayout.getPortrait())
	{
		printPage(g, pageIndex);
	}
	else
	{
		GraphicsConfiguration gc = this.progress.getGraphicsConfiguration();
		
		// Swap width and heith for landscape.
		BufferedImage buff = gc.createCompatibleImage(this.pageLayout.getPageHeight(), this.pageLayout.getPageWidth());
		Graphics2D g2 = buff.createGraphics();
		
		// Reset origin to the right most so rotation will show.
		g2.translate(buff.getWidth(), 0);
		
		// Rotate 90 degrees
		AffineTransform rotate =   
	  	   AffineTransform.getRotateInstance(Math.PI/2.0,0.0,0.0);
		g2.transform(rotate);
		
		printPage(g2, pageIndex);
		
		// This results in extremely poor image quality.
		g.drawImage(buff, 0, 0, null);			
		g2.dispose();
	}
}
*/

/*
GraphicsConfiguration gc = this.progress.getGraphicsConfiguration();
// Swap width and heith for landscape.
BufferedImage buff = gc.createCompatibleImage(this.pageLayout.getPageHeight(), this.pageLayout.getPageWidth());
Graphics2D g2 = buff.createGraphics();

//g2.setColor(Color.WHITE);
//g2.drawLine(0, 0, 200, 200);

//Rectangle2D rect = new Rectangle2D.Float(1.0f,1.0f,200.0f,300.0f);
// Math.PI/4.0 = 45 degrees
// Math.PI/4.0 = 90 degrees
AffineTransform rotate =   
	   AffineTransform.getRotateInstance(Math.PI/2.0,0.0,0.0);
//g2.translate(200, 200);

	g2.translate(buff.getWidth(), 0);

g2.transform(rotate);
//g2.draw(rect);

printReal(g2, pageIndex);

g.drawImage(buff, 0, 0, null);			
g2.dispose();
*/

