/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package org.gerhardb.lib.print.contactSheet;

import java.awt.*;
import java.awt.image.BufferedImage;

import org.gerhardb.lib.print.contactSheet.ContactSheetInfo.ResizeDPI;

import com.itextpdf.text.BaseColor;
import com.itextpdf.text.Image;

import com.itextpdf.text.pdf.PdfContentByte;

/**
 * ContactSheet
 */
public class DrawAPic
{
	private ContactSheetInfo csInfo;
	private int myPicBoxWidth;
	private int myPicBoxHeight;
	private static int IMAGE_PADDING = 3;

	// Calculated at startup.
	int textHeight;
	int frameOneSide;
	int frameTwoSides;
	int shadow;
	int netPicHeight;
	int netPicWidth;

	/**
	 *
	 */
	public DrawAPic(int fontSize, int width, int height, ContactSheetInfo csi)
	{
		this.myPicBoxWidth = width; 
		this.myPicBoxHeight = height;
		this.csInfo = csi;	
		
		if (this.csInfo.showPictureName || this.csInfo.showPictureSize)
		{
			this.textHeight = IMAGE_PADDING;
		}
			
		if (this.csInfo.showPictureName)
		{
			this.textHeight = this.textHeight + fontSize;
		}

		if (this.csInfo.showPictureSize)
		{
			this.textHeight = this.textHeight + fontSize;
		}

		if (this.csInfo.frameSize > 0)
		{
			this.frameOneSide = this.csInfo.frameSize;
			this.frameTwoSides = this.frameOneSide * 2;
		}

		if (this.csInfo.shadowSize > 0)
		{
			this.shadow = this.csInfo.shadowSize;
		}

		// Now that we have the text, we can size the image to fit in the box
		// with the text.
		this.netPicHeight = this.myPicBoxHeight - this.textHeight - this.frameTwoSides - this.shadow;
		this.netPicWidth = this.myPicBoxWidth - this.frameTwoSides - this.shadow;
		//System.out.println("DrawAPic w: " + this.myPicBoxWidth + "   h: " + this.myPicBoxHeight + "    sized to w: " + this.netPicWidth + "   h: " + this.netPicHeight);
	}
	
	/**
	 * 
	 * @param picCntList 0 for first picture in this directory.
	 * @param imageFile The file to print.
	 */
	public void print(Graphics g, ResizedImage resizedImage) //, int xOffset, int yOffset)
	{		
		
		int xCenterImg = (int) (((float) this.myPicBoxWidth - resizedImage.getResizedWidth()) / 2);
		int imgX = resizedImage.xOffset + xCenterImg;
		int imgY = resizedImage.yOffset + this.frameOneSide;

		int frameX = imgX - this.frameOneSide;
		int frameY = resizedImage.yOffset;
		// The left and right edges of the rectangle are at x and x + width. 
		// The top and bottom edges are at y and y + height.
		int frameWidth = resizedImage.getResizedWidth() + this.frameTwoSides;
		int frameHeight = resizedImage.getResizedHeight() + this.frameTwoSides;

		if (this.csInfo.shadowSize > 0)
		{
			int x = frameX + this.shadow;
			int y = frameY + this.shadow;
			int width = frameWidth;
			int height = frameHeight;

			/*
			GradientPaint gradient = 
				new GradientPaint(frameX + frameWidth, y, Color.YELLOW,
						frameX + frameWidth + shadow, 
						y + height, Color.GREEN);
			g.setPaint(gradient);
			gradient = 
				new GradientPaint(x, frameY + frameHeight, Color.YELLOW,
						x + width, 
						frameY + frameHeight + shadow, Color.GREEN);
			g.setPaint(gradient);
			 */
			Color fill = new Color(
					this.csInfo.pictureTextColor.getRed(), 
					this.csInfo.pictureTextColor.getGreen(), 
					this.csInfo.pictureTextColor.getBlue(), 
					125);
			g.setColor(fill);
			g.fillRect(x, y, width, height);
			g.setColor(this.csInfo.pictureTextColor);
		}

		if (this.csInfo.frameSize > 0)
		{
			g.fillRect(frameX, frameY, frameWidth, frameHeight);
		}

		try
		{
			if (resizedImage.getImage() != null)
			{
				boolean drawn = g.drawImage(
						resizedImage.getImage(), 
						imgX, imgY, 
						null);
				if (!drawn)
				{
					System.out.println("DrawAPic moved on before image was printed: " + resizedImage.picName);
				}
			}
		}
		catch (Exception e)
		{
			e.printStackTrace();
		}

		int textY = (resizedImage.yOffset + this.frameTwoSides + resizedImage.getResizedHeight() + IMAGE_PADDING);
		
		FontMetrics fm = g.getFontMetrics();
		if (resizedImage.picName != null)
		{
			// Text is draw up from the Ascent and down from the Descent.
			int xCenterPic = ((this.myPicBoxWidth - fm.stringWidth(resizedImage.picName)) / 2);
			int textX = resizedImage.xOffset + xCenterPic;
			// This puts the text at the bottom of the box.
			//int textY = (int) (yOffset + myPicBoxHeight - tLay.getDescent());
			textY = (textY + fm.getAscent());
			g.drawString(resizedImage.picName, textX, textY);
			// Set up for next calc
			textY = (textY + fm.getDescent());
		}

		if (resizedImage.picSize != null)
		{
			// Text is draw up from the Ascent and down from the Descent.
			int xCenterPic = ((this.myPicBoxWidth - fm.stringWidth(resizedImage.picSize)) / 2);
			int textX = resizedImage.xOffset + xCenterPic;
			// This puts the text at the bottom of the box.
			//int textY = (int) (yOffset + myPicBoxHeight - tLay.getDescent());
			textY = (textY + fm.getAscent());
			g.drawString(resizedImage.picSize, textX, textY);
		}
	}
	
	public void print(ContactSheetPDF pdf, ResizedImage resizedImage)
	{		
		int xCenterImg = (int) (((float) this.myPicBoxWidth - resizedImage.getResizedWidth()) / 2);
		int imgX = resizedImage.xOffset + xCenterImg;
		int imgY = resizedImage.yOffset - this.frameOneSide;
		//int imgY = resizedImage.yOffset + this.frameOneSide;

		int frameX = imgX - this.frameOneSide;
		int frameY = resizedImage.yOffset;
		// The left and right edges of the rectangle are at x and x + width. 
		// The top and bottom edges are at y and y + height.
		int frameWidth = resizedImage.getResizedWidth() + this.frameTwoSides;
		int frameHeight = resizedImage.getResizedHeight() + this.frameTwoSides;

		if (ContactSheetPDF.SHOW_TEST)
		{
			// Print red line around where picture will go.
			pdf.cb.setColorStroke(BaseColor.RED);
			pdf.cb.rectangle(frameX, frameY, frameWidth, -frameHeight); 
			pdf.cb.stroke();
		}
		
		if (this.csInfo.shadowSize > 0)
		{
			BaseColor fill = ContactSheetPDF.getBaseColor(this.csInfo.pictureTextColor);
			pdf.cb.setColorFill(fill);
			int x = frameX + this.shadow;
			int y = frameY - this.shadow;
			pdf.cb.rectangle(x, y, frameWidth, -frameHeight);  
			pdf.cb.fill();
			pdf.cb.stroke();
		}

		if (this.csInfo.frameSize > 0)
		{
			pdf.cb.setColorFill(ContactSheetPDF.getBaseColor(this.csInfo.pictureTextColor));
			pdf.cb.rectangle(frameX, frameY, frameWidth, -frameHeight);  
			pdf.cb.fill();
			pdf.cb.stroke();
		}
		
		try
		{
			BufferedImage buff = resizedImage.getImage();
			//System.out.println("Using PDF buff: " + buff.getWidth() + " " + buff.getHeight());
			Image image = Image.getInstance(buff, null);
			if (pdf.csInfo.resizeDPI != ResizeDPI.DPI_72)
			{
				//System.out.println("Using PDF RESIZED IMAGE: " + this.netPicWidth + " " + this.netPicHeight);
				int imageDrawnFromTop = imgY - frameHeight;
				image.setAbsolutePosition(imgX, imageDrawnFromTop);
				image.scaleToFit(frameWidth, frameHeight);
			}
			else
			{
				int imageDrawnFromTop = imgY - buff.getHeight();
				image.setAbsolutePosition(imgX, imageDrawnFromTop);				
			}
			pdf.cb.addImage(image);
		}
		catch (Exception e)
		{
			System.out.println("DrawAPic moved on before image was printed: " + resizedImage.picName);
			e.printStackTrace();
		}
		
		//g.setColor(this.csInfo.pictureTextColor);
		
		int xCenterPic = this.myPicBoxWidth/2;
		int textX = resizedImage.xOffset + xCenterPic;			
		//int textY = (resizedImage.yOffset - this.frameTwoSides - IMAGE_PADDING);
		int textY = (resizedImage.yOffset - this.frameTwoSides - resizedImage.getResizedHeight() - IMAGE_PADDING - pdf.csInfo.pictureFontSize - pdf.csInfo.pictureFontSize);
		
		pdf.cb.beginText();
		pdf.cb.setFontAndSize(pdf.bf_helv, pdf.csInfo.pictureFontSize);
		pdf.cb.setColorFill(ContactSheetPDF.getBaseColor(this.csInfo.pictureTextColor));
		
		if (resizedImage.picName != null)
		{
			pdf.cb.showTextAligned(
				PdfContentByte.ALIGN_CENTER, 
				resizedImage.picName, 
				textX,
				textY, 0);
		}

		textY = textY + pdf.csInfo.pictureFontSize;
		if (resizedImage.picSize != null)
		{
			pdf.cb.showTextAligned(
					PdfContentByte.ALIGN_CENTER, 
					resizedImage.picSize, 
					textX,
					textY, 0);
		}

		pdf.cb.endText();
	}
}















