/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package org.gerhardb.lib.print.contactSheet;

import ij.ImagePlus;
import ij.io.Opener;

import java.awt.*;
import java.awt.image.BufferedImage;
import java.io.File;

import org.filthyrichclients.FilthyRichClientsUtils;
import org.gerhardb.lib.image.ImageFactory;
import org.gerhardb.lib.print.contactSheet.ContactSheetInfo.ResizeDPI;

/**
 * ContactSheet
 */
public class ResizedImageFactory
{
	int targetWidth;
	int targetHeight;
	ContactSheetInfo csInfo;
	Opener opener = new Opener();
	
	public ResizedImageFactory(ContactSheetInfo csi, int width, int height)
	{
		this.targetWidth = width;
		this.targetHeight = height;
		this.csInfo = csi;		
		//System.out.println("ResizedImageFactory w: " + this.targetWidth + "   h: " + this.targetHeight);
	}
	
	public ResizedImage makeResizedImage(File imageFile, int picCntList, boolean pdf)
	{
		if (imageFile == null) { return null; }
		//System.out.println("ResizedImageFactory.makeResizedImage: " + imageFile);		
		if (pdf && this.csInfo.resizeDPI == ResizeDPI.DPI_ALL)
		{
			return new FullImageResizer(imageFile, picCntList, pdf);			
		}
		switch(this.csInfo.resizeType)
		{
		case EXSHARP:
		case SHARP:
			return new ImageJImageResizer(imageFile, picCntList, pdf);
		case FUZY:
			return new BufferedImageResizer(imageFile, picCntList, pdf);
		}
		return null; 			
	}
	
	class FullImageResizer extends ResizedImage
	{
		BufferedImage image;
		
		FullImageResizer(File imageFile, int picCntList, boolean pdf)
		{
			super(imageFile, ResizedImageFactory.this.csInfo, picCntList);
			this.image = ImageFactory.getImageFactory().getImageEZ(imageFile);
			
			// Set up calculations in super class for resizing.
			setOriginalWidth(this.image.getWidth(), this.image.getHeight(), ResizedImageFactory.this.csInfo);
			
			int dpi = ResizeDPI.DPI_72.dpi();
			if (pdf)
			{
				dpi = ResizedImageFactory.this.csInfo.resizeDPI.dpi();
			}
			calcToSize(
					ResizedImageFactory.this.targetWidth, 
					ResizedImageFactory.this.targetHeight,
					dpi);
		}

		@Override
		public BufferedImage getImage()
		{
			return this.image;
		}
	}
	
	class BufferedImageResizer extends ResizedImage
	{
		BufferedImage image;
		
		BufferedImageResizer(File imageFile, int picCntList, boolean pdf)
		{
			super(imageFile, ResizedImageFactory.this.csInfo, picCntList);
			
			//System.out.println("BufferedImageResizer");
			BufferedImage resizeMe = ImageFactory.getImageFactory().getImageEZ(imageFile);
			
			// Set up calculations in super class for resizing.
			setOriginalWidth(resizeMe.getWidth(), resizeMe.getHeight(), ResizedImageFactory.this.csInfo);
			
			int dpi = ResizeDPI.DPI_72.dpi();
			if (pdf)
			{
				dpi = ResizedImageFactory.this.csInfo.resizeDPI.dpi();
			}
			calcToSize(
					ResizedImageFactory.this.targetWidth, 
					ResizedImageFactory.this.targetHeight,
					dpi);
      	this.image = FilthyRichClientsUtils.getFasterScaledInstance(
      			resizeMe, 
      			super.dpiWidth, 
      			super.dpiHeight, 
      			RenderingHints.VALUE_INTERPOLATION_BILINEAR, true);
		}

		@Override
		public BufferedImage getImage()
		{
			return this.image;
		}
	}

	class ImageJImageResizer extends ResizedImage
	{
		BufferedImage image;
		
		ImageJImageResizer(File imageFile, int picCntList, boolean pdf)
		{
			super(imageFile, ResizedImageFactory.this.csInfo, picCntList);
			//System.out.println("ImageJImageResizer");
	      try
	      {
	      	String path = imageFile.getAbsolutePath();      	
	      	ImagePlus resizeMe = ResizedImageFactory.this.opener.openImage(path);
	      	
				// Set up calculations in super class for resizing.
				setOriginalWidth(resizeMe.getWidth(), resizeMe.getHeight(), ResizedImageFactory.this.csInfo);
				
				int dpi = ResizeDPI.DPI_72.dpi();
				if (pdf)
				{
					dpi = ResizedImageFactory.this.csInfo.resizeDPI.dpi();
				}
				calcToSize(
						ResizedImageFactory.this.targetWidth, 
						ResizedImageFactory.this.targetHeight,
						dpi);
	   	   ij.process.ImageProcessor ip = resizeMe.getProcessor();  
	   	   ip.setInterpolate(true); // bilinear  
	   	   ij.process.ImageProcessor ip2 = ip.resize(
	      			super.dpiWidth, 
	      			super.dpiHeight);
	   	   resizeMe.setProcessor(ip2);
	   	   this.image = ip2.getBufferedImage(); 
	      }
	      catch (Exception ex)
	      {
	         System.out.println("ImagePlus  (" + imageFile.getName() + "): " + ex.getMessage());
	         ex.printStackTrace();
	      }
		}

		@Override
		public BufferedImage getImage()
		{
			return this.image;
		}
	}
}















